// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.remote;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import de.undercouch.citeproc.csl.CSLItemData;
import de.undercouch.citeproc.helper.oauth.RequestException;
import de.undercouch.citeproc.helper.oauth.UnauthorizedException;

/**
 * Reads items from a remote service. Needs an OAuth API key and
 * secret in order to authenticate.
 * @author Michel Kraemer
 */
public interface RemoteConnector {
	/**
	 * Gets a new authorization URL from the remote server. This is the
	 * first step in the OAuth authorization process. The user has to
	 * visit the authorization URL and grant access to this app. Then they
	 * will get an verification code which has to be passed to {@link #authorize(String)}
	 * @return the URL the authorization URL
	 * @throws IOException if the authorization URL could not be retrieved
	 */
	String getAuthorizationURL() throws IOException;

	/**
	 * Authorizes the app. This is the third step in the OAuth authorization
	 * process. After the user has visited the authorization URL that was
	 * generated by {@link #getAuthorizationURL()}, this method has to be
	 * called to finish authorization
	 * @param verificationCode the verification code the user received from
	 * the authorization URL
	 * @throws IOException if authorization failed
	 */
	void authorize(String verificationCode) throws IOException;

	/**
	 * <p>Authorizes the app. If you already know a valid OAuth access token
	 * (for example if you cached it from previous sessions) you may call
	 * this method instead of {@link #getAuthorizationURL()} and
	 * {@link #authorize(String)}.</p>
	 * <p>This method just saves the provided access token. It does not
	 * act with the remote service in any way.</p>
	 * @see #getAccessTokenValue()
	 * @see #getAccessTokenSecret()
	 * @param token the access token value
	 * @param secret the access token's secret
	 */
	void setAccessToken(String token, String secret);
	
	/**
	 * @return the access token's value used for authentication or null if
	 * the app is not yet authenticated
	 * @see #setAccessToken(String, String)
	 */
	String getAccessTokenValue();
	
	/**
	 * @return the access token's secret used for authentication or null if
	 * the app is not yet authenticated
	 * @see #setAccessToken(String, String)
	 */
	String getAccessTokenSecret();

	/**
	 * Requests a list of item IDs from the server. The user has to be
	 * authenticated before this method can be called.
	 * @return the item IDs
	 * @throws UnauthorizedException if the user is not authenticated
	 * @throws RequestException if the server returns an error code
	 * @throws IOException if the items could not be read from the server
	 */
	List<String> getItemIDs() throws IOException;
	
	/**
	 * Requests an item from the service. The user has to be
	 * authenticated before this method can be called.
	 * @param itemId the item's ID
	 * @return the item
	 * @throws UnauthorizedException if the user is not authenticated
	 * @throws RequestException if the server returns an error code
	 * @throws IOException if the item could not be read from the server
	 */
	CSLItemData getItem(String itemId) throws IOException;
	
	/**
	 * <p>Requests several items from the service. The user has to be
	 * authenticated before this method can be called.</p>
	 * <p>Some services support bulk loading. In this case, this method
	 * makes as few requests as possible. The {@link #getMaxBulkItems()}
	 * method returns the maximum number of items that can be queried
	 * within one request.</p>
	 * @param itemIds the IDs of the items to request
	 * @return a map mapping item IDs to retrieved items
	 * @throws UnauthorizedException if the user is not authenticated
	 * @throws RequestException if the server returns an error code
	 * @throws IOException if one of the items could not be read from the server
	 * @see #getMaxBulkItems()
	 */
	Map<String, CSLItemData> getItems(List<String> itemIds) throws IOException;
	
	/**
	 * @return the maximum number of items that can be queried within one request
	 * @see #getItems(List)
	 */
	int getMaxBulkItems();
}
