// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A set of citation items. This class is used to register citations in the CSL
 * processor.
 * 
 * @author Michel Kraemer
 */
public class CSLCitation implements JsonObject {
	private final CSLCitationItem[] citationItems;

	private final String citationID;
	private final CSLProperties properties;

	public CSLCitation(CSLCitationItem... citationItems) {
		this.citationItems = citationItems;

		this.citationID = Long.toString(
				(long) Math.floor(Math.random() * 100000000000000L), 32);
		this.properties = new CSLProperties();

	}

	public CSLCitation(CSLCitationItem[] citationItems, String citationID,
			CSLProperties properties) {
		this.citationItems = citationItems;

		this.citationID = citationID;
		this.properties = properties;

	}

	/**
	 * @return the citation's citationItems
	 */
	public CSLCitationItem[] getCitationItems() {
		return citationItems;
	}

	/**
	 * @return the citation's citationID
	 */
	public String getCitationID() {
		return citationID;
	}
	/**
	 * @return the citation's properties
	 */
	public CSLProperties getProperties() {
		return properties;
	}

	@Override
	public Object toJson(JsonBuilder builder) {
		builder.add("citationItems", citationItems);
		if (citationID != null) {
			builder.add("citationID", citationID);
		}
		if (properties != null) {
			builder.add("properties", properties);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLCitation object. The JSON object must at
	 * least contain the following required properties:
	 * <code>citationItems</code>
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLCitation object
	 */
	@SuppressWarnings("unchecked")
	public static CSLCitation fromJson(Map<String, Object> obj) {
		CSLCitationItem[] citationItems;

		{
			Object v = obj.get("citationItems");
			if (v == null) {
				throw new IllegalArgumentException(
						"Missing property `citationItems'");
			}
			if (v instanceof Map) {
				v = ((Map<?, ?>) v).values();
			} else if (!(v instanceof Collection)) {
				throw new IllegalArgumentException(
						"`citationItems' must be an array");
			}
			Collection<?> cv = (Collection<?>) v;
			citationItems = new CSLCitationItem[cv.size()];
			int i = 0;
			for (Object vo : cv) {
				if (!(vo instanceof Map)) {
					throw new IllegalArgumentException(
							"`citationItems' must be an array of objects");
				}
				citationItems[i] = CSLCitationItem
						.fromJson((Map<String, Object>) vo);
				++i;
			}
		}

		CSLCitationBuilder builder = new CSLCitationBuilder(citationItems);

		{
			Object v = obj.get("citationID");
			if (v != null) {
				String citationID;
				citationID = v.toString();
				builder.citationID(citationID);
			} else {
				builder.citationID(Long.toString(
						(long) Math.floor(Math.random() * 100000000000000L), 32));
			}

		}
		{
			Object v = obj.get("properties");
			if (v != null) {
				CSLProperties properties;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`properties' must be an object");
				}
				properties = CSLProperties.fromJson((Map<String, Object>) v);
				builder.properties(properties);
			} else {
				builder.properties(new CSLProperties());
			}

		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result
				+ ((citationID == null) ? 0 : citationID.hashCode());
		result = 31 * result
				+ ((properties == null) ? 0 : properties.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLCitation))
			return false;
		CSLCitation other = (CSLCitation) obj;

		if (citationID == null) {
			if (other.citationID != null)
				return false;
		} else if (!citationID.equals(other.citationID))
			return false;

		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;

		return true;
	}
}
