// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * Citation cluster properties.
 * 
 * @author Michel Kraemer
 */
public class CSLProperties implements JsonObject {

	private final Integer noteIndex;
	private final Boolean unsorted;

	public CSLProperties() {

		this.noteIndex = 0;
		this.unsorted = null;

	}

	public CSLProperties(Integer noteIndex, Boolean unsorted) {

		this.noteIndex = noteIndex;
		this.unsorted = unsorted;

	}

	/**
	 * @return the noteIndex
	 */
	public Integer getNoteIndex() {
		return noteIndex;
	}
	/**
	 * @return the unsorted
	 */
	public Boolean getUnsorted() {
		return unsorted;
	}

	@Override
	public Object toJson(JsonBuilder builder) {

		if (noteIndex != null) {
			builder.add("noteIndex", noteIndex);
		}
		if (unsorted != null) {
			builder.add("unsorted", unsorted);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLProperties object.
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLProperties object
	 */
	@SuppressWarnings("unchecked")
	public static CSLProperties fromJson(Map<String, Object> obj) {

		CSLPropertiesBuilder builder = new CSLPropertiesBuilder();

		{
			Object v = obj.get("noteIndex");
			if (v != null) {
				Integer noteIndex;
				noteIndex = toInt(v);
				builder.noteIndex(noteIndex);
			} else {
				builder.noteIndex(0);
			}

		}
		{
			Object v = obj.get("unsorted");
			if (v != null) {
				Boolean unsorted;
				unsorted = toBool(v);
				builder.unsorted(unsorted);
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((noteIndex == null) ? 0 : noteIndex.hashCode());
		result = 31 * result + ((unsorted == null) ? 0 : unsorted.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLProperties))
			return false;
		CSLProperties other = (CSLProperties) obj;

		if (noteIndex == null) {
			if (other.noteIndex != null)
				return false;
		} else if (!noteIndex.equals(other.noteIndex))
			return false;

		if (unsorted == null) {
			if (other.unsorted != null)
				return false;
		} else if (!unsorted.equals(other.unsorted))
			return false;

		return true;
	}
}
