// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A human being's name
 * 
 * @author Michel Kraemer
 */
public class CSLName implements JsonObject {

	private final String family;
	private final String given;
	private final String droppingParticle;
	private final String nonDroppingParticle;
	private final String suffix;
	private final Boolean commaPrefix;
	private final Boolean commaSuffix;
	private final Boolean staticOrdering;
	private final Boolean staticParticles;
	private final String literal;
	private final Boolean parseNames;
	private final Boolean isInstitution;

	public CSLName() {

		this.family = null;
		this.given = null;
		this.droppingParticle = null;
		this.nonDroppingParticle = null;
		this.suffix = null;
		this.commaPrefix = null;
		this.commaSuffix = null;
		this.staticOrdering = null;
		this.staticParticles = null;
		this.literal = null;
		this.parseNames = null;
		this.isInstitution = null;

	}

	public CSLName(String family, String given, String droppingParticle,
			String nonDroppingParticle, String suffix, Boolean commaPrefix,
			Boolean commaSuffix, Boolean staticOrdering,
			Boolean staticParticles, String literal, Boolean parseNames,
			Boolean isInstitution) {

		this.family = family;
		this.given = given;
		this.droppingParticle = droppingParticle;
		this.nonDroppingParticle = nonDroppingParticle;
		this.suffix = suffix;
		this.commaPrefix = commaPrefix;
		this.commaSuffix = commaSuffix;
		this.staticOrdering = staticOrdering;
		this.staticParticles = staticParticles;
		this.literal = literal;
		this.parseNames = parseNames;
		this.isInstitution = isInstitution;

	}

	/**
	 * @return the family
	 */
	public String getFamily() {
		return family;
	}
	/**
	 * @return the given
	 */
	public String getGiven() {
		return given;
	}
	/**
	 * @return the dropping-particle
	 */
	public String getDroppingParticle() {
		return droppingParticle;
	}
	/**
	 * @return the non-dropping-particle
	 */
	public String getNonDroppingParticle() {
		return nonDroppingParticle;
	}
	/**
	 * @return the suffix
	 */
	public String getSuffix() {
		return suffix;
	}
	/**
	 * @return the comma-prefix
	 */
	public Boolean getCommaPrefix() {
		return commaPrefix;
	}
	/**
	 * @return the comma-suffix
	 */
	public Boolean getCommaSuffix() {
		return commaSuffix;
	}
	/**
	 * @return the static-ordering
	 */
	public Boolean getStaticOrdering() {
		return staticOrdering;
	}
	/**
	 * @return the static-particles
	 */
	public Boolean getStaticParticles() {
		return staticParticles;
	}
	/**
	 * @return the literal
	 */
	public String getLiteral() {
		return literal;
	}
	/**
	 * @return the parse-names
	 */
	public Boolean getParseNames() {
		return parseNames;
	}
	/**
	 * @return the isInstitution
	 */
	public Boolean getIsInstitution() {
		return isInstitution;
	}

	@Override
	public Object toJson(JsonBuilder builder) {

		if (family != null) {
			builder.add("family", family);
		}
		if (given != null) {
			builder.add("given", given);
		}
		if (droppingParticle != null) {
			builder.add("dropping-particle", droppingParticle);
		}
		if (nonDroppingParticle != null) {
			builder.add("non-dropping-particle", nonDroppingParticle);
		}
		if (suffix != null) {
			builder.add("suffix", suffix);
		}
		if (commaPrefix != null) {
			builder.add("comma-prefix", commaPrefix);
		}
		if (commaSuffix != null) {
			builder.add("comma-suffix", commaSuffix);
		}
		if (staticOrdering != null) {
			builder.add("static-ordering", staticOrdering);
		}
		if (staticParticles != null) {
			builder.add("static-particles", staticParticles);
		}
		if (literal != null) {
			builder.add("literal", literal);
		}
		if (parseNames != null) {
			builder.add("parse-names", parseNames);
		}
		if (isInstitution != null) {
			builder.add("isInstitution", isInstitution);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLName object.
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLName object
	 */
	@SuppressWarnings("unchecked")
	public static CSLName fromJson(Map<String, Object> obj) {

		CSLNameBuilder builder = new CSLNameBuilder();

		{
			Object v = obj.get("family");
			if (v != null) {
				String family;
				family = v.toString();
				builder.family(family);
			}
		}
		{
			Object v = obj.get("given");
			if (v != null) {
				String given;
				given = v.toString();
				builder.given(given);
			}
		}
		{
			Object v = obj.get("dropping-particle");
			if (v != null) {
				String droppingParticle;
				droppingParticle = v.toString();
				builder.droppingParticle(droppingParticle);
			}
		}
		{
			Object v = obj.get("non-dropping-particle");
			if (v != null) {
				String nonDroppingParticle;
				nonDroppingParticle = v.toString();
				builder.nonDroppingParticle(nonDroppingParticle);
			}
		}
		{
			Object v = obj.get("suffix");
			if (v != null) {
				String suffix;
				suffix = v.toString();
				builder.suffix(suffix);
			}
		}
		{
			Object v = obj.get("comma-prefix");
			if (v != null) {
				Boolean commaPrefix;
				commaPrefix = toBool(v);
				builder.commaPrefix(commaPrefix);
			}
		}
		{
			Object v = obj.get("comma-suffix");
			if (v != null) {
				Boolean commaSuffix;
				commaSuffix = toBool(v);
				builder.commaSuffix(commaSuffix);
			}
		}
		{
			Object v = obj.get("static-ordering");
			if (v != null) {
				Boolean staticOrdering;
				staticOrdering = toBool(v);
				builder.staticOrdering(staticOrdering);
			}
		}
		{
			Object v = obj.get("static-particles");
			if (v != null) {
				Boolean staticParticles;
				staticParticles = toBool(v);
				builder.staticParticles(staticParticles);
			}
		}
		{
			Object v = obj.get("literal");
			if (v != null) {
				String literal;
				literal = v.toString();
				builder.literal(literal);
			}
		}
		{
			Object v = obj.get("parse-names");
			if (v != null) {
				Boolean parseNames;
				parseNames = toBool(v);
				builder.parseNames(parseNames);
			}
		}
		{
			Object v = obj.get("isInstitution");
			if (v != null) {
				Boolean isInstitution;
				isInstitution = toBool(v);
				builder.isInstitution(isInstitution);
			}
		}

		{
			Object v = obj.get("multi");
			if (v != null && v instanceof Map) {
				Map<?, ?> vc = (Map<?, ?>) v;
				Object vcmain = vc.get("main");
				if (vcmain != null && vcmain instanceof String) {
					String mstr = (String) vcmain;
					mstr = mstr.toLowerCase().substring(0, 2);
					if (mstr.equals("hu") || mstr.equals("vi")) {
						builder.staticOrdering(true);
					}
				}
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((family == null) ? 0 : family.hashCode());
		result = 31 * result + ((given == null) ? 0 : given.hashCode());
		result = 31
				* result
				+ ((droppingParticle == null) ? 0 : droppingParticle.hashCode());
		result = 31
				* result
				+ ((nonDroppingParticle == null) ? 0 : nonDroppingParticle
						.hashCode());
		result = 31 * result + ((suffix == null) ? 0 : suffix.hashCode());
		result = 31 * result
				+ ((commaPrefix == null) ? 0 : commaPrefix.hashCode());
		result = 31 * result
				+ ((commaSuffix == null) ? 0 : commaSuffix.hashCode());
		result = 31 * result
				+ ((staticOrdering == null) ? 0 : staticOrdering.hashCode());
		result = 31 * result
				+ ((staticParticles == null) ? 0 : staticParticles.hashCode());
		result = 31 * result + ((literal == null) ? 0 : literal.hashCode());
		result = 31 * result
				+ ((parseNames == null) ? 0 : parseNames.hashCode());
		result = 31 * result
				+ ((isInstitution == null) ? 0 : isInstitution.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLName))
			return false;
		CSLName other = (CSLName) obj;

		if (family == null) {
			if (other.family != null)
				return false;
		} else if (!family.equals(other.family))
			return false;

		if (given == null) {
			if (other.given != null)
				return false;
		} else if (!given.equals(other.given))
			return false;

		if (droppingParticle == null) {
			if (other.droppingParticle != null)
				return false;
		} else if (!droppingParticle.equals(other.droppingParticle))
			return false;

		if (nonDroppingParticle == null) {
			if (other.nonDroppingParticle != null)
				return false;
		} else if (!nonDroppingParticle.equals(other.nonDroppingParticle))
			return false;

		if (suffix == null) {
			if (other.suffix != null)
				return false;
		} else if (!suffix.equals(other.suffix))
			return false;

		if (commaPrefix == null) {
			if (other.commaPrefix != null)
				return false;
		} else if (!commaPrefix.equals(other.commaPrefix))
			return false;

		if (commaSuffix == null) {
			if (other.commaSuffix != null)
				return false;
		} else if (!commaSuffix.equals(other.commaSuffix))
			return false;

		if (staticOrdering == null) {
			if (other.staticOrdering != null)
				return false;
		} else if (!staticOrdering.equals(other.staticOrdering))
			return false;

		if (staticParticles == null) {
			if (other.staticParticles != null)
				return false;
		} else if (!staticParticles.equals(other.staticParticles))
			return false;

		if (literal == null) {
			if (other.literal != null)
				return false;
		} else if (!literal.equals(other.literal))
			return false;

		if (parseNames == null) {
			if (other.parseNames != null)
				return false;
		} else if (!parseNames.equals(other.parseNames))
			return false;

		if (isInstitution == null) {
			if (other.isInstitution != null)
				return false;
		} else if (!isInstitution.equals(other.isInstitution))
			return false;

		return true;
	}
}
