// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A citation item contains all attributes that make up a citation.
 * 
 * @author Michel Kraemer
 */
public class CSLItemData implements JsonObject {

	private final String id;
	private final CSLType type;
	private final String[] categories;
	private final String language;
	private final String journalAbbreviation;
	private final String shortTitle;
	private final CSLName[] author;
	private final CSLName[] collectionEditor;
	private final CSLName[] composer;
	private final CSLName[] containerAuthor;
	private final CSLName[] director;
	private final CSLName[] editor;
	private final CSLName[] editorialDirector;
	private final CSLName[] interviewer;
	private final CSLName[] illustrator;
	private final CSLName[] originalAuthor;
	private final CSLName[] recipient;
	private final CSLName[] reviewedAuthor;
	private final CSLName[] translator;
	private final CSLDate accessed;
	private final CSLDate container;
	private final CSLDate eventDate;
	private final CSLDate issued;
	private final CSLDate originalDate;
	private final CSLDate submitted;
	private final String abstrct;
	private final String annote;
	private final String archive;
	private final String archiveLocation;
	private final String archivePlace;
	private final String authority;
	private final String callNumber;
	private final String chapterNumber;
	private final String citationNumber;
	private final String citationLabel;
	private final String collectionNumber;
	private final String collectionTitle;
	private final String containerTitle;
	private final String containerTitleShort;
	private final String dimensions;
	private final String DOI;
	private final String edition;
	private final String event;
	private final String eventPlace;
	private final String firstReferenceNoteNumber;
	private final String genre;
	private final String ISBN;
	private final String ISSN;
	private final String issue;
	private final String jurisdiction;
	private final String keyword;
	private final String locator;
	private final String medium;
	private final String note;
	private final String number;
	private final String numberOfPages;
	private final String numberOfVolumes;
	private final String originalPublisher;
	private final String originalPublisherPlace;
	private final String originalTitle;
	private final String page;
	private final String pageFirst;
	private final String PMCID;
	private final String PMID;
	private final String publisher;
	private final String publisherPlace;
	private final String references;
	private final String reviewedTitle;
	private final String scale;
	private final String section;
	private final String source;
	private final String status;
	private final String title;
	private final String titleShort;
	private final String URL;
	private final String version;
	private final String volume;
	private final String yearSuffix;

	public CSLItemData() {

		this.id = "-GEN-"
				+ Long.toString(
						(long) Math.floor(Math.random() * 100000000000000L), 32);
		this.type = null;
		this.categories = null;
		this.language = null;
		this.journalAbbreviation = null;
		this.shortTitle = null;
		this.author = null;
		this.collectionEditor = null;
		this.composer = null;
		this.containerAuthor = null;
		this.director = null;
		this.editor = null;
		this.editorialDirector = null;
		this.interviewer = null;
		this.illustrator = null;
		this.originalAuthor = null;
		this.recipient = null;
		this.reviewedAuthor = null;
		this.translator = null;
		this.accessed = null;
		this.container = null;
		this.eventDate = null;
		this.issued = null;
		this.originalDate = null;
		this.submitted = null;
		this.abstrct = null;
		this.annote = null;
		this.archive = null;
		this.archiveLocation = null;
		this.archivePlace = null;
		this.authority = null;
		this.callNumber = null;
		this.chapterNumber = null;
		this.citationNumber = null;
		this.citationLabel = null;
		this.collectionNumber = null;
		this.collectionTitle = null;
		this.containerTitle = null;
		this.containerTitleShort = null;
		this.dimensions = null;
		this.DOI = null;
		this.edition = null;
		this.event = null;
		this.eventPlace = null;
		this.firstReferenceNoteNumber = null;
		this.genre = null;
		this.ISBN = null;
		this.ISSN = null;
		this.issue = null;
		this.jurisdiction = null;
		this.keyword = null;
		this.locator = null;
		this.medium = null;
		this.note = null;
		this.number = null;
		this.numberOfPages = null;
		this.numberOfVolumes = null;
		this.originalPublisher = null;
		this.originalPublisherPlace = null;
		this.originalTitle = null;
		this.page = null;
		this.pageFirst = null;
		this.PMCID = null;
		this.PMID = null;
		this.publisher = null;
		this.publisherPlace = null;
		this.references = null;
		this.reviewedTitle = null;
		this.scale = null;
		this.section = null;
		this.source = null;
		this.status = null;
		this.title = null;
		this.titleShort = null;
		this.URL = null;
		this.version = null;
		this.volume = null;
		this.yearSuffix = null;

	}

	public CSLItemData(String id, CSLType type, String[] categories,
			String language, String journalAbbreviation, String shortTitle,
			CSLName[] author, CSLName[] collectionEditor, CSLName[] composer,
			CSLName[] containerAuthor, CSLName[] director, CSLName[] editor,
			CSLName[] editorialDirector, CSLName[] interviewer,
			CSLName[] illustrator, CSLName[] originalAuthor,
			CSLName[] recipient, CSLName[] reviewedAuthor,
			CSLName[] translator, CSLDate accessed, CSLDate container,
			CSLDate eventDate, CSLDate issued, CSLDate originalDate,
			CSLDate submitted, String abstrct, String annote, String archive,
			String archiveLocation, String archivePlace, String authority,
			String callNumber, String chapterNumber, String citationNumber,
			String citationLabel, String collectionNumber,
			String collectionTitle, String containerTitle,
			String containerTitleShort, String dimensions, String DOI,
			String edition, String event, String eventPlace,
			String firstReferenceNoteNumber, String genre, String ISBN,
			String ISSN, String issue, String jurisdiction, String keyword,
			String locator, String medium, String note, String number,
			String numberOfPages, String numberOfVolumes,
			String originalPublisher, String originalPublisherPlace,
			String originalTitle, String page, String pageFirst, String PMCID,
			String PMID, String publisher, String publisherPlace,
			String references, String reviewedTitle, String scale,
			String section, String source, String status, String title,
			String titleShort, String URL, String version, String volume,
			String yearSuffix) {

		this.id = id;
		this.type = type;
		this.categories = categories;
		this.language = language;
		this.journalAbbreviation = journalAbbreviation;
		this.shortTitle = shortTitle;
		this.author = author;
		this.collectionEditor = collectionEditor;
		this.composer = composer;
		this.containerAuthor = containerAuthor;
		this.director = director;
		this.editor = editor;
		this.editorialDirector = editorialDirector;
		this.interviewer = interviewer;
		this.illustrator = illustrator;
		this.originalAuthor = originalAuthor;
		this.recipient = recipient;
		this.reviewedAuthor = reviewedAuthor;
		this.translator = translator;
		this.accessed = accessed;
		this.container = container;
		this.eventDate = eventDate;
		this.issued = issued;
		this.originalDate = originalDate;
		this.submitted = submitted;
		this.abstrct = abstrct;
		this.annote = annote;
		this.archive = archive;
		this.archiveLocation = archiveLocation;
		this.archivePlace = archivePlace;
		this.authority = authority;
		this.callNumber = callNumber;
		this.chapterNumber = chapterNumber;
		this.citationNumber = citationNumber;
		this.citationLabel = citationLabel;
		this.collectionNumber = collectionNumber;
		this.collectionTitle = collectionTitle;
		this.containerTitle = containerTitle;
		this.containerTitleShort = containerTitleShort;
		this.dimensions = dimensions;
		this.DOI = DOI;
		this.edition = edition;
		this.event = event;
		this.eventPlace = eventPlace;
		this.firstReferenceNoteNumber = firstReferenceNoteNumber;
		this.genre = genre;
		this.ISBN = ISBN;
		this.ISSN = ISSN;
		this.issue = issue;
		this.jurisdiction = jurisdiction;
		this.keyword = keyword;
		this.locator = locator;
		this.medium = medium;
		this.note = note;
		this.number = number;
		this.numberOfPages = numberOfPages;
		this.numberOfVolumes = numberOfVolumes;
		this.originalPublisher = originalPublisher;
		this.originalPublisherPlace = originalPublisherPlace;
		this.originalTitle = originalTitle;
		this.page = page;
		this.pageFirst = pageFirst;
		this.PMCID = PMCID;
		this.PMID = PMID;
		this.publisher = publisher;
		this.publisherPlace = publisherPlace;
		this.references = references;
		this.reviewedTitle = reviewedTitle;
		this.scale = scale;
		this.section = section;
		this.source = source;
		this.status = status;
		this.title = title;
		this.titleShort = titleShort;
		this.URL = URL;
		this.version = version;
		this.volume = volume;
		this.yearSuffix = yearSuffix;

	}

	/**
	 * @return the citation item's id
	 */
	public String getId() {
		return id;
	}
	/**
	 * @return the citation item's type
	 */
	public CSLType getType() {
		return type;
	}
	/**
	 * @return the citation item's categories
	 */
	public String[] getCategories() {
		return categories;
	}
	/**
	 * @return the citation item's language
	 */
	public String getLanguage() {
		return language;
	}
	/**
	 * @return the citation item's journalAbbreviation
	 */
	public String getJournalAbbreviation() {
		return journalAbbreviation;
	}
	/**
	 * @return the citation item's shortTitle
	 */
	public String getShortTitle() {
		return shortTitle;
	}
	/**
	 * @return the citation item's author
	 */
	public CSLName[] getAuthor() {
		return author;
	}
	/**
	 * @return the citation item's collection-editor
	 */
	public CSLName[] getCollectionEditor() {
		return collectionEditor;
	}
	/**
	 * @return the citation item's composer
	 */
	public CSLName[] getComposer() {
		return composer;
	}
	/**
	 * @return the citation item's container-author
	 */
	public CSLName[] getContainerAuthor() {
		return containerAuthor;
	}
	/**
	 * @return the citation item's director
	 */
	public CSLName[] getDirector() {
		return director;
	}
	/**
	 * @return the citation item's editor
	 */
	public CSLName[] getEditor() {
		return editor;
	}
	/**
	 * @return the citation item's editorial-director
	 */
	public CSLName[] getEditorialDirector() {
		return editorialDirector;
	}
	/**
	 * @return the citation item's interviewer
	 */
	public CSLName[] getInterviewer() {
		return interviewer;
	}
	/**
	 * @return the citation item's illustrator
	 */
	public CSLName[] getIllustrator() {
		return illustrator;
	}
	/**
	 * @return the citation item's original-author
	 */
	public CSLName[] getOriginalAuthor() {
		return originalAuthor;
	}
	/**
	 * @return the citation item's recipient
	 */
	public CSLName[] getRecipient() {
		return recipient;
	}
	/**
	 * @return the citation item's reviewed-author
	 */
	public CSLName[] getReviewedAuthor() {
		return reviewedAuthor;
	}
	/**
	 * @return the citation item's translator
	 */
	public CSLName[] getTranslator() {
		return translator;
	}
	/**
	 * @return the citation item's accessed
	 */
	public CSLDate getAccessed() {
		return accessed;
	}
	/**
	 * @return the citation item's container
	 */
	public CSLDate getContainer() {
		return container;
	}
	/**
	 * @return the citation item's event-date
	 */
	public CSLDate getEventDate() {
		return eventDate;
	}
	/**
	 * @return the citation item's issued
	 */
	public CSLDate getIssued() {
		return issued;
	}
	/**
	 * @return the citation item's original-date
	 */
	public CSLDate getOriginalDate() {
		return originalDate;
	}
	/**
	 * @return the citation item's submitted
	 */
	public CSLDate getSubmitted() {
		return submitted;
	}
	/**
	 * @return the citation item's abstract
	 */
	public String getAbstrct() {
		return abstrct;
	}
	/**
	 * @return the citation item's annote
	 */
	public String getAnnote() {
		return annote;
	}
	/**
	 * @return the citation item's archive
	 */
	public String getArchive() {
		return archive;
	}
	/**
	 * @return the citation item's archive_location
	 */
	public String getArchiveLocation() {
		return archiveLocation;
	}
	/**
	 * @return the citation item's archive-place
	 */
	public String getArchivePlace() {
		return archivePlace;
	}
	/**
	 * @return the citation item's authority
	 */
	public String getAuthority() {
		return authority;
	}
	/**
	 * @return the citation item's call-number
	 */
	public String getCallNumber() {
		return callNumber;
	}
	/**
	 * @return the citation item's chapter-number
	 */
	public String getChapterNumber() {
		return chapterNumber;
	}
	/**
	 * @return the citation item's citation-number
	 */
	public String getCitationNumber() {
		return citationNumber;
	}
	/**
	 * @return the citation item's citation-label
	 */
	public String getCitationLabel() {
		return citationLabel;
	}
	/**
	 * @return the citation item's collection-number
	 */
	public String getCollectionNumber() {
		return collectionNumber;
	}
	/**
	 * @return the citation item's collection-title
	 */
	public String getCollectionTitle() {
		return collectionTitle;
	}
	/**
	 * @return the citation item's container-title
	 */
	public String getContainerTitle() {
		return containerTitle;
	}
	/**
	 * @return the citation item's container-title-short
	 */
	public String getContainerTitleShort() {
		return containerTitleShort;
	}
	/**
	 * @return the citation item's dimensions
	 */
	public String getDimensions() {
		return dimensions;
	}
	/**
	 * @return the citation item's DOI
	 */
	public String getDOI() {
		return DOI;
	}
	/**
	 * @return the citation item's edition
	 */
	public String getEdition() {
		return edition;
	}
	/**
	 * @return the citation item's event
	 */
	public String getEvent() {
		return event;
	}
	/**
	 * @return the citation item's event-place
	 */
	public String getEventPlace() {
		return eventPlace;
	}
	/**
	 * @return the citation item's first-reference-note-number
	 */
	public String getFirstReferenceNoteNumber() {
		return firstReferenceNoteNumber;
	}
	/**
	 * @return the citation item's genre
	 */
	public String getGenre() {
		return genre;
	}
	/**
	 * @return the citation item's ISBN
	 */
	public String getISBN() {
		return ISBN;
	}
	/**
	 * @return the citation item's ISSN
	 */
	public String getISSN() {
		return ISSN;
	}
	/**
	 * @return the citation item's issue
	 */
	public String getIssue() {
		return issue;
	}
	/**
	 * @return the citation item's jurisdiction
	 */
	public String getJurisdiction() {
		return jurisdiction;
	}
	/**
	 * @return the citation item's keyword
	 */
	public String getKeyword() {
		return keyword;
	}
	/**
	 * @return the citation item's locator
	 */
	public String getLocator() {
		return locator;
	}
	/**
	 * @return the citation item's medium
	 */
	public String getMedium() {
		return medium;
	}
	/**
	 * @return the citation item's note
	 */
	public String getNote() {
		return note;
	}
	/**
	 * @return the citation item's number
	 */
	public String getNumber() {
		return number;
	}
	/**
	 * @return the citation item's number-of-pages
	 */
	public String getNumberOfPages() {
		return numberOfPages;
	}
	/**
	 * @return the citation item's number-of-volumes
	 */
	public String getNumberOfVolumes() {
		return numberOfVolumes;
	}
	/**
	 * @return the citation item's original-publisher
	 */
	public String getOriginalPublisher() {
		return originalPublisher;
	}
	/**
	 * @return the citation item's original-publisher-place
	 */
	public String getOriginalPublisherPlace() {
		return originalPublisherPlace;
	}
	/**
	 * @return the citation item's original-title
	 */
	public String getOriginalTitle() {
		return originalTitle;
	}
	/**
	 * @return the citation item's page
	 */
	public String getPage() {
		return page;
	}
	/**
	 * @return the citation item's page-first
	 */
	public String getPageFirst() {
		return pageFirst;
	}
	/**
	 * @return the citation item's PMCID
	 */
	public String getPMCID() {
		return PMCID;
	}
	/**
	 * @return the citation item's PMID
	 */
	public String getPMID() {
		return PMID;
	}
	/**
	 * @return the citation item's publisher
	 */
	public String getPublisher() {
		return publisher;
	}
	/**
	 * @return the citation item's publisher-place
	 */
	public String getPublisherPlace() {
		return publisherPlace;
	}
	/**
	 * @return the citation item's references
	 */
	public String getReferences() {
		return references;
	}
	/**
	 * @return the citation item's reviewed-title
	 */
	public String getReviewedTitle() {
		return reviewedTitle;
	}
	/**
	 * @return the citation item's scale
	 */
	public String getScale() {
		return scale;
	}
	/**
	 * @return the citation item's section
	 */
	public String getSection() {
		return section;
	}
	/**
	 * @return the citation item's source
	 */
	public String getSource() {
		return source;
	}
	/**
	 * @return the citation item's status
	 */
	public String getStatus() {
		return status;
	}
	/**
	 * @return the citation item's title
	 */
	public String getTitle() {
		return title;
	}
	/**
	 * @return the citation item's title-short
	 */
	public String getTitleShort() {
		return titleShort;
	}
	/**
	 * @return the citation item's URL
	 */
	public String getURL() {
		return URL;
	}
	/**
	 * @return the citation item's version
	 */
	public String getVersion() {
		return version;
	}
	/**
	 * @return the citation item's volume
	 */
	public String getVolume() {
		return volume;
	}
	/**
	 * @return the citation item's year-suffix
	 */
	public String getYearSuffix() {
		return yearSuffix;
	}

	@Override
	public Object toJson(JsonBuilder builder) {

		if (id != null) {
			builder.add("id", id);
		}
		if (type != null) {
			builder.add("type", type);
		}
		if (categories != null) {
			builder.add("categories", categories);
		}
		if (language != null) {
			builder.add("language", language);
		}
		if (journalAbbreviation != null) {
			builder.add("journalAbbreviation", journalAbbreviation);
		}
		if (shortTitle != null) {
			builder.add("shortTitle", shortTitle);
		}
		if (author != null) {
			builder.add("author", author);
		}
		if (collectionEditor != null) {
			builder.add("collection-editor", collectionEditor);
		}
		if (composer != null) {
			builder.add("composer", composer);
		}
		if (containerAuthor != null) {
			builder.add("container-author", containerAuthor);
		}
		if (director != null) {
			builder.add("director", director);
		}
		if (editor != null) {
			builder.add("editor", editor);
		}
		if (editorialDirector != null) {
			builder.add("editorial-director", editorialDirector);
		}
		if (interviewer != null) {
			builder.add("interviewer", interviewer);
		}
		if (illustrator != null) {
			builder.add("illustrator", illustrator);
		}
		if (originalAuthor != null) {
			builder.add("original-author", originalAuthor);
		}
		if (recipient != null) {
			builder.add("recipient", recipient);
		}
		if (reviewedAuthor != null) {
			builder.add("reviewed-author", reviewedAuthor);
		}
		if (translator != null) {
			builder.add("translator", translator);
		}
		if (accessed != null) {
			builder.add("accessed", accessed);
		}
		if (container != null) {
			builder.add("container", container);
		}
		if (eventDate != null) {
			builder.add("event-date", eventDate);
		}
		if (issued != null) {
			builder.add("issued", issued);
		}
		if (originalDate != null) {
			builder.add("original-date", originalDate);
		}
		if (submitted != null) {
			builder.add("submitted", submitted);
		}
		if (abstrct != null) {
			builder.add("abstract", abstrct);
		}
		if (annote != null) {
			builder.add("annote", annote);
		}
		if (archive != null) {
			builder.add("archive", archive);
		}
		if (archiveLocation != null) {
			builder.add("archive_location", archiveLocation);
		}
		if (archivePlace != null) {
			builder.add("archive-place", archivePlace);
		}
		if (authority != null) {
			builder.add("authority", authority);
		}
		if (callNumber != null) {
			builder.add("call-number", callNumber);
		}
		if (chapterNumber != null) {
			builder.add("chapter-number", chapterNumber);
		}
		if (citationNumber != null) {
			builder.add("citation-number", citationNumber);
		}
		if (citationLabel != null) {
			builder.add("citation-label", citationLabel);
		}
		if (collectionNumber != null) {
			builder.add("collection-number", collectionNumber);
		}
		if (collectionTitle != null) {
			builder.add("collection-title", collectionTitle);
		}
		if (containerTitle != null) {
			builder.add("container-title", containerTitle);
		}
		if (containerTitleShort != null) {
			builder.add("container-title-short", containerTitleShort);
		}
		if (dimensions != null) {
			builder.add("dimensions", dimensions);
		}
		if (DOI != null) {
			builder.add("DOI", DOI);
		}
		if (edition != null) {
			builder.add("edition", edition);
		}
		if (event != null) {
			builder.add("event", event);
		}
		if (eventPlace != null) {
			builder.add("event-place", eventPlace);
		}
		if (firstReferenceNoteNumber != null) {
			builder.add("first-reference-note-number", firstReferenceNoteNumber);
		}
		if (genre != null) {
			builder.add("genre", genre);
		}
		if (ISBN != null) {
			builder.add("ISBN", ISBN);
		}
		if (ISSN != null) {
			builder.add("ISSN", ISSN);
		}
		if (issue != null) {
			builder.add("issue", issue);
		}
		if (jurisdiction != null) {
			builder.add("jurisdiction", jurisdiction);
		}
		if (keyword != null) {
			builder.add("keyword", keyword);
		}
		if (locator != null) {
			builder.add("locator", locator);
		}
		if (medium != null) {
			builder.add("medium", medium);
		}
		if (note != null) {
			builder.add("note", note);
		}
		if (number != null) {
			builder.add("number", number);
		}
		if (numberOfPages != null) {
			builder.add("number-of-pages", numberOfPages);
		}
		if (numberOfVolumes != null) {
			builder.add("number-of-volumes", numberOfVolumes);
		}
		if (originalPublisher != null) {
			builder.add("original-publisher", originalPublisher);
		}
		if (originalPublisherPlace != null) {
			builder.add("original-publisher-place", originalPublisherPlace);
		}
		if (originalTitle != null) {
			builder.add("original-title", originalTitle);
		}
		if (page != null) {
			builder.add("page", page);
		}
		if (pageFirst != null) {
			builder.add("page-first", pageFirst);
		}
		if (PMCID != null) {
			builder.add("PMCID", PMCID);
		}
		if (PMID != null) {
			builder.add("PMID", PMID);
		}
		if (publisher != null) {
			builder.add("publisher", publisher);
		}
		if (publisherPlace != null) {
			builder.add("publisher-place", publisherPlace);
		}
		if (references != null) {
			builder.add("references", references);
		}
		if (reviewedTitle != null) {
			builder.add("reviewed-title", reviewedTitle);
		}
		if (scale != null) {
			builder.add("scale", scale);
		}
		if (section != null) {
			builder.add("section", section);
		}
		if (source != null) {
			builder.add("source", source);
		}
		if (status != null) {
			builder.add("status", status);
		}
		if (title != null) {
			builder.add("title", title);
		}
		if (titleShort != null) {
			builder.add("title-short", titleShort);
		}
		if (URL != null) {
			builder.add("URL", URL);
		}
		if (version != null) {
			builder.add("version", version);
		}
		if (volume != null) {
			builder.add("volume", volume);
		}
		if (yearSuffix != null) {
			builder.add("year-suffix", yearSuffix);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLItemData object.
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLItemData object
	 */
	@SuppressWarnings("unchecked")
	public static CSLItemData fromJson(Map<String, Object> obj) {

		CSLItemDataBuilder builder = new CSLItemDataBuilder();

		{
			Object v = obj.get("id");
			if (v != null) {
				String id;
				id = v.toString();
				builder.id(id);
			} else {
				builder.id("-GEN-"
						+ Long.toString(
								(long) Math
										.floor(Math.random() * 100000000000000L),
								32));
			}

		}
		{
			Object v = obj.get("type");
			if (!isFalsy(v)) {
				CSLType type;
				type = CSLType.fromString(v.toString());
				builder.type(type);
			}
		}
		{
			Object v = obj.get("categories");
			if (v != null) {
				String[] categories;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`categories' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				categories = new String[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					categories[i] = vo.toString();
					++i;
				}
				builder.categories(categories);
			}
		}
		{
			Object v = obj.get("language");
			if (v != null) {
				String language;
				language = v.toString();
				builder.language(language);
			}
		}
		{
			Object v = obj.get("journalAbbreviation");
			if (v != null) {
				String journalAbbreviation;
				journalAbbreviation = v.toString();
				builder.journalAbbreviation(journalAbbreviation);
			}
		}
		{
			Object v = obj.get("shortTitle");
			if (v != null) {
				String shortTitle;
				shortTitle = v.toString();
				builder.shortTitle(shortTitle);
			}
		}
		{
			Object v = obj.get("author");
			if (v != null) {
				CSLName[] author;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`author' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				author = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`author' must be an array of objects");
					}
					author[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.author(author);
			}
		}
		{
			Object v = obj.get("collection-editor");
			if (v != null) {
				CSLName[] collectionEditor;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`collection-editor' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				collectionEditor = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`collection-editor' must be an array of objects");
					}
					collectionEditor[i] = CSLName
							.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.collectionEditor(collectionEditor);
			}
		}
		{
			Object v = obj.get("composer");
			if (v != null) {
				CSLName[] composer;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`composer' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				composer = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`composer' must be an array of objects");
					}
					composer[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.composer(composer);
			}
		}
		{
			Object v = obj.get("container-author");
			if (v != null) {
				CSLName[] containerAuthor;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`container-author' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				containerAuthor = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`container-author' must be an array of objects");
					}
					containerAuthor[i] = CSLName
							.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.containerAuthor(containerAuthor);
			}
		}
		{
			Object v = obj.get("director");
			if (v != null) {
				CSLName[] director;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`director' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				director = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`director' must be an array of objects");
					}
					director[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.director(director);
			}
		}
		{
			Object v = obj.get("editor");
			if (v != null) {
				CSLName[] editor;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`editor' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				editor = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`editor' must be an array of objects");
					}
					editor[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.editor(editor);
			}
		}
		{
			Object v = obj.get("editorial-director");
			if (v != null) {
				CSLName[] editorialDirector;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`editorial-director' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				editorialDirector = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`editorial-director' must be an array of objects");
					}
					editorialDirector[i] = CSLName
							.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.editorialDirector(editorialDirector);
			}
		}
		{
			Object v = obj.get("interviewer");
			if (v != null) {
				CSLName[] interviewer;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`interviewer' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				interviewer = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`interviewer' must be an array of objects");
					}
					interviewer[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.interviewer(interviewer);
			}
		}
		{
			Object v = obj.get("illustrator");
			if (v != null) {
				CSLName[] illustrator;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`illustrator' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				illustrator = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`illustrator' must be an array of objects");
					}
					illustrator[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.illustrator(illustrator);
			}
		}
		{
			Object v = obj.get("original-author");
			if (v != null) {
				CSLName[] originalAuthor;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`original-author' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				originalAuthor = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`original-author' must be an array of objects");
					}
					originalAuthor[i] = CSLName
							.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.originalAuthor(originalAuthor);
			}
		}
		{
			Object v = obj.get("recipient");
			if (v != null) {
				CSLName[] recipient;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`recipient' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				recipient = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`recipient' must be an array of objects");
					}
					recipient[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.recipient(recipient);
			}
		}
		{
			Object v = obj.get("reviewed-author");
			if (v != null) {
				CSLName[] reviewedAuthor;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`reviewed-author' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				reviewedAuthor = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`reviewed-author' must be an array of objects");
					}
					reviewedAuthor[i] = CSLName
							.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.reviewedAuthor(reviewedAuthor);
			}
		}
		{
			Object v = obj.get("translator");
			if (v != null) {
				CSLName[] translator;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`translator' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				translator = new CSLName[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					if (!(vo instanceof Map)) {
						throw new IllegalArgumentException(
								"`translator' must be an array of objects");
					}
					translator[i] = CSLName.fromJson((Map<String, Object>) vo);
					++i;
				}
				builder.translator(translator);
			}
		}
		{
			Object v = obj.get("accessed");
			if (v != null) {
				CSLDate accessed;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`accessed' must be an object");
				}
				accessed = CSLDate.fromJson((Map<String, Object>) v);
				builder.accessed(accessed);
			}
		}
		{
			Object v = obj.get("container");
			if (v != null) {
				CSLDate container;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`container' must be an object");
				}
				container = CSLDate.fromJson((Map<String, Object>) v);
				builder.container(container);
			}
		}
		{
			Object v = obj.get("event-date");
			if (v != null) {
				CSLDate eventDate;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`event-date' must be an object");
				}
				eventDate = CSLDate.fromJson((Map<String, Object>) v);
				builder.eventDate(eventDate);
			}
		}
		{
			Object v = obj.get("issued");
			if (v != null) {
				CSLDate issued;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`issued' must be an object");
				}
				issued = CSLDate.fromJson((Map<String, Object>) v);
				builder.issued(issued);
			}
		}
		{
			Object v = obj.get("original-date");
			if (v != null) {
				CSLDate originalDate;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`original-date' must be an object");
				}
				originalDate = CSLDate.fromJson((Map<String, Object>) v);
				builder.originalDate(originalDate);
			}
		}
		{
			Object v = obj.get("submitted");
			if (v != null) {
				CSLDate submitted;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`submitted' must be an object");
				}
				submitted = CSLDate.fromJson((Map<String, Object>) v);
				builder.submitted(submitted);
			}
		}
		{
			Object v = obj.get("abstract");
			if (v != null) {
				String abstrct;
				abstrct = v.toString();
				builder.abstrct(abstrct);
			}
		}
		{
			Object v = obj.get("annote");
			if (v != null) {
				String annote;
				annote = v.toString();
				builder.annote(annote);
			}
		}
		{
			Object v = obj.get("archive");
			if (v != null) {
				String archive;
				archive = v.toString();
				builder.archive(archive);
			}
		}
		{
			Object v = obj.get("archive_location");
			if (v != null) {
				String archiveLocation;
				archiveLocation = v.toString();
				builder.archiveLocation(archiveLocation);
			}
		}
		{
			Object v = obj.get("archive-place");
			if (v != null) {
				String archivePlace;
				archivePlace = v.toString();
				builder.archivePlace(archivePlace);
			}
		}
		{
			Object v = obj.get("authority");
			if (v != null) {
				String authority;
				authority = v.toString();
				builder.authority(authority);
			}
		}
		{
			Object v = obj.get("call-number");
			if (v != null) {
				String callNumber;
				callNumber = v.toString();
				builder.callNumber(callNumber);
			}
		}
		{
			Object v = obj.get("chapter-number");
			if (v != null) {
				String chapterNumber;
				chapterNumber = v.toString();
				builder.chapterNumber(chapterNumber);
			}
		}
		{
			Object v = obj.get("citation-number");
			if (v != null) {
				String citationNumber;
				citationNumber = v.toString();
				builder.citationNumber(citationNumber);
			}
		}
		{
			Object v = obj.get("citation-label");
			if (v != null) {
				String citationLabel;
				citationLabel = v.toString();
				builder.citationLabel(citationLabel);
			}
		}
		{
			Object v = obj.get("collection-number");
			if (v != null) {
				String collectionNumber;
				collectionNumber = v.toString();
				builder.collectionNumber(collectionNumber);
			}
		}
		{
			Object v = obj.get("collection-title");
			if (v != null) {
				String collectionTitle;
				collectionTitle = v.toString();
				builder.collectionTitle(collectionTitle);
			}
		}
		{
			Object v = obj.get("container-title");
			if (v != null) {
				String containerTitle;
				containerTitle = v.toString();
				builder.containerTitle(containerTitle);
			}
		}
		{
			Object v = obj.get("container-title-short");
			if (v != null) {
				String containerTitleShort;
				containerTitleShort = v.toString();
				builder.containerTitleShort(containerTitleShort);
			}
		}
		{
			Object v = obj.get("dimensions");
			if (v != null) {
				String dimensions;
				dimensions = v.toString();
				builder.dimensions(dimensions);
			}
		}
		{
			Object v = obj.get("DOI");
			if (v != null) {
				String DOI;
				DOI = v.toString();
				builder.DOI(DOI);
			}
		}
		{
			Object v = obj.get("edition");
			if (v != null) {
				String edition;
				edition = v.toString();
				builder.edition(edition);
			}
		}
		{
			Object v = obj.get("event");
			if (v != null) {
				String event;
				event = v.toString();
				builder.event(event);
			}
		}
		{
			Object v = obj.get("event-place");
			if (v != null) {
				String eventPlace;
				eventPlace = v.toString();
				builder.eventPlace(eventPlace);
			}
		}
		{
			Object v = obj.get("first-reference-note-number");
			if (v != null) {
				String firstReferenceNoteNumber;
				firstReferenceNoteNumber = v.toString();
				builder.firstReferenceNoteNumber(firstReferenceNoteNumber);
			}
		}
		{
			Object v = obj.get("genre");
			if (v != null) {
				String genre;
				genre = v.toString();
				builder.genre(genre);
			}
		}
		{
			Object v = obj.get("ISBN");
			if (v != null) {
				String ISBN;
				ISBN = v.toString();
				builder.ISBN(ISBN);
			}
		}
		{
			Object v = obj.get("ISSN");
			if (v != null) {
				String ISSN;
				ISSN = v.toString();
				builder.ISSN(ISSN);
			}
		}
		{
			Object v = obj.get("issue");
			if (v != null) {
				String issue;
				issue = v.toString();
				builder.issue(issue);
			}
		}
		{
			Object v = obj.get("jurisdiction");
			if (v != null) {
				String jurisdiction;
				jurisdiction = v.toString();
				builder.jurisdiction(jurisdiction);
			}
		}
		{
			Object v = obj.get("keyword");
			if (v != null) {
				String keyword;
				keyword = v.toString();
				builder.keyword(keyword);
			}
		}
		{
			Object v = obj.get("locator");
			if (v != null) {
				String locator;
				locator = v.toString();
				builder.locator(locator);
			}
		}
		{
			Object v = obj.get("medium");
			if (v != null) {
				String medium;
				medium = v.toString();
				builder.medium(medium);
			}
		}
		{
			Object v = obj.get("note");
			if (v != null) {
				String note;
				note = v.toString();
				builder.note(note);
			}
		}
		{
			Object v = obj.get("number");
			if (v != null) {
				String number;
				number = v.toString();
				builder.number(number);
			}
		}
		{
			Object v = obj.get("number-of-pages");
			if (v != null) {
				String numberOfPages;
				numberOfPages = v.toString();
				builder.numberOfPages(numberOfPages);
			}
		}
		{
			Object v = obj.get("number-of-volumes");
			if (v != null) {
				String numberOfVolumes;
				numberOfVolumes = v.toString();
				builder.numberOfVolumes(numberOfVolumes);
			}
		}
		{
			Object v = obj.get("original-publisher");
			if (v != null) {
				String originalPublisher;
				originalPublisher = v.toString();
				builder.originalPublisher(originalPublisher);
			}
		}
		{
			Object v = obj.get("original-publisher-place");
			if (v != null) {
				String originalPublisherPlace;
				originalPublisherPlace = v.toString();
				builder.originalPublisherPlace(originalPublisherPlace);
			}
		}
		{
			Object v = obj.get("original-title");
			if (v != null) {
				String originalTitle;
				originalTitle = v.toString();
				builder.originalTitle(originalTitle);
			}
		}
		{
			Object v = obj.get("page");
			if (v != null) {
				String page;
				page = v.toString();
				builder.page(page);
			}
		}
		{
			Object v = obj.get("page-first");
			if (v != null) {
				String pageFirst;
				pageFirst = v.toString();
				builder.pageFirst(pageFirst);
			}
		}
		{
			Object v = obj.get("PMCID");
			if (v != null) {
				String PMCID;
				PMCID = v.toString();
				builder.PMCID(PMCID);
			}
		}
		{
			Object v = obj.get("PMID");
			if (v != null) {
				String PMID;
				PMID = v.toString();
				builder.PMID(PMID);
			}
		}
		{
			Object v = obj.get("publisher");
			if (v != null) {
				String publisher;
				publisher = v.toString();
				builder.publisher(publisher);
			}
		}
		{
			Object v = obj.get("publisher-place");
			if (v != null) {
				String publisherPlace;
				publisherPlace = v.toString();
				builder.publisherPlace(publisherPlace);
			}
		}
		{
			Object v = obj.get("references");
			if (v != null) {
				String references;
				references = v.toString();
				builder.references(references);
			}
		}
		{
			Object v = obj.get("reviewed-title");
			if (v != null) {
				String reviewedTitle;
				reviewedTitle = v.toString();
				builder.reviewedTitle(reviewedTitle);
			}
		}
		{
			Object v = obj.get("scale");
			if (v != null) {
				String scale;
				scale = v.toString();
				builder.scale(scale);
			}
		}
		{
			Object v = obj.get("section");
			if (v != null) {
				String section;
				section = v.toString();
				builder.section(section);
			}
		}
		{
			Object v = obj.get("source");
			if (v != null) {
				String source;
				source = v.toString();
				builder.source(source);
			}
		}
		{
			Object v = obj.get("status");
			if (v != null) {
				String status;
				status = v.toString();
				builder.status(status);
			}
		}
		{
			Object v = obj.get("title");
			if (v != null) {
				String title;
				title = v.toString();
				builder.title(title);
			}
		}
		{
			Object v = obj.get("title-short");
			if (v != null) {
				String titleShort;
				titleShort = v.toString();
				builder.titleShort(titleShort);
			}
		}
		{
			Object v = obj.get("URL");
			if (v != null) {
				String URL;
				URL = v.toString();
				builder.URL(URL);
			}
		}
		{
			Object v = obj.get("version");
			if (v != null) {
				String version;
				version = v.toString();
				builder.version(version);
			}
		}
		{
			Object v = obj.get("volume");
			if (v != null) {
				String volume;
				volume = v.toString();
				builder.volume(volume);
			}
		}
		{
			Object v = obj.get("year-suffix");
			if (v != null) {
				String yearSuffix;
				yearSuffix = v.toString();
				builder.yearSuffix(yearSuffix);
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((id == null) ? 0 : id.hashCode());
		result = 31 * result + ((type == null) ? 0 : type.hashCode());
		result = 31 * result + Arrays.hashCode(categories);
		result = 31 * result + ((language == null) ? 0 : language.hashCode());
		result = 31
				* result
				+ ((journalAbbreviation == null) ? 0 : journalAbbreviation
						.hashCode());
		result = 31 * result
				+ ((shortTitle == null) ? 0 : shortTitle.hashCode());
		result = 31 * result + Arrays.hashCode(author);
		result = 31 * result + Arrays.hashCode(collectionEditor);
		result = 31 * result + Arrays.hashCode(composer);
		result = 31 * result + Arrays.hashCode(containerAuthor);
		result = 31 * result + Arrays.hashCode(director);
		result = 31 * result + Arrays.hashCode(editor);
		result = 31 * result + Arrays.hashCode(editorialDirector);
		result = 31 * result + Arrays.hashCode(interviewer);
		result = 31 * result + Arrays.hashCode(illustrator);
		result = 31 * result + Arrays.hashCode(originalAuthor);
		result = 31 * result + Arrays.hashCode(recipient);
		result = 31 * result + Arrays.hashCode(reviewedAuthor);
		result = 31 * result + Arrays.hashCode(translator);
		result = 31 * result + ((accessed == null) ? 0 : accessed.hashCode());
		result = 31 * result + ((container == null) ? 0 : container.hashCode());
		result = 31 * result + ((eventDate == null) ? 0 : eventDate.hashCode());
		result = 31 * result + ((issued == null) ? 0 : issued.hashCode());
		result = 31 * result
				+ ((originalDate == null) ? 0 : originalDate.hashCode());
		result = 31 * result + ((submitted == null) ? 0 : submitted.hashCode());
		result = 31 * result + ((abstrct == null) ? 0 : abstrct.hashCode());
		result = 31 * result + ((annote == null) ? 0 : annote.hashCode());
		result = 31 * result + ((archive == null) ? 0 : archive.hashCode());
		result = 31 * result
				+ ((archiveLocation == null) ? 0 : archiveLocation.hashCode());
		result = 31 * result
				+ ((archivePlace == null) ? 0 : archivePlace.hashCode());
		result = 31 * result + ((authority == null) ? 0 : authority.hashCode());
		result = 31 * result
				+ ((callNumber == null) ? 0 : callNumber.hashCode());
		result = 31 * result
				+ ((chapterNumber == null) ? 0 : chapterNumber.hashCode());
		result = 31 * result
				+ ((citationNumber == null) ? 0 : citationNumber.hashCode());
		result = 31 * result
				+ ((citationLabel == null) ? 0 : citationLabel.hashCode());
		result = 31
				* result
				+ ((collectionNumber == null) ? 0 : collectionNumber.hashCode());
		result = 31 * result
				+ ((collectionTitle == null) ? 0 : collectionTitle.hashCode());
		result = 31 * result
				+ ((containerTitle == null) ? 0 : containerTitle.hashCode());
		result = 31
				* result
				+ ((containerTitleShort == null) ? 0 : containerTitleShort
						.hashCode());
		result = 31 * result
				+ ((dimensions == null) ? 0 : dimensions.hashCode());
		result = 31 * result + ((DOI == null) ? 0 : DOI.hashCode());
		result = 31 * result + ((edition == null) ? 0 : edition.hashCode());
		result = 31 * result + ((event == null) ? 0 : event.hashCode());
		result = 31 * result
				+ ((eventPlace == null) ? 0 : eventPlace.hashCode());
		result = 31
				* result
				+ ((firstReferenceNoteNumber == null)
						? 0
						: firstReferenceNoteNumber.hashCode());
		result = 31 * result + ((genre == null) ? 0 : genre.hashCode());
		result = 31 * result + ((ISBN == null) ? 0 : ISBN.hashCode());
		result = 31 * result + ((ISSN == null) ? 0 : ISSN.hashCode());
		result = 31 * result + ((issue == null) ? 0 : issue.hashCode());
		result = 31 * result
				+ ((jurisdiction == null) ? 0 : jurisdiction.hashCode());
		result = 31 * result + ((keyword == null) ? 0 : keyword.hashCode());
		result = 31 * result + ((locator == null) ? 0 : locator.hashCode());
		result = 31 * result + ((medium == null) ? 0 : medium.hashCode());
		result = 31 * result + ((note == null) ? 0 : note.hashCode());
		result = 31 * result + ((number == null) ? 0 : number.hashCode());
		result = 31 * result
				+ ((numberOfPages == null) ? 0 : numberOfPages.hashCode());
		result = 31 * result
				+ ((numberOfVolumes == null) ? 0 : numberOfVolumes.hashCode());
		result = 31
				* result
				+ ((originalPublisher == null) ? 0 : originalPublisher
						.hashCode());
		result = 31
				* result
				+ ((originalPublisherPlace == null)
						? 0
						: originalPublisherPlace.hashCode());
		result = 31 * result
				+ ((originalTitle == null) ? 0 : originalTitle.hashCode());
		result = 31 * result + ((page == null) ? 0 : page.hashCode());
		result = 31 * result + ((pageFirst == null) ? 0 : pageFirst.hashCode());
		result = 31 * result + ((PMCID == null) ? 0 : PMCID.hashCode());
		result = 31 * result + ((PMID == null) ? 0 : PMID.hashCode());
		result = 31 * result + ((publisher == null) ? 0 : publisher.hashCode());
		result = 31 * result
				+ ((publisherPlace == null) ? 0 : publisherPlace.hashCode());
		result = 31 * result
				+ ((references == null) ? 0 : references.hashCode());
		result = 31 * result
				+ ((reviewedTitle == null) ? 0 : reviewedTitle.hashCode());
		result = 31 * result + ((scale == null) ? 0 : scale.hashCode());
		result = 31 * result + ((section == null) ? 0 : section.hashCode());
		result = 31 * result + ((source == null) ? 0 : source.hashCode());
		result = 31 * result + ((status == null) ? 0 : status.hashCode());
		result = 31 * result + ((title == null) ? 0 : title.hashCode());
		result = 31 * result
				+ ((titleShort == null) ? 0 : titleShort.hashCode());
		result = 31 * result + ((URL == null) ? 0 : URL.hashCode());
		result = 31 * result + ((version == null) ? 0 : version.hashCode());
		result = 31 * result + ((volume == null) ? 0 : volume.hashCode());
		result = 31 * result
				+ ((yearSuffix == null) ? 0 : yearSuffix.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLItemData))
			return false;
		CSLItemData other = (CSLItemData) obj;

		if (id == null) {
			if (other.id != null)
				return false;
		} else if (!id.equals(other.id))
			return false;

		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;

		if (!Arrays.equals(categories, other.categories))
			return false;

		if (language == null) {
			if (other.language != null)
				return false;
		} else if (!language.equals(other.language))
			return false;

		if (journalAbbreviation == null) {
			if (other.journalAbbreviation != null)
				return false;
		} else if (!journalAbbreviation.equals(other.journalAbbreviation))
			return false;

		if (shortTitle == null) {
			if (other.shortTitle != null)
				return false;
		} else if (!shortTitle.equals(other.shortTitle))
			return false;

		if (!Arrays.equals(author, other.author))
			return false;

		if (!Arrays.equals(collectionEditor, other.collectionEditor))
			return false;

		if (!Arrays.equals(composer, other.composer))
			return false;

		if (!Arrays.equals(containerAuthor, other.containerAuthor))
			return false;

		if (!Arrays.equals(director, other.director))
			return false;

		if (!Arrays.equals(editor, other.editor))
			return false;

		if (!Arrays.equals(editorialDirector, other.editorialDirector))
			return false;

		if (!Arrays.equals(interviewer, other.interviewer))
			return false;

		if (!Arrays.equals(illustrator, other.illustrator))
			return false;

		if (!Arrays.equals(originalAuthor, other.originalAuthor))
			return false;

		if (!Arrays.equals(recipient, other.recipient))
			return false;

		if (!Arrays.equals(reviewedAuthor, other.reviewedAuthor))
			return false;

		if (!Arrays.equals(translator, other.translator))
			return false;

		if (accessed == null) {
			if (other.accessed != null)
				return false;
		} else if (!accessed.equals(other.accessed))
			return false;

		if (container == null) {
			if (other.container != null)
				return false;
		} else if (!container.equals(other.container))
			return false;

		if (eventDate == null) {
			if (other.eventDate != null)
				return false;
		} else if (!eventDate.equals(other.eventDate))
			return false;

		if (issued == null) {
			if (other.issued != null)
				return false;
		} else if (!issued.equals(other.issued))
			return false;

		if (originalDate == null) {
			if (other.originalDate != null)
				return false;
		} else if (!originalDate.equals(other.originalDate))
			return false;

		if (submitted == null) {
			if (other.submitted != null)
				return false;
		} else if (!submitted.equals(other.submitted))
			return false;

		if (abstrct == null) {
			if (other.abstrct != null)
				return false;
		} else if (!abstrct.equals(other.abstrct))
			return false;

		if (annote == null) {
			if (other.annote != null)
				return false;
		} else if (!annote.equals(other.annote))
			return false;

		if (archive == null) {
			if (other.archive != null)
				return false;
		} else if (!archive.equals(other.archive))
			return false;

		if (archiveLocation == null) {
			if (other.archiveLocation != null)
				return false;
		} else if (!archiveLocation.equals(other.archiveLocation))
			return false;

		if (archivePlace == null) {
			if (other.archivePlace != null)
				return false;
		} else if (!archivePlace.equals(other.archivePlace))
			return false;

		if (authority == null) {
			if (other.authority != null)
				return false;
		} else if (!authority.equals(other.authority))
			return false;

		if (callNumber == null) {
			if (other.callNumber != null)
				return false;
		} else if (!callNumber.equals(other.callNumber))
			return false;

		if (chapterNumber == null) {
			if (other.chapterNumber != null)
				return false;
		} else if (!chapterNumber.equals(other.chapterNumber))
			return false;

		if (citationNumber == null) {
			if (other.citationNumber != null)
				return false;
		} else if (!citationNumber.equals(other.citationNumber))
			return false;

		if (citationLabel == null) {
			if (other.citationLabel != null)
				return false;
		} else if (!citationLabel.equals(other.citationLabel))
			return false;

		if (collectionNumber == null) {
			if (other.collectionNumber != null)
				return false;
		} else if (!collectionNumber.equals(other.collectionNumber))
			return false;

		if (collectionTitle == null) {
			if (other.collectionTitle != null)
				return false;
		} else if (!collectionTitle.equals(other.collectionTitle))
			return false;

		if (containerTitle == null) {
			if (other.containerTitle != null)
				return false;
		} else if (!containerTitle.equals(other.containerTitle))
			return false;

		if (containerTitleShort == null) {
			if (other.containerTitleShort != null)
				return false;
		} else if (!containerTitleShort.equals(other.containerTitleShort))
			return false;

		if (dimensions == null) {
			if (other.dimensions != null)
				return false;
		} else if (!dimensions.equals(other.dimensions))
			return false;

		if (DOI == null) {
			if (other.DOI != null)
				return false;
		} else if (!DOI.equals(other.DOI))
			return false;

		if (edition == null) {
			if (other.edition != null)
				return false;
		} else if (!edition.equals(other.edition))
			return false;

		if (event == null) {
			if (other.event != null)
				return false;
		} else if (!event.equals(other.event))
			return false;

		if (eventPlace == null) {
			if (other.eventPlace != null)
				return false;
		} else if (!eventPlace.equals(other.eventPlace))
			return false;

		if (firstReferenceNoteNumber == null) {
			if (other.firstReferenceNoteNumber != null)
				return false;
		} else if (!firstReferenceNoteNumber
				.equals(other.firstReferenceNoteNumber))
			return false;

		if (genre == null) {
			if (other.genre != null)
				return false;
		} else if (!genre.equals(other.genre))
			return false;

		if (ISBN == null) {
			if (other.ISBN != null)
				return false;
		} else if (!ISBN.equals(other.ISBN))
			return false;

		if (ISSN == null) {
			if (other.ISSN != null)
				return false;
		} else if (!ISSN.equals(other.ISSN))
			return false;

		if (issue == null) {
			if (other.issue != null)
				return false;
		} else if (!issue.equals(other.issue))
			return false;

		if (jurisdiction == null) {
			if (other.jurisdiction != null)
				return false;
		} else if (!jurisdiction.equals(other.jurisdiction))
			return false;

		if (keyword == null) {
			if (other.keyword != null)
				return false;
		} else if (!keyword.equals(other.keyword))
			return false;

		if (locator == null) {
			if (other.locator != null)
				return false;
		} else if (!locator.equals(other.locator))
			return false;

		if (medium == null) {
			if (other.medium != null)
				return false;
		} else if (!medium.equals(other.medium))
			return false;

		if (note == null) {
			if (other.note != null)
				return false;
		} else if (!note.equals(other.note))
			return false;

		if (number == null) {
			if (other.number != null)
				return false;
		} else if (!number.equals(other.number))
			return false;

		if (numberOfPages == null) {
			if (other.numberOfPages != null)
				return false;
		} else if (!numberOfPages.equals(other.numberOfPages))
			return false;

		if (numberOfVolumes == null) {
			if (other.numberOfVolumes != null)
				return false;
		} else if (!numberOfVolumes.equals(other.numberOfVolumes))
			return false;

		if (originalPublisher == null) {
			if (other.originalPublisher != null)
				return false;
		} else if (!originalPublisher.equals(other.originalPublisher))
			return false;

		if (originalPublisherPlace == null) {
			if (other.originalPublisherPlace != null)
				return false;
		} else if (!originalPublisherPlace.equals(other.originalPublisherPlace))
			return false;

		if (originalTitle == null) {
			if (other.originalTitle != null)
				return false;
		} else if (!originalTitle.equals(other.originalTitle))
			return false;

		if (page == null) {
			if (other.page != null)
				return false;
		} else if (!page.equals(other.page))
			return false;

		if (pageFirst == null) {
			if (other.pageFirst != null)
				return false;
		} else if (!pageFirst.equals(other.pageFirst))
			return false;

		if (PMCID == null) {
			if (other.PMCID != null)
				return false;
		} else if (!PMCID.equals(other.PMCID))
			return false;

		if (PMID == null) {
			if (other.PMID != null)
				return false;
		} else if (!PMID.equals(other.PMID))
			return false;

		if (publisher == null) {
			if (other.publisher != null)
				return false;
		} else if (!publisher.equals(other.publisher))
			return false;

		if (publisherPlace == null) {
			if (other.publisherPlace != null)
				return false;
		} else if (!publisherPlace.equals(other.publisherPlace))
			return false;

		if (references == null) {
			if (other.references != null)
				return false;
		} else if (!references.equals(other.references))
			return false;

		if (reviewedTitle == null) {
			if (other.reviewedTitle != null)
				return false;
		} else if (!reviewedTitle.equals(other.reviewedTitle))
			return false;

		if (scale == null) {
			if (other.scale != null)
				return false;
		} else if (!scale.equals(other.scale))
			return false;

		if (section == null) {
			if (other.section != null)
				return false;
		} else if (!section.equals(other.section))
			return false;

		if (source == null) {
			if (other.source != null)
				return false;
		} else if (!source.equals(other.source))
			return false;

		if (status == null) {
			if (other.status != null)
				return false;
		} else if (!status.equals(other.status))
			return false;

		if (title == null) {
			if (other.title != null)
				return false;
		} else if (!title.equals(other.title))
			return false;

		if (titleShort == null) {
			if (other.titleShort != null)
				return false;
		} else if (!titleShort.equals(other.titleShort))
			return false;

		if (URL == null) {
			if (other.URL != null)
				return false;
		} else if (!URL.equals(other.URL))
			return false;

		if (version == null) {
			if (other.version != null)
				return false;
		} else if (!version.equals(other.version))
			return false;

		if (volume == null) {
			if (other.volume != null)
				return false;
		} else if (!volume.equals(other.volume))
			return false;

		if (yearSuffix == null) {
			if (other.yearSuffix != null)
				return false;
		} else if (!yearSuffix.equals(other.yearSuffix))
			return false;

		return true;
	}
}
