// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Map;

/**
 * Builder for {@link CSLCitationItem}
 * 
 * @author Michel Kraemer
 */
public class CSLCitationItemBuilder {
	private String id;

	private CSLItemData itemData;
	private String prefix;
	private String suffix;
	private String locator;
	private Integer position;
	private Boolean nearNote;
	private Integer noteNumber;
	private Integer firstReferenceNoteNumber;
	private CSLLabel label;
	private Boolean suppressAuthor;
	private Boolean authorOnly;
	private String[] uris;

	public CSLCitationItemBuilder(String id) {
		this.id = id;

		this.itemData = null;
		this.prefix = null;
		this.suffix = null;
		this.locator = null;
		this.position = null;
		this.nearNote = null;
		this.noteNumber = null;
		this.firstReferenceNoteNumber = null;
		this.label = null;
		this.suppressAuthor = null;
		this.authorOnly = null;
		this.uris = null;

	}

	public CSLCitationItemBuilder itemData(CSLItemData itemData) {
		this.itemData = itemData;
		return this;
	}

	public CSLCitationItemBuilder prefix(String prefix) {
		this.prefix = prefix;
		return this;
	}

	public CSLCitationItemBuilder suffix(String suffix) {
		this.suffix = suffix;
		return this;
	}

	public CSLCitationItemBuilder locator(String locator) {
		this.locator = locator;
		return this;
	}

	public CSLCitationItemBuilder position(Integer position) {
		this.position = position;
		return this;
	}

	public CSLCitationItemBuilder nearNote(Boolean nearNote) {
		this.nearNote = nearNote;
		return this;
	}

	public CSLCitationItemBuilder noteNumber(Integer noteNumber) {
		this.noteNumber = noteNumber;
		return this;
	}

	public CSLCitationItemBuilder firstReferenceNoteNumber(
			Integer firstReferenceNoteNumber) {
		this.firstReferenceNoteNumber = firstReferenceNoteNumber;
		return this;
	}

	public CSLCitationItemBuilder label(CSLLabel label) {
		this.label = label;
		return this;
	}

	public CSLCitationItemBuilder suppressAuthor(Boolean suppressAuthor) {
		this.suppressAuthor = suppressAuthor;
		return this;
	}

	public CSLCitationItemBuilder authorOnly(Boolean authorOnly) {
		this.authorOnly = authorOnly;
		return this;
	}

	public CSLCitationItemBuilder uris(String... uris) {
		this.uris = uris;
		return this;
	}

	/**
	 * Creates a builder that copies properties from the given original object
	 * 
	 * @param original
	 *            the original object
	 */
	public CSLCitationItemBuilder(CSLCitationItem original) {
		this.id = original.getId();
		this.itemData = original.getItemData();
		this.prefix = original.getPrefix();
		this.suffix = original.getSuffix();
		this.locator = original.getLocator();
		this.position = original.getPosition();
		this.nearNote = original.getNearNote();
		this.noteNumber = original.getNoteNumber();
		this.firstReferenceNoteNumber = original.getFirstReferenceNoteNumber();
		this.label = original.getLabel();
		this.suppressAuthor = original.getSuppressAuthor();
		this.authorOnly = original.getAuthorOnly();
		this.uris = original.getUris();
	}

	public CSLCitationItem build() {
		return new CSLCitationItem(id, itemData, prefix, suffix, locator,
				position, nearNote, noteNumber, firstReferenceNoteNumber,
				label, suppressAuthor, authorOnly, uris);
	}

}
