// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A citation item is used to register a citation in the CSL processor. It
 * usually only consists of the citation's ID but can also contain other
 * formatting parameters.
 * 
 * @author Michel Kraemer
 */
public class CSLCitationItem implements JsonObject {
	private final String id;

	private final CSLItemData itemData;
	private final String prefix;
	private final String suffix;
	private final String locator;
	private final Integer position;
	private final Boolean nearNote;
	private final Integer noteNumber;
	private final Integer firstReferenceNoteNumber;
	private final CSLLabel label;
	private final Boolean suppressAuthor;
	private final Boolean authorOnly;
	private final String[] uris;

	public CSLCitationItem(String id) {
		this.id = id;

		this.itemData = null;
		this.prefix = null;
		this.suffix = null;
		this.locator = null;
		this.position = null;
		this.nearNote = null;
		this.noteNumber = null;
		this.firstReferenceNoteNumber = null;
		this.label = null;
		this.suppressAuthor = null;
		this.authorOnly = null;
		this.uris = null;

	}

	public CSLCitationItem(String id, CSLItemData itemData, String prefix,
			String suffix, String locator, Integer position, Boolean nearNote,
			Integer noteNumber, Integer firstReferenceNoteNumber,
			CSLLabel label, Boolean suppressAuthor, Boolean authorOnly,
			String[] uris) {
		this.id = id;

		this.itemData = itemData;
		this.prefix = prefix;
		this.suffix = suffix;
		this.locator = locator;
		this.position = position;
		this.nearNote = nearNote;
		this.noteNumber = noteNumber;
		this.firstReferenceNoteNumber = firstReferenceNoteNumber;
		this.label = label;
		this.suppressAuthor = suppressAuthor;
		this.authorOnly = authorOnly;
		this.uris = uris;

	}

	/**
	 * @return the citation item's id
	 */
	public String getId() {
		return id;
	}

	/**
	 * @return the citation item's itemData
	 */
	public CSLItemData getItemData() {
		return itemData;
	}
	/**
	 * @return the citation item's prefix
	 */
	public String getPrefix() {
		return prefix;
	}
	/**
	 * @return the citation item's suffix
	 */
	public String getSuffix() {
		return suffix;
	}
	/**
	 * @return the citation item's locator
	 */
	public String getLocator() {
		return locator;
	}
	/**
	 * @return the citation item's position
	 */
	public Integer getPosition() {
		return position;
	}
	/**
	 * @return the citation item's near-note
	 */
	public Boolean getNearNote() {
		return nearNote;
	}
	/**
	 * @return the citation item's note-number
	 */
	public Integer getNoteNumber() {
		return noteNumber;
	}
	/**
	 * @return the citation item's first-reference-note-number
	 */
	public Integer getFirstReferenceNoteNumber() {
		return firstReferenceNoteNumber;
	}
	/**
	 * @return the citation item's label
	 */
	public CSLLabel getLabel() {
		return label;
	}
	/**
	 * @return the citation item's suppress-author
	 */
	public Boolean getSuppressAuthor() {
		return suppressAuthor;
	}
	/**
	 * @return the citation item's author-only
	 */
	public Boolean getAuthorOnly() {
		return authorOnly;
	}
	/**
	 * @return the citation item's uris
	 */
	public String[] getUris() {
		return uris;
	}

	@Override
	public Object toJson(JsonBuilder builder) {
		builder.add("id", id);
		if (itemData != null) {
			builder.add("itemData", itemData);
		}
		if (prefix != null) {
			builder.add("prefix", prefix);
		}
		if (suffix != null) {
			builder.add("suffix", suffix);
		}
		if (locator != null) {
			builder.add("locator", locator);
		}
		if (position != null) {
			builder.add("position", position);
		}
		if (nearNote != null) {
			builder.add("near-note", nearNote);
		}
		if (noteNumber != null) {
			builder.add("note-number", noteNumber);
		}
		if (firstReferenceNoteNumber != null) {
			builder.add("first-reference-note-number", firstReferenceNoteNumber);
		}
		if (label != null) {
			builder.add("label", label);
		}
		if (suppressAuthor != null) {
			builder.add("suppress-author", suppressAuthor);
		}
		if (authorOnly != null) {
			builder.add("author-only", authorOnly);
		}
		if (uris != null) {
			builder.add("uris", uris);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLCitationItem object. The JSON object must
	 * at least contain the following required properties: <code>id</code>
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLCitationItem object
	 */
	@SuppressWarnings("unchecked")
	public static CSLCitationItem fromJson(Map<String, Object> obj) {
		String id;

		{
			Object v = obj.get("id");
			if (v == null) {
				throw new IllegalArgumentException("Missing property `id'");
			}
			id = v.toString();
		}

		CSLCitationItemBuilder builder = new CSLCitationItemBuilder(id);

		{
			Object v = obj.get("itemData");
			if (v != null) {
				CSLItemData itemData;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`itemData' must be an object");
				}
				itemData = CSLItemData.fromJson((Map<String, Object>) v);
				builder.itemData(itemData);
			}
		}
		{
			Object v = obj.get("prefix");
			if (v != null) {
				String prefix;
				prefix = v.toString();
				builder.prefix(prefix);
			}
		}
		{
			Object v = obj.get("suffix");
			if (v != null) {
				String suffix;
				suffix = v.toString();
				builder.suffix(suffix);
			}
		}
		{
			Object v = obj.get("locator");
			if (v != null) {
				String locator;
				locator = v.toString();
				builder.locator(locator);
			}
		}
		{
			Object v = obj.get("position");
			if (v != null) {
				Integer position;
				position = toInt(v);
				builder.position(position);
			}
		}
		{
			Object v = obj.get("near-note");
			if (v != null) {
				Boolean nearNote;
				nearNote = toBool(v);
				builder.nearNote(nearNote);
			}
		}
		{
			Object v = obj.get("note-number");
			if (v != null) {
				Integer noteNumber;
				noteNumber = toInt(v);
				builder.noteNumber(noteNumber);
			}
		}
		{
			Object v = obj.get("first-reference-note-number");
			if (v != null) {
				Integer firstReferenceNoteNumber;
				firstReferenceNoteNumber = toInt(v);
				builder.firstReferenceNoteNumber(firstReferenceNoteNumber);
			}
		}
		{
			Object v = obj.get("label");
			if (!isFalsy(v)) {
				CSLLabel label;
				label = CSLLabel.fromString(v.toString());
				builder.label(label);
			}
		}
		{
			Object v = obj.get("suppress-author");
			if (v != null) {
				Boolean suppressAuthor;
				suppressAuthor = toBool(v);
				builder.suppressAuthor(suppressAuthor);
			}
		}
		{
			Object v = obj.get("author-only");
			if (v != null) {
				Boolean authorOnly;
				authorOnly = toBool(v);
				builder.authorOnly(authorOnly);
			}
		}
		{
			Object v = obj.get("uris");
			if (v != null) {
				String[] uris;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`uris' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				uris = new String[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					uris[i] = vo.toString();
					++i;
				}
				builder.uris(uris);
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((itemData == null) ? 0 : itemData.hashCode());
		result = 31 * result + ((prefix == null) ? 0 : prefix.hashCode());
		result = 31 * result + ((suffix == null) ? 0 : suffix.hashCode());
		result = 31 * result + ((locator == null) ? 0 : locator.hashCode());
		result = 31 * result + ((position == null) ? 0 : position.hashCode());
		result = 31 * result + ((nearNote == null) ? 0 : nearNote.hashCode());
		result = 31 * result
				+ ((noteNumber == null) ? 0 : noteNumber.hashCode());
		result = 31
				* result
				+ ((firstReferenceNoteNumber == null)
						? 0
						: firstReferenceNoteNumber.hashCode());
		result = 31 * result + ((label == null) ? 0 : label.hashCode());
		result = 31 * result
				+ ((suppressAuthor == null) ? 0 : suppressAuthor.hashCode());
		result = 31 * result
				+ ((authorOnly == null) ? 0 : authorOnly.hashCode());
		result = 31 * result + Arrays.hashCode(uris);

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLCitationItem))
			return false;
		CSLCitationItem other = (CSLCitationItem) obj;

		if (itemData == null) {
			if (other.itemData != null)
				return false;
		} else if (!itemData.equals(other.itemData))
			return false;

		if (prefix == null) {
			if (other.prefix != null)
				return false;
		} else if (!prefix.equals(other.prefix))
			return false;

		if (suffix == null) {
			if (other.suffix != null)
				return false;
		} else if (!suffix.equals(other.suffix))
			return false;

		if (locator == null) {
			if (other.locator != null)
				return false;
		} else if (!locator.equals(other.locator))
			return false;

		if (position == null) {
			if (other.position != null)
				return false;
		} else if (!position.equals(other.position))
			return false;

		if (nearNote == null) {
			if (other.nearNote != null)
				return false;
		} else if (!nearNote.equals(other.nearNote))
			return false;

		if (noteNumber == null) {
			if (other.noteNumber != null)
				return false;
		} else if (!noteNumber.equals(other.noteNumber))
			return false;

		if (firstReferenceNoteNumber == null) {
			if (other.firstReferenceNoteNumber != null)
				return false;
		} else if (!firstReferenceNoteNumber
				.equals(other.firstReferenceNoteNumber))
			return false;

		if (label == null) {
			if (other.label != null)
				return false;
		} else if (!label.equals(other.label))
			return false;

		if (suppressAuthor == null) {
			if (other.suppressAuthor != null)
				return false;
		} else if (!suppressAuthor.equals(other.suppressAuthor))
			return false;

		if (authorOnly == null) {
			if (other.authorOnly != null)
				return false;
		} else if (!authorOnly.equals(other.authorOnly))
			return false;

		if (!Arrays.equals(uris, other.uris))
			return false;

		return true;
	}
}
