// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

/**
 * A list of abbreviations for titles, authorities, institution names, etc.
 * 
 * @author Michel Kraemer
 */
public class CSLAbbreviationList implements JsonObject {

	private final Map<String, String> title;
	private final Map<String, String> collectionTitle;
	private final Map<String, String> containerTitle;
	private final Map<String, String> number;
	private final Map<String, String> place;
	private final Map<String, String> nickname;
	private final Map<String, String> classic;
	private final Map<String, String> institution;
	private final Map<String, String> hereinafter;

	public CSLAbbreviationList() {

		this.title = null;
		this.collectionTitle = null;
		this.containerTitle = null;
		this.number = null;
		this.place = null;
		this.nickname = null;
		this.classic = null;
		this.institution = null;
		this.hereinafter = null;

	}

	public CSLAbbreviationList(Map<String, String> title,
			Map<String, String> collectionTitle,
			Map<String, String> containerTitle, Map<String, String> number,
			Map<String, String> place, Map<String, String> nickname,
			Map<String, String> classic, Map<String, String> institution,
			Map<String, String> hereinafter) {

		this.title = title;
		this.collectionTitle = collectionTitle;
		this.containerTitle = containerTitle;
		this.number = number;
		this.place = place;
		this.nickname = nickname;
		this.classic = classic;
		this.institution = institution;
		this.hereinafter = hereinafter;

	}

	/**
	 * @return the abbreviation list's title
	 */
	public Map<String, String> getTitle() {
		return title;
	}
	/**
	 * @return the abbreviation list's collection-title
	 */
	public Map<String, String> getCollectionTitle() {
		return collectionTitle;
	}
	/**
	 * @return the abbreviation list's container-title
	 */
	public Map<String, String> getContainerTitle() {
		return containerTitle;
	}
	/**
	 * @return the abbreviation list's number
	 */
	public Map<String, String> getNumber() {
		return number;
	}
	/**
	 * @return the abbreviation list's place
	 */
	public Map<String, String> getPlace() {
		return place;
	}
	/**
	 * @return the abbreviation list's nickname
	 */
	public Map<String, String> getNickname() {
		return nickname;
	}
	/**
	 * @return the abbreviation list's classic
	 */
	public Map<String, String> getClassic() {
		return classic;
	}
	/**
	 * @return the abbreviation list's institution
	 */
	public Map<String, String> getInstitution() {
		return institution;
	}
	/**
	 * @return the abbreviation list's hereinafter
	 */
	public Map<String, String> getHereinafter() {
		return hereinafter;
	}

	@Override
	public Object toJson(JsonBuilder builder) {

		if (title != null) {
			builder.add("title", title);
		}
		if (collectionTitle != null) {
			builder.add("collection-title", collectionTitle);
		}
		if (containerTitle != null) {
			builder.add("container-title", containerTitle);
		}
		if (number != null) {
			builder.add("number", number);
		}
		if (place != null) {
			builder.add("place", place);
		}
		if (nickname != null) {
			builder.add("nickname", nickname);
		}
		if (classic != null) {
			builder.add("classic", classic);
		}
		if (institution != null) {
			builder.add("institution", institution);
		}
		if (hereinafter != null) {
			builder.add("hereinafter", hereinafter);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a CSLAbbreviationList object.
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted CSLAbbreviationList object
	 */
	@SuppressWarnings("unchecked")
	public static CSLAbbreviationList fromJson(Map<String, Object> obj) {

		CSLAbbreviationListBuilder builder = new CSLAbbreviationListBuilder();

		{
			Object v = obj.get("title");
			if (v != null) {
				Map<String, String> title;
				title = (Map<String, String>) v;
				builder.title(title);
			}
		}
		{
			Object v = obj.get("collection-title");
			if (v != null) {
				Map<String, String> collectionTitle;
				collectionTitle = (Map<String, String>) v;
				builder.collectionTitle(collectionTitle);
			}
		}
		{
			Object v = obj.get("container-title");
			if (v != null) {
				Map<String, String> containerTitle;
				containerTitle = (Map<String, String>) v;
				builder.containerTitle(containerTitle);
			}
		}
		{
			Object v = obj.get("number");
			if (v != null) {
				Map<String, String> number;
				number = (Map<String, String>) v;
				builder.number(number);
			}
		}
		{
			Object v = obj.get("place");
			if (v != null) {
				Map<String, String> place;
				place = (Map<String, String>) v;
				builder.place(place);
			}
		}
		{
			Object v = obj.get("nickname");
			if (v != null) {
				Map<String, String> nickname;
				nickname = (Map<String, String>) v;
				builder.nickname(nickname);
			}
		}
		{
			Object v = obj.get("classic");
			if (v != null) {
				Map<String, String> classic;
				classic = (Map<String, String>) v;
				builder.classic(classic);
			}
		}
		{
			Object v = obj.get("institution");
			if (v != null) {
				Map<String, String> institution;
				institution = (Map<String, String>) v;
				builder.institution(institution);
			}
		}
		{
			Object v = obj.get("hereinafter");
			if (v != null) {
				Map<String, String> hereinafter;
				hereinafter = (Map<String, String>) v;
				builder.hereinafter(hereinafter);
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((title == null) ? 0 : title.hashCode());
		result = 31 * result
				+ ((collectionTitle == null) ? 0 : collectionTitle.hashCode());
		result = 31 * result
				+ ((containerTitle == null) ? 0 : containerTitle.hashCode());
		result = 31 * result + ((number == null) ? 0 : number.hashCode());
		result = 31 * result + ((place == null) ? 0 : place.hashCode());
		result = 31 * result + ((nickname == null) ? 0 : nickname.hashCode());
		result = 31 * result + ((classic == null) ? 0 : classic.hashCode());
		result = 31 * result
				+ ((institution == null) ? 0 : institution.hashCode());
		result = 31 * result
				+ ((hereinafter == null) ? 0 : hereinafter.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof CSLAbbreviationList))
			return false;
		CSLAbbreviationList other = (CSLAbbreviationList) obj;

		if (title == null) {
			if (other.title != null)
				return false;
		} else if (!title.equals(other.title))
			return false;

		if (collectionTitle == null) {
			if (other.collectionTitle != null)
				return false;
		} else if (!collectionTitle.equals(other.collectionTitle))
			return false;

		if (containerTitle == null) {
			if (other.containerTitle != null)
				return false;
		} else if (!containerTitle.equals(other.containerTitle))
			return false;

		if (number == null) {
			if (other.number != null)
				return false;
		} else if (!number.equals(other.number))
			return false;

		if (place == null) {
			if (other.place != null)
				return false;
		} else if (!place.equals(other.place))
			return false;

		if (nickname == null) {
			if (other.nickname != null)
				return false;
		} else if (!nickname.equals(other.nickname))
			return false;

		if (classic == null) {
			if (other.classic != null)
				return false;
		} else if (!classic.equals(other.classic))
			return false;

		if (institution == null) {
			if (other.institution != null)
				return false;
		} else if (!institution.equals(other.institution))
			return false;

		if (hereinafter == null) {
			if (other.hereinafter != null)
				return false;
		} else if (!hereinafter.equals(other.hereinafter))
			return false;

		return true;
	}
}
