// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc;

import java.util.Arrays;
import java.util.Map;

import java.util.Collection;

import de.undercouch.citeproc.helper.json.JsonBuilder;
import de.undercouch.citeproc.helper.json.JsonObject;

import de.undercouch.citeproc.csl.CSLItemData;

/**
 * Parameters for a {@link de.undercouch.citeproc.VariableWrapper}
 * 
 * @author Michel Kraemer
 */
public class VariableWrapperParams implements JsonObject {

	private final CSLItemData itemData;
	private final String[] variableNames;
	private final Context context;
	private final String xclass;
	private final String position;
	private final Integer noteNumber;
	private final Integer firstReferenceNoteNumber;
	private final Integer citationNumber;
	private final Integer index;
	private final String mode;

	public VariableWrapperParams() {

		this.itemData = null;
		this.variableNames = null;
		this.context = null;
		this.xclass = null;
		this.position = null;
		this.noteNumber = null;
		this.firstReferenceNoteNumber = null;
		this.citationNumber = null;
		this.index = null;
		this.mode = null;

	}

	public VariableWrapperParams(CSLItemData itemData, String[] variableNames,
			Context context, String xclass, String position,
			Integer noteNumber, Integer firstReferenceNoteNumber,
			Integer citationNumber, Integer index, String mode) {

		this.itemData = itemData;
		this.variableNames = variableNames;
		this.context = context;
		this.xclass = xclass;
		this.position = position;
		this.noteNumber = noteNumber;
		this.firstReferenceNoteNumber = firstReferenceNoteNumber;
		this.citationNumber = citationNumber;
		this.index = index;
		this.mode = mode;

	}

	/**
	 * @return the itemData
	 */
	public CSLItemData getItemData() {
		return itemData;
	}
	/**
	 * @return the variableNames
	 */
	public String[] getVariableNames() {
		return variableNames;
	}
	/**
	 * @return the context
	 */
	public Context getContext() {
		return context;
	}
	/**
	 * @return the xclass
	 */
	public String getXclass() {
		return xclass;
	}
	/**
	 * @return the position
	 */
	public String getPosition() {
		return position;
	}
	/**
	 * @return the note-number
	 */
	public Integer getNoteNumber() {
		return noteNumber;
	}
	/**
	 * @return the first-reference-note-number
	 */
	public Integer getFirstReferenceNoteNumber() {
		return firstReferenceNoteNumber;
	}
	/**
	 * @return the citation-number
	 */
	public Integer getCitationNumber() {
		return citationNumber;
	}
	/**
	 * @return the index
	 */
	public Integer getIndex() {
		return index;
	}
	/**
	 * @return the mode
	 */
	public String getMode() {
		return mode;
	}

	@Override
	public Object toJson(JsonBuilder builder) {

		if (itemData != null) {
			builder.add("itemData", itemData);
		}
		if (variableNames != null) {
			builder.add("variableNames", variableNames);
		}
		if (context != null) {
			builder.add("context", context);
		}
		if (xclass != null) {
			builder.add("xclass", xclass);
		}
		if (position != null) {
			builder.add("position", position);
		}
		if (noteNumber != null) {
			builder.add("note-number", noteNumber);
		}
		if (firstReferenceNoteNumber != null) {
			builder.add("first-reference-note-number", firstReferenceNoteNumber);
		}
		if (citationNumber != null) {
			builder.add("citation-number", citationNumber);
		}
		if (index != null) {
			builder.add("index", index);
		}
		if (mode != null) {
			builder.add("mode", mode);
		}

		return builder.build();
	}

	/**
	 * Converts a JSON object to a VariableWrapperParams object.
	 * 
	 * @param obj
	 *            the JSON object to convert
	 * @return the converted VariableWrapperParams object
	 */
	@SuppressWarnings("unchecked")
	public static VariableWrapperParams fromJson(Map<String, Object> obj) {

		VariableWrapperParamsBuilder builder = new VariableWrapperParamsBuilder();

		{
			Object v = obj.get("itemData");
			if (v != null) {
				CSLItemData itemData;
				if (!(v instanceof Map)) {
					throw new IllegalArgumentException(
							"`itemData' must be an object");
				}
				itemData = CSLItemData.fromJson((Map<String, Object>) v);
				builder.itemData(itemData);
			}
		}
		{
			Object v = obj.get("variableNames");
			if (v != null) {
				String[] variableNames;
				if (v instanceof Map) {
					v = ((Map<?, ?>) v).values();
				} else if (!(v instanceof Collection)) {
					throw new IllegalArgumentException(
							"`variableNames' must be an array");
				}
				Collection<?> cv = (Collection<?>) v;
				variableNames = new String[cv.size()];
				int i = 0;
				for (Object vo : cv) {
					variableNames[i] = vo.toString();
					++i;
				}
				builder.variableNames(variableNames);
			}
		}
		{
			Object v = obj.get("context");
			if (!isFalsy(v)) {
				Context context;
				context = Context.fromString(v.toString());
				builder.context(context);
			}
		}
		{
			Object v = obj.get("xclass");
			if (v != null) {
				String xclass;
				xclass = v.toString();
				builder.xclass(xclass);
			}
		}
		{
			Object v = obj.get("position");
			if (v != null) {
				String position;
				position = v.toString();
				builder.position(position);
			}
		}
		{
			Object v = obj.get("note-number");
			if (v != null) {
				Integer noteNumber;
				noteNumber = toInt(v);
				builder.noteNumber(noteNumber);
			}
		}
		{
			Object v = obj.get("first-reference-note-number");
			if (v != null) {
				Integer firstReferenceNoteNumber;
				firstReferenceNoteNumber = toInt(v);
				builder.firstReferenceNoteNumber(firstReferenceNoteNumber);
			}
		}
		{
			Object v = obj.get("citation-number");
			if (v != null) {
				Integer citationNumber;
				citationNumber = toInt(v);
				builder.citationNumber(citationNumber);
			}
		}
		{
			Object v = obj.get("index");
			if (v != null) {
				Integer index;
				index = toInt(v);
				builder.index(index);
			}
		}
		{
			Object v = obj.get("mode");
			if (v != null) {
				String mode;
				mode = v.toString();
				builder.mode(mode);
			}
		}

		return builder.build();
	}

	private static boolean isFalsy(Object o) {
		if (o == null) {
			return true;
		}
		if (Boolean.FALSE.equals(o)) {
			return true;
		}
		if ("".equals(o)) {
			return true;
		}
		if (Integer.valueOf(0).equals(o)) {
			return true;
		}
		if (Long.valueOf(0L).equals(o)) {
			return true;
		}
		if (o instanceof Float
				&& (Float.valueOf(0f).equals(o) || ((Float) o).isNaN())) {
			return true;
		}
		if (o instanceof Double
				&& (Double.valueOf(0d).equals(o) || ((Double) o).isNaN())) {
			return true;
		}
		if (Byte.valueOf((byte) 0).equals(o)) {
			return true;
		}
		if (Short.valueOf((short) 0).equals(o)) {
			return true;
		}
		return false;
	}

	private static int toInt(Object o) {
		if (o instanceof CharSequence) {
			return Integer.parseInt(o.toString());
		}
		return ((Number) o).intValue();
	}

	private static boolean toBool(Object o) {
		if (o instanceof String) {
			return Boolean.parseBoolean((String) o);
		} else if (o instanceof Number) {
			return ((Number) o).intValue() != 0;
		}
		return (Boolean) o;
	}

	@Override
	public int hashCode() {
		int result = 1;

		result = 31 * result + ((itemData == null) ? 0 : itemData.hashCode());
		result = 31 * result + Arrays.hashCode(variableNames);
		result = 31 * result + ((context == null) ? 0 : context.hashCode());
		result = 31 * result + ((xclass == null) ? 0 : xclass.hashCode());
		result = 31 * result + ((position == null) ? 0 : position.hashCode());
		result = 31 * result
				+ ((noteNumber == null) ? 0 : noteNumber.hashCode());
		result = 31
				* result
				+ ((firstReferenceNoteNumber == null)
						? 0
						: firstReferenceNoteNumber.hashCode());
		result = 31 * result
				+ ((citationNumber == null) ? 0 : citationNumber.hashCode());
		result = 31 * result + ((index == null) ? 0 : index.hashCode());
		result = 31 * result + ((mode == null) ? 0 : mode.hashCode());

		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (!(obj instanceof VariableWrapperParams))
			return false;
		VariableWrapperParams other = (VariableWrapperParams) obj;

		if (itemData == null) {
			if (other.itemData != null)
				return false;
		} else if (!itemData.equals(other.itemData))
			return false;

		if (!Arrays.equals(variableNames, other.variableNames))
			return false;

		if (context == null) {
			if (other.context != null)
				return false;
		} else if (!context.equals(other.context))
			return false;

		if (xclass == null) {
			if (other.xclass != null)
				return false;
		} else if (!xclass.equals(other.xclass))
			return false;

		if (position == null) {
			if (other.position != null)
				return false;
		} else if (!position.equals(other.position))
			return false;

		if (noteNumber == null) {
			if (other.noteNumber != null)
				return false;
		} else if (!noteNumber.equals(other.noteNumber))
			return false;

		if (firstReferenceNoteNumber == null) {
			if (other.firstReferenceNoteNumber != null)
				return false;
		} else if (!firstReferenceNoteNumber
				.equals(other.firstReferenceNoteNumber))
			return false;

		if (citationNumber == null) {
			if (other.citationNumber != null)
				return false;
		} else if (!citationNumber.equals(other.citationNumber))
			return false;

		if (index == null) {
			if (other.index != null)
				return false;
		} else if (!index.equals(other.index))
			return false;

		if (mode == null) {
			if (other.mode != null)
				return false;
		} else if (!mode.equals(other.mode))
			return false;

		return true;
	}
}
