// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import de.undercouch.citeproc.helper.JsonHelper;
import de.undercouch.citeproc.helper.JsonObject;

/**
 * A citation item is used to register a citation in the CSL processor. It
 * usually only consists of the citation's ID but can also contain other
 * formatting parameters.
 * 
 * @author Michel Kraemer
 */
public class CSLCitationItem implements JsonObject {
	private final String id;

	private final CSLItemData itemData;
	private final String prefix;
	private final String suffix;
	private final String locator;
	private final CSLLabel label;
	private final Boolean suppressAuthor;
	private final Boolean authorOnly;
	private final String[] uris;

	public CSLCitationItem(String id) {
		this.id = id;

		this.itemData = null;
		this.prefix = null;
		this.suffix = null;
		this.locator = null;
		this.label = null;
		this.suppressAuthor = null;
		this.authorOnly = null;
		this.uris = null;

	}

	public CSLCitationItem(String id, CSLItemData itemData, String prefix,
			String suffix, String locator, CSLLabel label,
			Boolean suppressAuthor, Boolean authorOnly, String[] uris) {
		this.id = id;

		this.itemData = itemData;
		this.prefix = prefix;
		this.suffix = suffix;
		this.locator = locator;
		this.label = label;
		this.suppressAuthor = suppressAuthor;
		this.authorOnly = authorOnly;
		this.uris = uris;

	}

	/**
	 * @return the citation item's id
	 */
	public String getId() {
		return id;
	}

	/**
	 * @return the citation item's itemData
	 */
	public CSLItemData getItemData() {
		return itemData;
	}
	/**
	 * @return the citation item's prefix
	 */
	public String getPrefix() {
		return prefix;
	}
	/**
	 * @return the citation item's suffix
	 */
	public String getSuffix() {
		return suffix;
	}
	/**
	 * @return the citation item's locator
	 */
	public String getLocator() {
		return locator;
	}
	/**
	 * @return the citation item's label
	 */
	public CSLLabel getLabel() {
		return label;
	}
	/**
	 * @return the citation item's suppress-author
	 */
	public Boolean getSuppressAuthor() {
		return suppressAuthor;
	}
	/**
	 * @return the citation item's author-only
	 */
	public Boolean getAuthorOnly() {
		return authorOnly;
	}
	/**
	 * @return the citation item's uris
	 */
	public String[] getUris() {
		return uris;
	}

	@Override
	public String toJson() {
		StringBuilder r = new StringBuilder("{");
		r.append("\"id\": " + JsonHelper.toJson(id));

		if (itemData != null) {
			r.append(",\"itemData\": " + JsonHelper.toJson(itemData));
		}
		if (prefix != null) {
			r.append(",\"prefix\": " + JsonHelper.toJson(prefix));
		}
		if (suffix != null) {
			r.append(",\"suffix\": " + JsonHelper.toJson(suffix));
		}
		if (locator != null) {
			r.append(",\"locator\": " + JsonHelper.toJson(locator));
		}
		if (label != null) {
			r.append(",\"label\": " + JsonHelper.toJson(label));
		}
		if (suppressAuthor != null) {
			r.append(",\"suppress-author\": "
					+ JsonHelper.toJson(suppressAuthor));
		}
		if (authorOnly != null) {
			r.append(",\"author-only\": " + JsonHelper.toJson(authorOnly));
		}
		if (uris != null) {
			r.append(",\"uris\": " + JsonHelper.toJson(uris));
		}

		r.append("}");
		return r.toString();
	}

	@Override
	public String toString() {
		return toJson();
	}

}
