// Copyright 2013 Michel Kraemer
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package de.undercouch.citeproc.csl;

import de.undercouch.citeproc.helper.JsonHelper;
import de.undercouch.citeproc.helper.JsonObject;

/**
 * A set of citation items. This class is used to register citations in the CSL
 * processor.
 * 
 * @author Michel Kraemer
 */
public class CSLCitation implements JsonObject {
	private final CSLCitationItem[] citationItems;

	private final String citationID;
	private final CSLProperties properties;

	public CSLCitation(CSLCitationItem... citationItems) {
		this.citationItems = citationItems;

		this.citationID = Long.toString(
				(long) Math.floor(Math.random() * 100000000000000L), 32);
		this.properties = new CSLProperties();

	}

	public CSLCitation(CSLCitationItem[] citationItems, String citationID,
			CSLProperties properties) {
		this.citationItems = citationItems;

		this.citationID = citationID;
		this.properties = properties;

	}

	/**
	 * @return the citation's citationItems
	 */
	public CSLCitationItem[] getCitationItems() {
		return citationItems;
	}

	/**
	 * @return the citation's citationID
	 */
	public String getCitationID() {
		return citationID;
	}
	/**
	 * @return the citation's properties
	 */
	public CSLProperties getProperties() {
		return properties;
	}

	@Override
	public String toJson() {
		StringBuilder r = new StringBuilder("{");
		r.append("\"citationItems\": " + JsonHelper.toJson(citationItems));

		if (citationID != null) {
			r.append(",\"citationID\": " + JsonHelper.toJson(citationID));
		}
		if (properties != null) {
			r.append(",\"properties\": " + JsonHelper.toJson(properties));
		}

		r.append("}");
		return r.toString();
	}

	@Override
	public String toString() {
		return toJson();
	}

}
