/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.responses;

import org.codehaus.jackson.annotate.JsonIgnore;

import de.twenty11.skysail.common.PresentationStyle;

/**
 * A skysail server installation responds to (RESTful) Http-Requests by creating responses, which get converted
 * on-the-fly into the desired target format (like html, json, xml, csv, ...).
 * 
 * The actual data contained in the response is described by the generic type parameter <T>; there are no formal
 * restrictions on that type, but you have to keep in mind that it is supposed to be serializable into formats as JSON,
 * XML and the like.
 * 
 * Besides the actual data a SkysailResponse class provides meta-information about the request and response as follows:
 * 
 * A message: during the request/response cycle, the server can set a message on the response which is passed to the
 * requester as additional information.
 * 
 * A success flag: If an exception is thrown on the server side, skysail still will try to pass an FailureResponse back
 * to the client, the success flag of which would be false obviously.
 * 
 * The execution time and the server load: The server will try to provide the client generic information about the load
 * and the time the request took.
 */
// class not abstract as of http://jira.twentyeleven.de/browse/SC-1
public class SkysailResponse<T> {

    /** a message to be passed to the client. */
    private String message;

    /**
     * Indicates if the request was successful.
     * 
     * If not, the "message" field should contain an error description for the caller.
     */
    private Boolean success = true;

    /** the data or payload, of a type T extending SkysailData. */
    private T data = null;

    @JsonIgnore
    private PresentationStyle presentationStyleHint = PresentationStyle.LIST2;

    // TODO
    /** measured from ... to ... */
    private long executionTime = -1;

    /** server load from MXBean, if available. */
    private double serverLoad = -1;

    public SkysailResponse() {
    }

    public SkysailResponse(T data) {
        this();
        this.data = data;
    }

    // --- the getters and setters ---
    public final Boolean getSuccess() {
        return success;
    }

    public final void setSuccess(final Boolean successful) {
        this.success = successful;
    }

    public final String getMessage() {
        return message;
    }

    public final void setMessage(final String msg) {
        this.message = msg;
    }

    public T getData() {
        return data;
    }

    public final void setData(final T mydata) {
        this.data = mydata;
    }

    public long getExecutionTime() {
        return executionTime;
    }

    public void setExecutionTime(long executionTime) {
        this.executionTime = executionTime;
    }

    public double getServerLoad() {
        return serverLoad;
    }

    public void setServerLoad(double serverLoad) {
        this.serverLoad = serverLoad;
    }

    public void setPresentationStyleHint(PresentationStyle presentationStyleHint) {
        this.presentationStyleHint = presentationStyleHint;
    }

    public PresentationStyle getPresentationStyleHint() {
        return presentationStyleHint;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.getClass().getSimpleName()).append(": ");
        sb.append(getSuccess()).append(", '");
        sb.append(getMessage()).append("', Data: ");
        sb.append(getData().toString());
        return sb.toString();
    }

}
