/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.grids;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.Validate;
import org.codehaus.jackson.annotate.JsonIgnore;

/**
 * A rowData object basically holds a map between the columns name and a value, thus providing access to the cells
 * making up the row.
 * 
 * It can be constructed by either using the constructor taking a columns object and then 'filling it up' by calling
 * 'add' repeatedly, or, by using the default constructor and then calling setCells.
 * 
 * 
 * @author carsten
 * @NotThreadSafe
 */
public class RowData implements Serializable {

    private static final long serialVersionUID = -8906891746212796759L;

    private Map<String, String> cells = new HashMap<String, String>();

    /**
     * the associated columns, accessible via index, provided in the constructor.
     */
    @JsonIgnore
    private ColumnDefinition[] columnsAsArray;

    /**
     * an internal representation of columnInfo/values, build up whenever method 'add' is called.
     */
    @JsonIgnore
    private Map<ColumnDefinition, Object> columnDataMap = new LinkedHashMap<ColumnDefinition, Object>();

    /**
     * the counter of the position in the columnsAsArray member.
     */
    private int columnCounter = 0;

    public RowData() {
    }

    /**
     * constructor fed with the mapping between column names and columnDefinitions.
     * 
     * @param cols
     *            the mapping you can get from getSkysailData().getColumns();
     */
    public RowData(Columns cols) {
    	Validate.notNull(cols);
        columnsAsArray = cols.getAsList().toArray(new ColumnDefinition[cols.getAsList().size()]);
    }

    /**
     * A value object (think of a cell in a grid) gets added to the rowData instance, taking into account the position
     * of the column (by increasing an internal counter each time 'add' is called). The combination
     * columnDefinition/value is stored in an internal map (to be used later)
     * 
     * @param data
     *            a value object (like a cell in a datagrid)
     * @return this (DSL style)
     */
    public RowData add(final Object data) {
        if (columnCounter >= columnsAsArray.length) {
            throw new IllegalStateException("method add cannot be called " + (columnCounter + 1)
                    + " times if there are only " + columnCounter + " columns defined");
        }
        ColumnDefinition columnDefinition = columnsAsArray[columnCounter++];
        columnDataMap.put(columnDefinition, data);
        cells.put(columnDefinition.getName(), data.toString());
        return this;
    }
    
    public RowData add(String colName, String value) {
        cells.put(colName, value);
        return this;
    }

    /**
     * Adds "rows" to the rowData instance unless data is null; in this case the defaultString is added.
     * 
     * @param data
     *            a value object (like a cell in a datagrid)
     * @param defaultString
     *            this string will be added instead of "rows" if data is null. May not be null.
     */
    public void add(final Object data, final String defaultString) {
        if (defaultString == null) {
            throw new IllegalArgumentException(
                    "add(Object,String) in RowData may not be called with null value for String");
        }
        if (data == null) {
            add(defaultString);
        } else {
            add(data);
        }
    }

    /**
     * @return mapping between the name of the columns and the values for the current row
     * 
     */
    public Map<String, String> getCells() {
        return cells;
    }

    /**
     * This is being called for example by jackson when trying to deserialize from a string representation. All the
     * internal members are reconstructed as much as possible, but some information might be lost.
     * 
     * @param props
     */
    public void setCells(Map<String, String> props) {
    	Validate.notNull(props);

        this.cells = props;
        columnDataMap.clear();
        columnsAsArray = new ColumnDefinition[props.size()];
        this.columnCounter = props.size();
        int i = 0;
        for (Map.Entry<String, String> entry : props.entrySet()) {
            ColumnDefinition colDef = new ColumnDefinition(entry.getKey());
            columnDataMap.put(colDef, entry.getValue());
            columnsAsArray[i++] = colDef;
        }
    }

    /**
     * This method might not work like expected when the rowData was created by deserializing a string representation
     * (e.g. by jackson) (though not verified yet).
     * 
     * @return a list of the values, sorted like they were added.
     */
    @JsonIgnore
    public List<Object> getColumnData() {
        return new ArrayList<Object>(columnDataMap.values());
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer("RowData [current Col: ").append(columnCounter).append(", max columns#=")
                .append(columnsAsArray.length);
        for (Object colData : getColumnData()) {
            sb.append("\n  (").append(colData).append(")");
        }
        sb.append("]");
        return sb.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((cells == null) ? 0 : cells.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RowData)) {
            return false;
        }
        RowData other = (RowData) obj;
        if (getCells() == null) {
            if (other.getCells() != null) {
                return false;
            }
        } else if (!getCells().equals(other.getCells())) {
            return false;
        }
        return true;
    }

}
