/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.grids;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.Validate;
import org.codehaus.jackson.annotate.JsonCreator;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonSetter;

import de.twenty11.skysail.common.SkysailData;

/**
 * A grid-like data structure implementing the interface {@link SkysailData}.
 * 
 * A gridData object contains information about the provided columns (see {@link ColumnInfo}), the actual rows (a list
 * of {@link RowData} and an optional filter which will decide whether or not a row passed to "addRowData" will actually
 * be added.
 * 
 * The columns are created using a columnsBuilder which is passed to the constructor. Adding rows is done one by one,
 * using the addRowData method. If a row is added and matches the filter (if any), the availableRowsCount is increased.
 * 
 * @author carsten
 * 
 */
public class GridData implements SkysailData {

    /** serial number. */
    private static final long serialVersionUID = 3808733942569990251L;

    /** the columns, a managed list of columnDefinitions. */
    private Columns columns;

    /** a list of RowData elements forming the rows of the rows. */
    private final List<RowData> rows = new ArrayList<RowData>();

    /**
     * pagination: the total number of rows available (even though maybe less rows will be returned.
     */
    private Integer availableRowsCount = 0;

    String type;

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }

    /** default constructor. */
    @JsonCreator
    public GridData() {
    }

    /**
     * A constructor expecting a columnsBuilder which is used to create the initial columns.
     * 
     * @param builder
     *            the provided ColumnsBuilder
     */
    public GridData(final ColumnsBuilder builder) {
        columns = new Columns(builder);
    }

    /**
     * Adds a row to the rows.
     * 
     * Adding a row (which matches the filter) will increase the availableRowsCount, whereas removing a row from the
     * rows will not change it. The use case for this is to create a rows with all rows which match the filter (thus
     * yielding "availableRowsCount" elements) and then remove the elements which don't fit into the pagination (still
     * leaving the availableRowsCount constant.
     * 
     * @param rowData
     *            the rowData object to be added to the rows
     */
    public final void addRowData(RowData rowData) {
        Validate.notNull(rowData);
        rows.add(rowData);
        availableRowsCount++;
    }

    /**
     * @return how many rows does the grid have currently?
     */
    @JsonIgnore
    public final Integer getAvailableRowsCount() {
        return availableRowsCount;
    }

    /**
     * Delegates to columns object, provides the id of the column for the given name.
     * 
     * @param key
     *            the name of the column for the given id.
     * @return id of the column (0 < id < columns.size())
     */
    @JsonIgnore
    public final Integer getColumnId(String key) {
        return columns.getColumnId(key);
    }

    // @JsonProperty(value = "cols")
    public Columns getColumns() {
        return columns;
    }

    public Object getGridElement(int row, int col) {
        if (rows == null) {
            throw new IllegalArgumentException("trying to access row " + row + " of rows which is null");
        }
        if (rows.size() < row) {
            throw new IllegalArgumentException("trying to access row " + row + " of rows with size of " + rows.size());
        }
        List<Object> columnData = rows.get(row).getColumnData();
        if (columnData.size() < col) {
            throw new IllegalArgumentException("trying to access column " + col + " of grids row with size of "
                    + columnData.size());
        }
        return rows.get(row).getColumnData().get(col);
    }

    /** === simple getters and setters. ==================================== */

    /**
     * @return the rows
     */
    public List<RowData> getRows() {
        return rows;
    }

    @Override
    public int getSize() {
        return rows.size();
    }

    /**
     * needed for json deserialization.
     * 
     * @param size
     */
    public void setSize(int size) {
        // rows.setSize(size);
    }

    /**
     * @return a string representation like "s=0|1|2&" (which can be used as parameter) representing the current sorting
     */
    public final String getSortingRepresentation() {
        return columns.getSortingRepresentation();
    }

    /**
     * needed for json deserialization.
     */
    @JsonIgnore
    public void setSortingRepresentation(int sortingRep) {

    }

    /**
     * removes the grids row at provided position.
     * 
     * The availableRowsCount is not affected.
     * 
     * @param j
     *            the position of the row (0 <= j < size)
     */
    public void removeRow(int j) {
        if (j < rows.size()) {
            rows.remove(j);
        }
    }

    @JsonSetter
    public void setColumns(Columns theColumns) {
        columns = theColumns;
    }

    // @JsonDeserialize(as = ColumnsBuilder.class, contentAs =
    // ColumnDefinition.class)
    @JsonIgnore
    public void setColumns(ColumnsBuilder builder) {
        columns = new Columns(builder);
    }

    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer("GridData {\n");
        if (columns != null) {
            sb.append(columns.toString());
        }
        if (rows != null) {
            sb.append("  GridData (").append(getSize()).append("): ");
            for (RowData row : rows) {
                sb.append(row).append(",");
            }
            sb.append("\n");
        }
        sb.append("}");
        return sb.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((availableRowsCount == null) ? 0 : availableRowsCount.hashCode());
        result = prime * result + ((columns == null) ? 0 : columns.hashCode());
        result = prime * result + ((rows == null) ? 0 : rows.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !(obj instanceof GridData)) {
            return false;
        }
        GridData other = (GridData) obj;
        if (availableRowsCountIsDifferent(other)) {
            return false;
        }
        if (columnsAreDifferent(other)) {
            return false;
        }
        if (rowsAreDifferent(other)) {
            return false;
        }
        return true;
    }

    private boolean rowsAreDifferent(GridData other) {
        if (rows == null) {
            if (other.rows != null) {
                return true;
            }
        } else if (!rows.equals(other.rows)) {
            return true;
        }
        return false;
    }

    private boolean columnsAreDifferent(GridData other) {
        if (columns == null) {
            if (other.columns != null) {
                return true;
            }
        } else if (!columns.equals(other.columns)) {
            return true;
        }
        return false;
    }

    private boolean availableRowsCountIsDifferent(GridData other) {
        if (availableRowsCount == null) {
            if (other.availableRowsCount != null) {
                return true;
            }
        } else if (!availableRowsCount.equals(other.availableRowsCount)) {
            return true;
        }
        return false;
    }

}
