/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.responses;

import java.util.Set;

import org.restlet.Request;

import de.twenty11.skysail.common.SkysailData;
import de.twenty11.skysail.common.navigation.Navigation;

/**
 * The root class for restful responses served by skysail server. A skysail response always wraps an object of type
 * {@link SkysailData} which is available via the method {@link Response#getData()}.
 * 
 * Futhermore, a skysail response provides:
 * 
 * - a success flag: did the request run ok on the server side?
 * 
 * - a message: a string providing some information about the response (or an error message if appropriate)
 * 
 * This class is used in {@link SkysailServerResource} to define the common structure how skysail REST representations
 * of resources look like. The plain 'data' of the resource itself is augmented with the additional information shown
 * above.
 * 
 * Remark: This class is not abstract as of jackson serialization issues
 * 
 * @param <T>
 *            a type extending SkysailData
 */
public class Response<T> {

    /**
     * a message to be passed to the client.
     */
    private String message;

    /**
     * Indicates if the request was successful.
     * 
     * If not, the "message" field should contain an error description for the caller.
     */
    private Boolean success = true;

    /** A reference to a class providing information about the navigation context. */
    private Navigation navigation = null;

    /** LDAP search filter expression, see http://www.ietf.org/rfc/rfc1960.txt. */
    // private String filter;

    /** the data or payload, of a type T extending SkysailData. */
    private T data = null;

    /** violations of validating T. */
    protected T validationViolations;

    public Response() {
    }

    public Response(T data) {
        this.data = data;
    }

    public Response(T data, Request request, Set<String> mappings) {
        this(data);
        navigation = new Navigation(request, mappings);
    }

    // --- the getters and setters ---
    public final Boolean getSuccess() {
        return success;
    }

    public final void setSuccess(final Boolean successful) {
        this.success = successful;
    }

    public final String getMessage() {
        return message;
    }

    public final void setMessage(final String msg) {
        this.message = msg;
    }

    public void setNavigation(Navigation navigation) {
        this.navigation = navigation;
    }

    public T getData() {
        return data;
    }

    public T getValidationViolations() {
        return validationViolations;
    }

    public final void setData(final T mydata) {
        this.data = mydata;
    }

    public void setValidationViolations(T constraintViolations) {
        this.validationViolations = constraintViolations;
    }

    public Navigation getNavigation() {
        return navigation;
    }

}
