/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.grids;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Subclasses have to implement the configure method, which typically looks something like
 * addColumn("col1").setWidth(100).addColumn("col2")...
 * 
 * This class is meant to be used a domain-specific-language style which lets you define a chain of columns and provide
 * specific details to each column. The typical use is to create an anonymous inner class like in this snippet
 * <code>gridData.setColumnsBuilder(new ColumnsBuilder(...) {...});</code>. This use case doesn't have to deal with
 * concurrency issues, so no special care is taken to make this class ThreadSafe.
 * 
 * @NotThreadSafe
 * @author carsten
 * 
 */
public abstract class ColumnsBuilder {

    /** list of columns defined by their columnDefinition. */
    private final List<ColumnDefinition> columnDefinitions = new LinkedList<ColumnDefinition>();

    /** String-Set tracking the names of the columns. */
    private final Set<String> columnNames = new HashSet<String>();

    /**
     * parameters passed on creation to provide filter expressions on the columns.
     */
    private Map<String, String> filterKeyValues;

    /** is set to true in the constructor to avoid calls of addColumn after initialization. */
    private boolean configured = false;

    /**
     * The central method to configure the chain of columns. Subclasses will call addColumn repeatedly to add new
     * Columns.
     */
    public abstract void configure();

    // public abstract Map<String, String> getFilter();

    /**
     * Creates a new ColumnsBuilder object which gets a map of parameters.
     * 
     * These parameters are used to pass filter values, e.g. in a http request, you might find something like
     * column1=foo. If there is a column called "column1", its filtervalue will be set to "foo".
     * 
     */
    public ColumnsBuilder() {
        configure();
        applyFilter();
        configured = true;
    }

    /**
     * Add a new column to the chain.
     * 
     * @param name
     *            the name of the new column
     * @return the new column.
     */
    public final ColumnDefinition addColumn(final String name) {
        if (configured) {
            throw new IllegalStateException(
                    "addColumn is not to be called after getColumns has been called the first time.");
        }
        ColumnDefinition columnDefinition = new ColumnDefinition(name);
        columnDefinitions.add(columnDefinition);
        columnNames.add(name);
        return columnDefinition;
    }

    /**
     * return the chain of columns (which should have been created in the configure method).
     * 
     * @return the current chain of columns.
     */
    public final List<ColumnDefinition> getColumns() {
        return columnDefinitions;// Collections.unmodifiableList(columnDefinitions);
    }

    /**
     * must be called after configure().
     */
    private void applyFilter() {
    }


    @Override
    public final String toString() {
        return columnDefinitions.toString();
    }

}
