package de.twenty11.skysail.common.grids;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

import org.apache.commons.lang.Validate;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.annotate.JsonProperty;

/**
 * Manages a data grids' list of columns.
 * 
 * The columnsList are represented as a list of columnDefinitions which is created in the constructor utilizing a
 * columnBuilder.
 * 
 * @author carsten
 * 
 */
public class Columns {

    /** the columnsList, a list of columnDefinitions. */
    private List<ColumnDefinition> columnsList;

    /**
     * a string like "s=0|1|-2|3&" which can be used as a parameter to describe the current sorting.
     */
    private String sortingRepresentation;

    /**
     * Constructor creating the columnsList from a columnBuilder.
     * 
     * @param builder
     *            a ColumnsBuilder which is used to create the columnsList
     */
    public Columns(ColumnsBuilder builder) {
        Validate.notNull(builder);
        columnsList = builder.getColumns();
    }

    /**
     * for json.
     */
    public Columns() {
    }

    /**
     * @return the columnsList sorted by their sort order
     */
    @JsonIgnore
    public final List<ColumnDefinition> getColumnsInSortOrder() {
        List<ColumnDefinition> copyForSorting = new LinkedList<ColumnDefinition>();
        if (columnsList != null) {
            for (ColumnDefinition columnDefinition : columnsList) {
                copyForSorting.add(columnDefinition);
            }
            Collections.sort(copyForSorting, new ColumnSortOrderComparator());
        }
        return copyForSorting; // Collections.unmodifiableList(copyForSorting);
    }

    /**
     * needed for json deserialization.
     */
    public void setColumnsList(List<ColumnDefinition> theColumns) {
        this.columnsList = theColumns;
    }

    @JsonIgnore
    public int getMaxSortValueFromBuilder() {
        List<ColumnDefinition> sortedColumns = getColumnsInSortOrder();
        ColumnDefinition[] columnsAsArray = sortedColumns.toArray(new ColumnDefinition[sortedColumns.size()]);
        if (columnsAsArray.length == 0) {
            return 0;
        }
        Integer maxSorting = columnsAsArray[sortedColumns.size() - 1].getSorting();
        return maxSorting != null ? Math.abs(maxSorting) : 0;
    }

    public List<ColumnDefinition> sort(String sortingInstructionFromRequest, Integer colIdToToggle) {

        validate(sortingInstructionFromRequest, columnsList.size());

        Map<Integer, Integer> sortingMap = new TreeMap<Integer, Integer>();

        // 1.) get the default sorting for the grid by asking for the pre-sorted
        // columnsList (ascending by sort weight). The columnsList with the highest
        // sort-order come last columnsList.
        List<ColumnDefinition> preSortedColumns = getColumnsInSortOrder();

        // 2.) check if a sorting instruction exists on the request and
        // overwrite the default sorting in that case
        // String sortingInstructionFromRequest = getSortingFromRequest();
        for (ColumnDefinition currentColumn : preSortedColumns) {
            setSortingAsByRequest(sortingMap, sortingInstructionFromRequest, currentColumn);
        }

        // is there any column the sorting of which should be toggled?
        ColumnDefinition columnToToggle = getColumnDefinitionToToggle(preSortedColumns, colIdToToggle);

        int maxSortWeight = getMaxSortValueFromBuilder();

        if (columnToToggle != null) {
            toggleColumnsSorting(maxSortWeight, sortingMap, columnToToggle);
        }

        sortingRepresentation = calcSortingRepresentation(sortingMap);
        // setSorting(sortingRepresentation);

        Collections.sort(preSortedColumns, new ColumnSortOrderComparator());

        return preSortedColumns;
    }

    private void toggleColumnsSorting(int maxSortValue, Map<Integer, Integer> sortingMap,
            ColumnDefinition columnToToggle) {
        Integer sorting = columnToToggle.getSorting();
        if (sorting > 0) {
            columnToToggle.sortAsc((maxSortValue + 1));
            sortingMap.put(getColumnId(columnToToggle.getName()), -(maxSortValue + 1));
        } else if (sorting == 0) {
            columnToToggle.sortDesc((maxSortValue + 1));
            sortingMap.put(getColumnId(columnToToggle.getName()), (maxSortValue + 1));
        } else {
            columnToToggle.sortDesc(0);
            sortingMap.put(getColumnId(columnToToggle.getName()), 0);
        }
    }

    /**
     * checks the sorting parameter and throws unchecked exception is not valid.
     * 
     * @param override
     * @param size
     * @return
     */
    private void validate(String override, int size) {
        if (override == null || override.trim().equals("")) {
            return;
        }
        String[] sortValuesFromRequest = override.split("\\|");
        if (sortValuesFromRequest.length != size) {
            throw new IllegalArgumentException("parameter 's' for sorting has " + sortValuesFromRequest.length
                    + "parts, but should have " + size);
        }
        for (String part : sortValuesFromRequest) {
            try {
                Integer.valueOf(part);
            } catch (Exception e) {
                throw new IllegalArgumentException("could not parse '" + part + "' of sorting parameter " + override
                        + " as Integer");
            }
        }
    }

    private ColumnDefinition getColumnDefinitionToToggle(List<ColumnDefinition> columnsInSortOrder, Integer columnId) {
        if (columnId == null) {
            return null;
        }
        String columnNameToToggle = getColumnName(columnId);
        // return columnsInSortOrder.get(columnNameToToggle);
        for (ColumnDefinition columnDefinition : columnsInSortOrder) {
            if (columnDefinition.getName().equals(columnNameToToggle)) {
                return columnDefinition;
            }
        }
        return null;
    }

    private void setSortingAsByRequest(Map<Integer, Integer> sortingMap, String override, ColumnDefinition currentColumn) {
        Validate.notNull(currentColumn);
        Integer sorting = currentColumn.getSorting();
        Integer columnId = getColumnId(currentColumn.getName());
        if (override != null && override.trim().length() > 0) {
            String[] sortValuesFromRequest = override.split("\\|");
            sorting = Integer.valueOf(sortValuesFromRequest[columnId]);
        }
        sortingMap.put(columnId, sorting);
    }

    @JsonProperty(value = "columnsList")
    public List<ColumnDefinition> getAsList() {
        return columnsList;
    }

    @JsonIgnore
    public String getSortingRepresentation() {
        return sortingRepresentation;
    }

    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("  Columns (").append(columnsList.size()).append("): ");
        for (ColumnDefinition col : columnsList) {
            sb.append(col).append(",");
        }
        sb.append("\n");
        return sb.toString();
    }

    /**
     * @param key
     *            the key for the column map
     * @return the id of the column.
     */
    public final String getColumnName(final Integer key) {
        return columnsList.get(key).getName();
    }

    public Integer getColumnId(String columnName) {
        Validate.notNull(columnName);
        for (int i = 0; i < columnsList.size(); i++) {
            ColumnDefinition column = columnsList.get(i);
            if (column.getName().equals(columnName)) {
                return i;
            }
        }
        return null;
    }

    private String calcSortingRepresentation(Map<Integer, Integer> sortingMap) {
        Validate.notNull(sortingMap);
        String sortingString = "";
        if (sortingMap.size() > 0) {
            StringBuffer sb = new StringBuffer("s=");
            for (Entry<Integer, Integer> entry : sortingMap.entrySet()) {
                sb.append(entry.getValue() != null ? entry.getValue() + "|" : 0 + "|");
            }
            sortingString = sb.toString().substring(0, sb.length() - 1) + "&";
        }
        return sortingString;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((columnsList == null) ? 0 : columnsList.hashCode());
        result = prime * result + ((sortingRepresentation == null) ? 0 : sortingRepresentation.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Columns)) {
            return false;
        }
        Columns other = (Columns) obj;
        if (columnsList == null) {
            if (other.columnsList != null) {
                return false;
            }
        } else if (!columnsList.equals(other.columnsList)) {
            return false;
        }
        if (sortingRepresentation == null) {
            if (other.sortingRepresentation != null) {
                return false;
            }
        } else if (!sortingRepresentation.equals(other.sortingRepresentation)) {
            return false;
        }
        return true;
    }

}
