/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.grids;

import java.io.Serializable;

/**
 * Defines a grid column.
 * 
 * A grid column has a name, a filter (maybe empty), a "sorting" field (defaults
 * to 0), and a width (defaults to null). Columns can be "chained" together
 * using a {@link ColumnsBuilder} in a DSL-style fashion. The column definitions
 * can be used to compose a grid out of its defining columns and to display it
 * (with proper filtering, sorting and width).
 * 
 * @author carsten
 * 
 */
public class ColumnDefinition implements Serializable {

    private static final long serialVersionUID = 7871296812642842749L;

    private String name;

    /** width (hint) of the column. Null means: no hint. */
    private Integer width = null;

    /**
     * Idea: Positive values indicate descending ordering, negative ones
     * ascending. the higher the number (if positive), the higher the weight of
     * this column, e.g. if one column A has "3" and column "B" has value "2",
     * sorting will be like "sort by A desc, B desc". Accordingly for negative
     * numbers.
     * 
     * sorting set to NULL means no sorting should be possible on this column.
     * 
     */
    private Integer sorting = 0;

    /**
     * A string representation of the filter to be applied, if any.
     * 
     * Otherwise empty String.
     */
    private String filterValue = "";

    /**
     * default constructor needed for json / jackson de-/serialization.
     * 
     * @return
     */
    public ColumnDefinition() {
    }

    /**
     * constructor with name for the column definition.
     * 
     * @param columnName
     *            for the column
     */
    public ColumnDefinition(final String columnName) {
        this.name = columnName;
    }

    public final void apply() {
    }

    /**
     * @param weight
     *            value used for deciding about precedence
     * @return "this" to be used in DSL-style.
     */
    public final ColumnDefinition sortDesc(final Integer weight) {
        sorting = null;
        if (weight != null) {
            sorting = Math.abs(weight);
        }
        return this;
    }

    /**
     * @param weight
     *            value used for deciding about precedence
     * @return "this" to be used in DSL-style.
     */
    public final ColumnDefinition sortAsc(final Integer weight) {
        sorting = null;
        if (weight != null) {
            sorting = -Math.abs(weight);
        }
        return this;
    }

    /**
     * @param colWidth
     *            in pixel
     * @return "this" to be used in DSL-style.
     */
    public final ColumnDefinition setWidth(final Integer colWidth) {
        this.width = colWidth;
        return this;
    }

    /**
     * 
     * @return the width of this column.
     */
    public final Integer getWidth() {
        return width;
    }

    /**
     * returns the filter value.
     * 
     * @return a string filter representation
     */
    public final String getFilterValue() {
        return filterValue;
    }

    /**
     * needed for jackson.
     * 
     */
    public void setFilterValue(String filter) {
        this.filterValue = filter;
    }

    /**
     * needed for jackson.
     * 
     */
    public void setName(String theName) {
        this.name = theName;
    }

    /**
     * returns the name.
     * 
     * @return the name
     */
    public final String getName() {
        return name;
    }

    /**
     * returns the sorting.
     * 
     * @return the sorting
     */
    public final Integer getSorting() {
        return sorting;
    }

    /**
     * needed for jackson.
     * 
     */
    public void setSorting(Integer theSorting) {
        this.sorting = theSorting;
    }

    @Override
    public final String toString() {
        StringBuffer sb = new StringBuffer("ColumnDefinition: {");
        sb.append("Name: ").append(name);
        sb.append(", Width: ").append(width);
        sb.append(", Sorting: ").append(sorting);
        sb.append(", Filter: '").append(filterValue).append("'}\n");
        return sb.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((filterValue == null) ? 0 : filterValue.hashCode());
        result = prime * result + ((name == null) ? 0 : name.hashCode());
        result = prime * result + ((sorting == null) ? 0 : sorting.hashCode());
        result = prime * result + ((width == null) ? 0 : width.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    @Override
    public final boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ColumnDefinition)) {
            return false;
        }
        ColumnDefinition other = (ColumnDefinition) obj;
        if (filterValue == null) {
            if (other.filterValue != null) {
                return false;
            }
        } else if (!filterValue.equals(other.filterValue)) {
            return false;
        }
        if (name == null) {
            if (other.name != null) {
                return false;
            }
        } else if (!name.equals(other.name)) {
            return false;
        }
        if (sorting == null) {
            if (other.sorting != null) {
                return false;
            }
        } else if (!sorting.equals(other.sorting)) {
            return false;
        }
        if (width == null) {
            if (other.width != null) {
                return false;
            }
        } else if (!width.equals(other.width)) {
            return false;
        }
        return true;
    }

}
