/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.forms;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Subclasses have to implement the configure method, which typically looks
 * something like // TODO
 * 
 * This is a domain-specific-language style which lets you define a chain of
 * columns. The typical use is to create an anonymous inner class like in this
 * snippet <code>form.setFormBuilder(new FormBuilder(...) {...});</code>. This
 * use case doesn't have to deal with concurrency issues, so no special care is
 * taken to make this class ThreadSafe.
 * 
 * @author carsten
 * 
 */
public abstract class FormBuilder {

    /**
     * named chain of fields.
     */
    private Map<String, FieldDefinition> fieldDefinitions = new LinkedHashMap<String, FieldDefinition>();

    /**
     * parameters passed on creation to provide filter expressions on the
     * columns.
     */
    //private final Map<String, String> requestParameters;

    /**
     * The central method to configure the chain of columns.
     */
    public abstract void configure();

    /**
     * Creates a new ColumnsBuilder object which gets a map of parameters.
     * 
     * These parameters are used to pass filter values, e.g. in a http request,
     * you might find something like column1=foo. If there is a column called
     * "column1", its filtervalue will be set to "foo".
     * 
     * @param parameters
     *            typically the request parameters
     */
//    public FormBuilder(final Map<String, String> parameters) {
//        this.requestParameters = parameters;
//    }

    /**
     * Add a new field to the chain.
     * @param type 
     *            html type
     * @param fieldName
     *            the name of the new column
     * @return the new field definition.
     */
    public final FieldDefinition addField(final String fieldName, FieldType type) {
        FieldDefinition fieldDefinition = new FieldDefinition(fieldName, type, this);
        fieldDefinitions.put(fieldName, fieldDefinition);
        return fieldDefinition;
    }

    /**
     * return the current chain of columns.
     * 
     * This chain is recalculated for each call as the chain might have been
     * changed after the initial configuration (which happens when this method
     * is called the first time).
     * 
     * @return the current chain of columns.
     */
    public final Map<String, FieldDefinition> getFields() {
        clear();
        configure();
        // doing this prevents freemarker from properly sorting the map... ;(
        // return Collections.unmodifiableMap(columnDefinitions);
        return fieldDefinitions;
    }

    /**
     * clean the maps.
     */
    private void clear() {
        fieldDefinitions.clear();
    }

    @Override
    public final String toString() {
        if (fieldDefinitions == null || fieldDefinitions.size() == 0) {
            return "Empty column list. Did you run getColumns() yet?";
        }
        return fieldDefinitions.toString();
    }

    public Map<String, FieldDefinition> getColumns() {
        return getFields();
    }

}
