/**
 *  Copyright 2011 Carsten Gräf
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 * 
 */

package de.twenty11.skysail.common.forms;

import java.io.Serializable;
import java.util.logging.Logger;

import org.codehaus.jackson.annotate.JsonIgnore;

import com.thoughtworks.xstream.annotations.XStreamOmitField;

/**
 * Defines a html form field.
 * 
 * 
 * @author carsten
 * 
 */
public class FieldDefinition implements Serializable {

    /** serial id. */
    private static final long serialVersionUID = 6627992829570047981L;

    /** name of the field, should be unique per form. */
    private String name;

    /** http://www.w3.org/TR/html401/interact/forms.html. */
    private FieldType type;

    /**
     * A builder to create Columns using a DSL.
     * 
     * This field should not be serialized or taken into account by json,
     * xstream or the like
     */
    @XStreamOmitField
    @JsonIgnore
    private transient FormBuilder builder;

    /** the string representation of the value to be shown. */
    private String value;

    /**
     * java-util-logging based log4jLogger implementation, we dont want to make
     * skysail.common depend on the usual slf4j. If this is being run on the
     * server side, logging gets redirected to slf4j anyway.
     */
    private static Logger logger = Logger.getLogger(FieldDefinition.class.getName());

    /**
     * alternative constructor, taking name and builder.
     * 
     * @param fieldName
     *            name of the column
     * @param fieldType
     *            the html type
     * @param formBuilder
     *            the builder
     */
    public FieldDefinition(final String fieldName, final FieldType fieldType, final FormBuilder formBuilder) {
        this.name = fieldName;
        this.type = fieldType;
        this.builder = formBuilder;
    }

    /**
     * This method supports a DSL-style to add further columns to a chain of
     * already available columnDefinitions.
     * 
     * Attention: Call this method only from builer.configure!
     * 
     * Make sure to have a ColumnBuilder set first.
     * 
     * @param fieldName
     *            of the new field.
     * @param fieldType
     *            of the new field.
     * @return A new columnDefinition if a builder was available, null
     *         otherwise.
     */
    public final FieldDefinition addField(final String fieldName, final FieldType fieldType) {
        if (builder == null) {
            logger.warning("Field being added to fieldDefinition without a FieldBuilder available!");
            return null;
        }
        FieldDefinition addedField = builder.addField(fieldName, fieldType);
        return addedField;
    }

    /**
     * returns the name.
     * 
     * @return the name
     */
    public final String getName() {
        return name;
    }

    @Override
    public final String toString() {
        return "<input type='text' value='" + value + "'>";
    }

    public void setValue(final String provided) {
        this.value = provided;
    }

    public String getValue() {
        return value;
    }

}
