package de.tum.in.ase.parser.strategy;

import java.util.Objects;
import java.util.Optional;

public enum StaticCodeAnalysisTool {
    SPOTBUGS("BugCollection", "spotbugsXml.xml", new SpotbugsParser()),
    CHECKSTYLE("checkstyle", "checkstyle-result.xml", new CheckstyleParser()),
    SWIFTLINT("checkstyle", "swiftlint-result.xml", new SwiftLintParser()),
    PMD("pmd", "pmd.xml", new PMDParser()),
    PMD_CPD("pmd-cpd", "cpd.xml", new PMDCPDParser());

    private final String identifierTag;
    private final String filename;
    private final ParserStrategy strategy;

    StaticCodeAnalysisTool(String identifyingTag, String filename, ParserStrategy strategy) {
        this.identifierTag = identifyingTag;
        this.filename = filename;
        this.strategy = strategy;
    }

    public String getIdentifyingTag() {
        return this.identifierTag;
    }

    public String getFilename() {
        return this.filename;
    }

    public ParserStrategy getStrategy() {
        return this.strategy;
    }

    /**
     * Finds a tool by its identifying tag (unique element name for the specific report)
     *
     * @param identifierTag tag to search for
     * @return Optional with the corresponding tool or empty optional if no appropriate tool was found
     */
    public static Optional<StaticCodeAnalysisTool> getToolByIdentifierTag(String identifierTag) {
        for (StaticCodeAnalysisTool tool : StaticCodeAnalysisTool.values()) {
            if (Objects.equals(identifierTag, tool.getIdentifyingTag())) {
                return Optional.of(tool);
            }
        }
        return Optional.empty();
    }

    /**
     * Finds a tool by its filename (the xml file generated by the tool).
     * 
     * @param filename the name of the xml file generated by the tool
     * @return Optional with the corresponding tool or empty optional if no appropriate tool was found 
     */
    public static Optional<StaticCodeAnalysisTool> getToolByFilename(String filename) {
        for (StaticCodeAnalysisTool tool : StaticCodeAnalysisTool.values()) {
            if (Objects.equals(filename, tool.getFilename())) {
                return Optional.of(tool);
            }
        }
        return Optional.empty();
    }
}
