/*
 * Decompiled with CFR 0.152.
 */
package de.trustable.ca3s.core.service;

import de.trustable.ca3s.core.domain.Authority;
import de.trustable.ca3s.core.domain.Tenant;
import de.trustable.ca3s.core.domain.User;
import de.trustable.ca3s.core.repository.AuthorityRepository;
import de.trustable.ca3s.core.repository.TenantRepository;
import de.trustable.ca3s.core.repository.UserRepository;
import de.trustable.ca3s.core.security.SecurityUtils;
import de.trustable.ca3s.core.service.EmailAlreadyUsedException;
import de.trustable.ca3s.core.service.InvalidPasswordException;
import de.trustable.ca3s.core.service.UsernameAlreadyUsedException;
import de.trustable.ca3s.core.service.dto.UserDTO;
import de.trustable.ca3s.core.service.util.PasswordUtil;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.CacheManager;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import tech.jhipster.security.RandomUtil;

@Service
@Transactional
public class UserService {
    private final Logger log = LoggerFactory.getLogger(UserService.class);
    private final UserRepository userRepository;
    private final PasswordEncoder passwordEncoder;
    private final AuthorityRepository authorityRepository;
    private final TenantRepository tenantRepository;
    private final CacheManager cacheManager;
    private final PasswordUtil passwordUtil;

    public UserService(UserRepository userRepository, PasswordEncoder passwordEncoder, AuthorityRepository authorityRepository, TenantRepository tenantRepository, CacheManager cacheManager, @Value(value="${ca3s.ui.password.check.regexp:^(?=.*\\d)(?=.*[a-z]).{6,100}$}") String passwordCheckRegExp) {
        this.userRepository = userRepository;
        this.passwordEncoder = passwordEncoder;
        this.authorityRepository = authorityRepository;
        this.tenantRepository = tenantRepository;
        this.cacheManager = cacheManager;
        this.passwordUtil = new PasswordUtil(passwordCheckRegExp);
    }

    public Optional<User> activateRegistration(String key) {
        this.log.debug("Activating user for activation key {}", (Object)key);
        return this.userRepository.findOneByActivationKey(key).map(user -> {
            user.setActivated(true);
            user.setActivationKey(null);
            this.clearUserCaches(user);
            this.log.debug("Activated user: {}", user);
            return user;
        });
    }

    public Optional<User> completePasswordReset(String newPassword, String key) {
        this.log.debug("Reset user password for reset key {}", (Object)key);
        return this.userRepository.findOneByResetKey(key).filter(user -> user.getResetDate().isAfter(Instant.now().minusSeconds(86400L))).map(user -> {
            user.setPassword(this.passwordEncoder.encode((CharSequence)newPassword));
            user.setResetKey(null);
            user.setResetDate(null);
            this.clearUserCaches(user);
            return user;
        });
    }

    public Optional<User> requestPasswordReset(String username) {
        return this.userRepository.findOneByLogin(username).filter(User::getActivated).map(user -> {
            user.setResetKey(RandomUtil.generateResetKey());
            user.setResetDate(Instant.now());
            this.clearUserCaches(user);
            return user;
        });
    }

    public User registerUser(UserDTO userDTO, String password) {
        this.userRepository.findOneByLogin(userDTO.getLogin().toLowerCase()).ifPresent(existingUser -> {
            boolean removed = this.removeNonActivatedUser(existingUser);
            if (!removed) {
                throw new UsernameAlreadyUsedException();
            }
        });
        this.userRepository.findOneByEmailIgnoreCase(userDTO.getEmail()).ifPresent(existingUser -> {
            boolean removed = this.removeNonActivatedUser(existingUser);
            if (!removed) {
                throw new EmailAlreadyUsedException();
            }
        });
        User newUser = new User();
        String encryptedPassword = this.passwordEncoder.encode((CharSequence)password);
        newUser.setLogin(userDTO.getLogin().toLowerCase());
        newUser.setPassword(encryptedPassword);
        newUser.setFirstName(userDTO.getFirstName());
        newUser.setLastName(userDTO.getLastName());
        if (userDTO.getEmail() != null) {
            newUser.setEmail(userDTO.getEmail().toLowerCase());
        }
        newUser.setImageUrl(userDTO.getImageUrl());
        newUser.setLangKey(userDTO.getLangKey());
        newUser.setActivated(false);
        newUser.setActivationKey(RandomUtil.generateActivationKey());
        HashSet authorities = new HashSet();
        this.authorityRepository.findById((Object)"ROLE_USER").ifPresent(authorities::add);
        newUser.setAuthorities(authorities);
        this.updateTenant(newUser, userDTO);
        this.userRepository.save((Object)newUser);
        this.clearUserCaches(newUser);
        this.log.debug("Created Information for User: {}", (Object)newUser);
        return newUser;
    }

    private boolean removeNonActivatedUser(User existingUser) {
        if (existingUser.getActivated()) {
            return false;
        }
        this.userRepository.delete((Object)existingUser);
        this.userRepository.flush();
        this.clearUserCaches(existingUser);
        return true;
    }

    public User createUser(UserDTO userDTO) {
        User user = new User();
        user.setLogin(userDTO.getLogin().toLowerCase());
        user.setFirstName(userDTO.getFirstName());
        user.setLastName(userDTO.getLastName());
        if (userDTO.getEmail() != null) {
            user.setEmail(userDTO.getEmail().toLowerCase());
        }
        user.setImageUrl(userDTO.getImageUrl());
        if (userDTO.getLangKey() == null) {
            user.setLangKey("en");
        } else {
            user.setLangKey(userDTO.getLangKey());
        }
        String encryptedPassword = this.passwordEncoder.encode((CharSequence)RandomUtil.generatePassword());
        user.setPassword(encryptedPassword);
        user.setResetKey(RandomUtil.generateResetKey());
        user.setResetDate(Instant.now());
        user.setActivated(true);
        if (userDTO.getAuthorities() != null) {
            Set authorities = userDTO.getAuthorities().stream().map(arg_0 -> ((AuthorityRepository)this.authorityRepository).findById(arg_0)).filter(Optional::isPresent).map(Optional::get).collect(Collectors.toSet());
            user.setAuthorities(authorities);
        }
        this.updateTenant(user, userDTO);
        this.userRepository.save((Object)user);
        this.clearUserCaches(user);
        this.log.debug("Created Information for User: {}", (Object)user);
        return user;
    }

    public void updateUser(String firstName, String lastName, String email, String langKey, String imageUrl, Long tenantId) {
        SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneByLogin(arg_0)).ifPresent(user -> {
            user.setFirstName(firstName);
            user.setLastName(lastName);
            if (email != null) {
                user.setEmail(email.toLowerCase());
            }
            user.setLangKey(langKey);
            user.setImageUrl(imageUrl);
            this.updateTenant(user, tenantId);
            this.clearUserCaches(user);
            this.log.debug("Changed Information for User: {}", user);
        });
    }

    public Optional<UserDTO> updateUser(UserDTO userDTO) {
        return Optional.of(this.userRepository.findById((Object)userDTO.getId())).filter(Optional::isPresent).map(Optional::get).map(user -> {
            this.clearUserCaches(user);
            user.setLogin(userDTO.getLogin().toLowerCase());
            user.setFirstName(userDTO.getFirstName());
            user.setLastName(userDTO.getLastName());
            if (userDTO.getEmail() != null) {
                user.setEmail(userDTO.getEmail().toLowerCase());
            }
            user.setImageUrl(userDTO.getImageUrl());
            user.setActivated(userDTO.isActivated());
            user.setLangKey(userDTO.getLangKey());
            this.updateTenant(user, userDTO);
            user.setBlockedUntilDate(userDTO.getBlockedUntilDate());
            user.setFailedLogins(userDTO.getFailedLogins());
            Set managedAuthorities = user.getAuthorities();
            managedAuthorities.clear();
            userDTO.getAuthorities().stream().map(arg_0 -> ((AuthorityRepository)this.authorityRepository).findById(arg_0)).filter(Optional::isPresent).map(Optional::get).forEach(managedAuthorities::add);
            this.clearUserCaches(user);
            this.log.debug("Changed Information for User: {}", user);
            return user;
        }).map(UserDTO::new);
    }

    private void updateTenant(User user, UserDTO userDTO) {
        this.updateTenant(user, userDTO.getTenantId());
    }

    private void updateTenant(User user, Long tenantId) {
        Optional optionalTenant;
        Tenant tenant = null;
        if (tenantId != null && tenantId != 0L && (optionalTenant = this.tenantRepository.findById((Object)tenantId)).isPresent()) {
            tenant = (Tenant)optionalTenant.get();
        }
        user.setTenant(tenant);
    }

    public void deleteUser(String login) {
        this.userRepository.findOneByLogin(login).ifPresent(user -> {
            this.userRepository.delete(user);
            this.clearUserCaches(user);
            this.log.debug("Deleted User: {}", user);
        });
    }

    public void changePassword(String currentClearTextPassword, String newPassword) {
        SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneByLogin(arg_0)).ifPresent(user -> {
            String currentEncryptedPassword = user.getPassword();
            if (!this.passwordEncoder.matches((CharSequence)currentClearTextPassword, currentEncryptedPassword)) {
                throw new InvalidPasswordException();
            }
            String encryptedPassword = this.passwordEncoder.encode((CharSequence)newPassword);
            user.setPassword(encryptedPassword);
            this.clearUserCaches(user);
            this.log.debug("Changed password for User: {}", user);
        });
    }

    @Transactional(readOnly=true)
    public Page<UserDTO> getAllManagedUsers(Pageable pageable) {
        return this.userRepository.findAllByLoginNot(pageable, "anonymoususer").map(UserDTO::new);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthoritiesByLogin(String login) {
        return this.userRepository.findOneWithAuthoritiesByLogin(login);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthorities(Long id) {
        return this.userRepository.findOneWithAuthoritiesById(id);
    }

    @Transactional(readOnly=true)
    public Optional<User> getUserWithAuthorities() {
        return SecurityUtils.getCurrentUserLogin().flatMap(arg_0 -> ((UserRepository)this.userRepository).findOneWithAuthoritiesByLogin(arg_0));
    }

    @Scheduled(cron="0 0 1 * * ?")
    public void removeNotActivatedUsers() {
        this.userRepository.findAllByActivatedIsFalseAndActivationKeyIsNotNullAndCreatedDateBefore(Instant.now().minus(3L, ChronoUnit.DAYS)).forEach(user -> {
            this.log.debug("Deleting not activated user {}", (Object)user.getLogin());
            this.userRepository.delete(user);
            this.clearUserCaches(user);
        });
    }

    public List<String> getAuthorities() {
        return this.authorityRepository.findAll().stream().map(Authority::getName).collect(Collectors.toList());
    }

    @Transactional(readOnly=true)
    public List<User> getUsersByRole(String role) {
        return this.userRepository.findActiveByRole(role);
    }

    private void clearUserCaches(User user) {
        Objects.requireNonNull(this.cacheManager.getCache("usersByLogin")).evict((Object)user.getLogin());
        if (user.getEmail() != null) {
            Objects.requireNonNull(this.cacheManager.getCache("usersByEmail")).evict((Object)user.getEmail());
        }
    }

    public void checkPassword(String password) {
        this.passwordUtil.checkPassword(password, "user password");
    }
}

