package de.thomas_oster.rest2typescript;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.reflections.Reflections;
import org.reflections.scanners.MethodAnnotationsScanner;
import org.reflections.util.ConfigurationBuilder;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.reflections.scanners.SubTypesScanner;
import org.reflections.scanners.TypeAnnotationsScanner;

/**
 * Goal for scanning class files looking for Spring Controller annotations
 * and generate a Typescirpt file
 */
@Mojo(name = "generate", defaultPhase = LifecyclePhase.PROCESS_TEST_CLASSES,
        requiresDependencyResolution = ResolutionScope.TEST)
public class Rest2TypescriptPlugin extends AbstractMojo {

    /**
     * Injected by maven to give us a reference to the project so we can get the
     * output directory and other properties.
     */
    @Parameter(defaultValue = "${project}", readonly = true)
    private MavenProject project;

    /**
     * Injected value of our classpath elements. This is used in conjunction with
     * the Reflections API in order to help identify where annotations appear on
     * methods.
     */
    @Parameter( defaultValue = "${project.compileClasspathElements}", readonly = true, required = true )
    private List<String> projectClasspathElements;

    /**
     * The path of the typescript file (relative to pom) which is generated by
     * this plugin
     */
    @Parameter( defaultValue = "src/main/resources/META-INF/resources/ts/javatypes.ts")
    private File generatedFile;
    
    
    /**
     * Scans the source code for this module to look for instances of the
     * annotations we're looking for.
     *
     * @throws MojoExecutionException thrown if there are errors during analysis
     */
    @Override
    public void execute() throws MojoExecutionException {

        try {
            // these paths should include our module's source root plus any generated code
            List<String> compileSourceOutputs = Collections.singletonList(project.getBuild().getOutputDirectory());
            URL[] sourceFiles = buildMavenClasspath(compileSourceOutputs);

            // the project classpath includes the source roots plus the transitive
            // dependencies according to our Mojo annotation's requiresDependencyResolution
            // attribute.
            URL[] projectClasspath = buildMavenClasspath(this.projectClasspathElements);

            URLClassLoader projectClassloader = new URLClassLoader(projectClasspath);
            // todo - is there any savings to be had here by caching the reflections but tweaking the filters for the given module?
            Reflections reflections = new Reflections(
                    new ConfigurationBuilder()
                            .setUrls(sourceFiles)
                            .addClassLoaders(projectClassloader)
                            .setScanners(new MethodAnnotationsScanner(), new TypeAnnotationsScanner(), new SubTypesScanner())
                    // todo here is where to filter the packages
//                    .filterInputsBy(new Predicate<String>() {
//                        @Override
//                        public boolean apply(String input) {
//                            return true;
//                        }
//                    })
            );
            getLog().info("Generating typescript to "+generatedFile.getAbsolutePath());
            Generator g = new Generator();
            g.generate(reflections, generatedFile);
            
        } catch (Throwable e) {
            throw new MojoExecutionException("A fatal error occurred while generating Typescript," +
                    " see stack trace for details.", e);
        }
    }

    /**
     * Extracted this method simply for unit testing.
     * @param classpathElements List of class path entries from the maven project
     * @return array of URL's for the classpath
     * @throws MojoExecutionException only thrown if we can't convert a classpath element to a URL which shouldn't happen
     */
    protected URL[] buildMavenClasspath(List<String> classpathElements) throws MojoExecutionException {
        List<URL> projectClasspathList = new ArrayList<>();
        for (String element : classpathElements) {
            try {
                projectClasspathList.add(new File(element).toURI().toURL());
            } catch (MalformedURLException e) {
                throw new MojoExecutionException(element + " is an invalid classpath element", e);
            }
        }

        return projectClasspathList.toArray(new URL[projectClasspathList.size()]);
    }
    
}
