/*
 * tksCommons
 * 
 * Author : Thomas Kuhlmann (ThK-Systems, http://www.thk-systems.de) License : LGPL (https://www.gnu.org/licenses/lgpl.html)
 */
package de.thksystems.util.text;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

/**
 * Just about parsing text ...
 */
public final class ParseUtils {

	public static final BigDecimal VALUE_BIGDECIMAL_MINUSONE = new BigDecimal("-1");

	private static final Pattern FILESIZE_PATTERN = Pattern.compile("^([0-9.]+)([ETGMK]B?)$", Pattern.CASE_INSENSITIVE);

	private static Map<String, Integer> fileSizePowMap = null;

	private static final Pattern DURATION_PATTERN = Pattern.compile("^([0-9.]+)(ms|s|m|h|d|w|M|y)?$");

	private static Map<String, Long> durationFactorMap = null;

	private ParseUtils() {
	}

	/**
	 * Parse filesize given as a {@link String} (e.g. '0.003EB', '2.3GB', '5M', '30', '705.23kB') and return the size in bytes as
	 * {@link BigDecimal}.
	 * 
	 * @param filesize
	 *            Size as {@link String} (KB, MB, GB, TB, EB are supported as suffixes, case-insensitive, the 'B' may be omitted; If no
	 *            suffix is given, the filesize is interpreted as bytes; Negative values are not supported.)
	 * @return size in bytes as {@link BigDecimal} (<code>null</code>, if a invalid 'filesize' is given.)
	 */
	public static BigDecimal parseFileSize(String filesize) {
		if (filesize == null || filesize.length() == 0) {
			return null;
		}
		Matcher matcher = FILESIZE_PATTERN.matcher(filesize);
		if (matcher.find()) {
			String number = matcher.group(1);
			int pow = getFileSizePowMap().get(matcher.group(2).toUpperCase());
			BigDecimal bytes = new BigDecimal(number);
			bytes = bytes.multiply(BigDecimal.valueOf(1024).pow(pow));
			return bytes;
		}
		try {
			BigDecimal value = new BigDecimal(filesize);
			if (value.compareTo(VALUE_BIGDECIMAL_MINUSONE) <= 0) {
				return null;
			} else {
				return value;
			}
		} catch (NumberFormatException e) {
			return null;
		}
	}

	/** Lazy get fileSizePowMap. */
	private static Map<String, Integer> getFileSizePowMap() {
		if (fileSizePowMap == null) {
			fileSizePowMap = new HashMap<>();
			fileSizePowMap.put("EB", 5);
			fileSizePowMap.put("TB", 4);
			fileSizePowMap.put("GB", 3);
			fileSizePowMap.put("MB", 2);
			fileSizePowMap.put("KB", 1);
			fileSizePowMap.put("E", 5);
			fileSizePowMap.put("T", 4);
			fileSizePowMap.put("G", 3);
			fileSizePowMap.put("M", 2);
			fileSizePowMap.put("K", 1);
		}
		return fileSizePowMap;
	}

	/**
	 * Parse duration.
	 * <p>
	 * Input-Pattern: [0-9]+(ms|s|m|h|d|w|M|y), e.g. "50ms", "133453m", "7d", "13M"
	 * <p>
	 * Units:
	 * <ul>
	 * <li>ms -> milliseconds</li>
	 * <li>s -> seconds</li>
	 * <li>m -> minutes</li>
	 * <li>h -> hours</li>
	 * <li>d -> days</li>
	 * <li>w -> weeks</li>
	 * <li>M -> months (30 days)</li>
	 * <li>y -> years (365 days)</li>
	 * </ul>
	 * 
	 * @return duration in milliseconds
	 */
	public static BigDecimal parseDuration(String durationString) {
		// TODO - Add multiple groups like "1h 5m 3s"
		if (StringUtils.isNotEmpty(durationString)) {
			Matcher matcher = DURATION_PATTERN.matcher(durationString);
			if (matcher.find()) {
				BigDecimal value = new BigDecimal(matcher.group(1));
				String unit = matcher.group(2);
				BigDecimal factor = new BigDecimal(getDurationFactorMap().get(unit));
				return value.multiply(factor);
			}
		}
		return null;
	}

	/** Lazy get durationFactorMap. */
	private static Map<String, Long> getDurationFactorMap() {
		if (durationFactorMap == null) {
			durationFactorMap = new HashMap<>();
			durationFactorMap.put(null, 1l);
			durationFactorMap.put("ms", 1l);
			durationFactorMap.put("s", 1000l);
			durationFactorMap.put("m", 1000l * 60l);
			durationFactorMap.put("h", 1000l * 60l * 60l);
			durationFactorMap.put("d", 1000l * 60l * 60l * 24l);
			durationFactorMap.put("w", 1000l * 60l * 60l * 24l * 7l);
			durationFactorMap.put("M", 1000l * 60l * 60l * 24l * 30l);
			durationFactorMap.put("y", 1000l * 60l * 60l * 24l * 365l);
		}
		return durationFactorMap;
	}

}