/*
 * tksCommons
 * 
 * Author : Thomas Kuhlmann (ThK-Systems, http://www.thk-systems.de) License : LGPL (https://www.gnu.org/licenses/lgpl.html)
 */
package de.thksystems.util.crypto;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.Hex;

/**
 * Tools to encrypt passwords.
 */
public final class PasswordUtils {

	public static final String DEFAULT_PASSWORD_HASHALGO = "SHA-256";

	private PasswordUtils() {
	}

	/**
	 * Creates a hash of the given password using the given hash algorithm.
	 * 
	 * @return Hex string of password hash.
	 * @throws NoSuchAlgorithmException
	 *             Hash algorithm is not available
	 */
	public static String createPasswordHash(String plainPassword, String hashAlgo) throws NoSuchAlgorithmException {
		if (plainPassword == null || plainPassword.length() == 0) {
			return null;
		}
		MessageDigest md = MessageDigest.getInstance(DEFAULT_PASSWORD_HASHALGO);
		byte[] digest = md.digest(plainPassword.getBytes());
		byte[] digestHex = Hex.encode(digest);
		return new String(digestHex);
	}

	/**
	 * Creates a hash of the given password using {@value #DEFAULT_PASSWORD_HASHALGO}.
	 * 
	 * @return Hex string of password hash.
	 */
	public static String createPasswordHash(String plainPassword) {
		try {
			return createPasswordHash(plainPassword, DEFAULT_PASSWORD_HASHALGO);
		} catch (NoSuchAlgorithmException e) {
			throw new CryptoRuntimeException(e); // Should not happen
		}
	}

	/**
	 * Encrypts given password using SHA-256 and encodes Base64 .
	 * 
	 * @Deprecated Use {@link #createPasswordHash(String)}
	 */
	public static String encryptAsPassword(String password) {
		if (password == null || password.length() == 0) {
			return null;
		}
		try {
			MessageDigest md = MessageDigest.getInstance(DEFAULT_PASSWORD_HASHALGO);
			byte[] digest = md.digest(password.getBytes());
			String encryptedPassword = new String(Base64.encode(digest));
			return encryptedPassword;
		} catch (NoSuchAlgorithmException e) {
			throw new CryptoRuntimeException(e);
		}
	}
}
