package de.svenkubiak.mangooio.mongodb;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bson.types.ObjectId;
import org.mongodb.morphia.Datastore;
import org.mongodb.morphia.Morphia;

import com.google.common.base.Preconditions;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoCredential;
import com.mongodb.ServerAddress;

import io.mangoo.core.Config;

/**
 * Convenient class for interacting with MongoDB and/or Morphia
 * in the mangoo IO framework
 *
 * @author svenkubiak
 *
 */
@Singleton
public class MongoDB {
    private static final Logger LOG = LogManager.getLogger(MongoDB.class);
    private static final int DEFAULT_MONGODB_PORT = 27017;
    private static final String DEFAULT_MORPHIA_PACKAGE = "MyMorphiaPackage";
    private static final String DEFAULT_MONGODB_NAME = "MyMongoDB";
    private static final String DEFAULT_MONGODB_HOST = "localhost";
    private static final String MONGODB_HOST = "mongodb.host";
    private static final String MONGODB_PORT = "mongodb.port";
    private static final String MONGODB_USER = "mongodb.user";
    private static final String MONGODB_PASS = "mongodb.pass";
    private static final String MONGODB_AUTH = "mongodb.auth";
    private static final String MONGODB_DBNAME = "mongodb.dbname";
    private static final String MONGODB_AUTHDB = "mongodb.authdb";
    private static final String MORPHIA_PACKAGE = "morphia.package";
    private static final String MORPHIA_INIT = "morphia.init";
    private Datastore datastore;
    private Morphia morphia;
    private MongoClient mongoClient;
    private Config config;

    @Inject
    public MongoDB(Config config) {
        this.config = config;

        connect();
        if (this.config.getBoolean(MORPHIA_INIT, false)) {
            morphify();
        }
    }

    public Datastore getDatastore() {
        return this.datastore;
    }

    public Morphia getMorphia() {
        return this.morphia;
    }

    public MongoClient getMongoClient() {
        return this.mongoClient;
    }

    private void connect() {
        String host = this.config.getString(MONGODB_HOST, DEFAULT_MONGODB_HOST);
        int port = this.config.getInt(MONGODB_PORT, DEFAULT_MONGODB_PORT);

        String username = this.config.getString(MONGODB_USER, null);
        String password = this.config.getString(MONGODB_PASS, null);
        String authdb = this.config.getString(MONGODB_AUTHDB);

        if (this.config.getBoolean(MONGODB_AUTH, false) && StringUtils.isNotBlank(username) && StringUtils.isNotBlank(password) && StringUtils.isNotBlank(authdb)) {
            MongoClientOptions mongoClientOptions = MongoClientOptions.builder().build();
            MongoCredential mongoCredential = MongoCredential.createScramSha1Credential(username, authdb, password.toCharArray());
            
            this.mongoClient = new MongoClient(new ServerAddress(host, port), mongoCredential, mongoClientOptions);
            LOG.info("Successfully created MongoClient @ {}:{} with authentication", host, port);
        } else {
            this.mongoClient = new MongoClient(host, port);
            LOG.info("Successfully created MongoClient @ {}:{} *without* authentication", host, port);
        }
    }

    private void morphify() {
        String packageName = this.config.getString(MORPHIA_PACKAGE, DEFAULT_MORPHIA_PACKAGE);
        String dbName = this.config.getString(MONGODB_DBNAME, DEFAULT_MONGODB_NAME);

        this.morphia = new Morphia().mapPackage(packageName);
        this.datastore = this.morphia.createDatastore(this.mongoClient, dbName);

        LOG.info("Mapped Morphia models of package '" + packageName + "' and created Morphia Datastore to database '" + dbName + "'");
    }

    /**
     * Ensures (creating if necessary) the indexes found during class mapping (using @Indexed, @Indexes), possibly in the background
     *
     * @param background True if background process, false otherwise
     */
    public void ensureIndexes(boolean background) {
        this.datastore.ensureIndexes(background);
    }

    /**
     * Ensure capped DBCollections for Entity(s)
     */
    public void ensureCaps() {
        this.datastore.ensureCaps();
    }

    /**
     * Retrieves a mapped Morphia object from MongoDB. If the id is not of
     * type ObjectId, it will be converted to ObjectId
     *
     * @param id The id of the object
     * @param clazz The mapped Morphia class
     * @param <T> JavaDoc requires this - please ignore
     *
     * @return The requested class from MongoDB or null if none found
     */
    public <T extends Object> T findById(Object id, Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to find an object by id, but given class is null");
        Preconditions.checkNotNull(id, "Tryed to find an object by id, but given id is null");

        return this.datastore.get(clazz, (id instanceof ObjectId) ? id : new ObjectId(String.valueOf(id)));
    }

    /**
     * Retrieves a list of mapped Morphia objects from MongoDB
     *
     * @param clazz The mapped Morphia class
     * @param <T> JavaDoc requires this - please ignore
     * 
     * @return A list of mapped Morphia objects or an empty list if none found
     */
    public <T extends Object> List<T> findAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to get all morphia objects of a given object, but given object is null");

        return this.datastore.find(clazz).asList();
    }

    /**
     * Counts all objected of a mapped Morphia class
     *
     * @param clazz The mapped Morphia class
     * @param <T> JavaDoc requires this - please ignore
     *      
     * @return The number of objects in MongoDB
     */
    public <T extends Object> long countAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to count all a morphia objects of a given object, but given object is null");

        return this.datastore.find(clazz).count();
    }

    /**
     * Saves a mapped Morphia object to MongoDB
     *
     * @param object The object to save
     */
    public void save(Object object) {
        Preconditions.checkNotNull(object, "Tryed to save a morphia object, but a given object is null");

        this.datastore.save(object);
    }

    /**
     * Deletes a mapped Morphia object in MongoDB
     *
     * @param object The object to delete
     */
    public void delete(Object object) {
        Preconditions.checkNotNull(object, "Tryed to delete a morphia object, but given object is null");

        this.datastore.delete(object);
    }

    /**
     * Deletes all mapped Morphia objects of a given class

     * @param <T> JavaDoc requires this - please ignore
     * @param clazz The mapped Morphia class
     */
    public <T extends Object> void deleteAll(Class<T> clazz) {
        Preconditions.checkNotNull(clazz, "Tryed to delete list of mapped morphia objects, but given class is null");

        this.datastore.delete(this.datastore.createQuery(clazz));
    }

    /**
     * Drops all data in MongoDB on the connected database
     */
    public void dropDatabase() {
        this.datastore.getDB().dropDatabase();
    }
}