package de.sstoehr.imageoptimizermavenplugin.jobs;

import de.sstoehr.imageoptimizermavenplugin.optimizers.OptimizerResult;
import net.oneandone.sushi.fs.Node;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

public class JobRunner {

    private static final Logger LOG = LoggerFactory.getLogger(JobRunner.class);

    private static final ThreadLocal<DecimalFormat> PERCENTAGE_FORMAT = new ThreadLocal<DecimalFormat>() {

        @Override
        protected DecimalFormat initialValue() {
            return new DecimalFormat("0.00%");
        }

    };

    private List<Job> jobs = new ArrayList<>();
    private Node base;

    public JobRunner(Node base) {
        this.base = base;
    }

    public void schedule(Job job) {
        jobs.add(job);
    }

    public boolean run() {
        if (jobs.size() > 0) {
            LOG.info("Processing {} job(s).", jobs.size());

            ExecutorService executorService = Executors.newCachedThreadPool();

            try {
                List<Future<JobResult>> results = executorService.invokeAll(jobs);

                for (Future<JobResult> future : results) {
                    JobResult result = future.get();

                    LOG.info("=== {} ===", result.getName());


                    for (OptimizerResult optimizerResult : result.getResults()) {
                        long saved = optimizerResult.getPreSize() - optimizerResult.getPostSize();
                        double percentage = (double) saved / optimizerResult.getPreSize();

                        if (saved == 0) {
                            continue;
                        }

                        String path = optimizerResult.getPath();
                        path = path.replaceFirst(base.getPath(), "");

                        String savedReadable = FileUtils.byteCountToDisplaySize(saved);
                        String percentageReadable = PERCENTAGE_FORMAT.get().format(percentage);
                        LOG.info("{} - {} ({})", path, savedReadable, percentageReadable);
                    }

                }

            } catch (Exception e) {
                LOG.error("Error processing images", e);
            }

            executorService.shutdown();

            return true;

        } else {
            LOG.info("No jobs configured.");
        }

        return false;
    }
}
