/* © SRSoftware 2025 */
package de.srsoftware.document.zugferd.data;

import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;

/**
 * Datensatz für die Dokumenten-Erzeugung
 * @param documentNumber Kennung des Dokuments
 * @param currency Währung
 * @param typeCode Art des Dokuments
 * @param date Datum des Dokuments
 * @param author Aussteller des Dokuments
 * @param customer Empfänger des Dokuments
 * @param notes Anmerkungen
 * @param deliveryDate Lieferdatum
 * @param dueDate Fälligkeitsdatum
 * @param lineItems Positionen
 * @param paymentTerms Zahlungsbedingungen
 */
public record DocumentData(
		String documentNumber,
		Currency currency,
		TypeCode typeCode,
		LocalDate date,
		Author author,
		Customer customer,
		List<String> notes,
		LocalDate deliveryDate,
		LocalDate dueDate,
		String paymentTerms,
		List<LineItem> lineItems) {

	/**
	 * Summe der unversteuerten Beträge
	 * @return Summe der unversteuerten Beträge
	 */
	public long taxBasisTotalAmount() {
		var sum = 0L;
		for (var taxSet : taxSets()) sum += taxSet.basisAmount();
		return sum;
	}

	/**
	 * Summe der einzelnen Steuerbeträge
	 * @return Summe der einzelnen Steuerbeträge
	 */
	public long taxTotalAmount() {
		var sum = 0L;
		for (var taxSet : taxSets()) sum += taxSet.calculatedAmount();
		return sum;
	}

	/**
	 * Gsamtsumme
	 * @return Summe aller unversteuerten Beträge + Summe aller Steuer-Beträge
	 */
	public long grandTotalAmount() {
		return taxBasisTotalAmount()+taxTotalAmount();
	}

	/**
	 * Zahlungspflichtiger Betrag
	 * @return Zahlungspflichtiger Betrag
	 */
	public long duePayableAmount() {
		return grandTotalAmount();
	}

	/**
	 * Zuschläge
	 * @return 0
	 */
	public long chargeTotalAmount() {
		return 0;
	}

	/**
	 * Liste der Steuer-Daten
	 * @return Liste der Steuer-Daten
	 */
	public List<CombinedTaxSet> taxSets(){
		var map = new HashMap<String,CombinedTaxSet>();
		for (var lineItem : lineItems()){
			var lineTaxSet = lineItem.taxSet();
			var combined = map.computeIfAbsent(lineTaxSet.key(), k -> new CombinedTaxSet(0, lineTaxSet.percent(), lineTaxSet.type()));
			combined.add(lineItem.totalAmount());
		}
		return List.copyOf(map.values());
	}
}
