/* © SRSoftware 2025 */
package de.srsoftware.document.zugferd;

import static de.srsoftware.document.mustang.Constants.*;
import static de.srsoftware.document.mustang.Constants.KEY_XML;
import static de.srsoftware.tools.MimeType.MIME_PDF;
import static java.lang.System.Logger.Level.ERROR;

import de.srsoftware.document.api.*;
import de.srsoftware.document.mustang.A3Exporter;
import de.srsoftware.document.processor.FileProcessor;
import de.srsoftware.document.processor.weasyprint.WeasyDoc;
import de.srsoftware.document.zugferd.data.DocumentData;
import java.io.IOException;
import java.util.*;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.xmpbox.xml.DomXmpParser;
import org.apache.xmpbox.xml.XmpParsingException;

/**
 * Document type, that allows embedding of a Factur-X-XML file within an Zugferd-PDF file
 */
public class Zugferd implements Document {
	private static final System.Logger LOG = System.getLogger(Zugferd.class.getSimpleName());

	private final DocumentRegistry registry;

	/**
	 * Create a new Instance
	 * @param registry the Document registry, which is used to locate required documents to build the combined PDF
	 */
	public Zugferd(DocumentRegistry registry) {
		this.registry = registry;
	}

	private byte[] combine(Content pdf, Content xml, String profile, String creator, String producer, DocumentData docData) throws IOException {
		var doc = Loader.loadPDF(pdf.bytes());
		return getExporter(doc)
				.setAuthor(docData.author().name())
				.setTitle(docData.documentNumber())
				.setProducer(producer)
				.setCreator(creator)
				.setProfile(profile)
				.setXML(xml.bytes())
				.export();

	}

	@Override
	public String description() {
		return "Zugferd-PDF mit eingebettetem XML";
	}

	@Override
	public String id() {
		return "Zugferd";
	}

	private A3Exporter getExporter(PDDocument doc) throws IOException {
		return switch (getPDFAVersion(doc)) {
			case 1 -> {
				//return new A1Exporter();
				throw notImplemented();
				//return new A1Exporter();
			}
			case 3 -> new A3Exporter(doc);
			default -> throw new IllegalArgumentException("PDF-A version not supported");
		};
	}

	private int getPDFAVersion(PDDocument document) throws IOException {
		var metadata = document.getDocumentCatalog().getMetadata();
		// the PDF version we could get through the document but we want the PDF-A version,
		// which is different (and can probably base on different PDF versions)
		if (metadata != null) {
			try {
				var pdfaSchema = new DomXmpParser().parse(metadata.createInputStream()).getPDFAIdentificationSchema();
				if (pdfaSchema != null) return pdfaSchema.getPart();
			} catch (XmpParsingException e) {
				LOG.log(ERROR,"XmpParsingException", e);
			}
		}
		return 0;
	}

	@Override
	public String mimeType() {
		return MIME_PDF;
	}

	@Override
	public String name() {
		return "zugferd.pdf";
	}

	private static RuntimeException notImplemented() {
		return new RuntimeException("not implemented");
	}

	/**
	 * create a new zugferd PDF
	 * @param data a map with the following structure:
	 *             {
	 *             		xml : {
	 *             			profile : «String»,
	 *             			data: «DocumentData»
	 *             		},
	 *             		pdf : {
	 *             			template : «document id»,
	 *                      data: «Object/what ever is needed to render the pdf»
	 *             		},
	 *             		producer : «String»
	 *             }
	 * @return a RenderResult
	 */
	@Override
	public RenderResult render(Map<String, Object> data) {
		if (!(data.get(KEY_XML) instanceof Map<?,?> xmlConfig)) return RenderError.of(ERR_KEY_MISSING,KEY_XML);
		if (!(data.get(KEY_PDF) instanceof Map<?,?> pdfConfig)) return RenderError.of(ERR_KEY_MISSING,KEY_PDF);
		if (!(data.get(KEY_PRODUCER) instanceof String producer)) return RenderError.of(ERR_KEY_MISSING,KEY_PRODUCER);

		if (!(xmlConfig.get(KEY_PROFILE) instanceof String profile)) return RenderError.of(ERR_KEY_MISSING,KEY_XML+"."+KEY_PROFILE);
		if (!(xmlConfig.get(KEY_DATA) instanceof DocumentData docData)) return RenderError.of(ERR_KEY_MISSING,KEY_XML+"."+KEY_DATA);

		if (!(pdfConfig.get(KEY_TEMPLATE) instanceof String templateName)) return RenderError.of(ERR_KEY_MISSING,KEY_PDF+"."+KEY_TEMPLATE);
		@SuppressWarnings("unchecked") Map<String,Object> pdfData = pdfConfig.get(KEY_DATA) instanceof @SuppressWarnings("rawtypes") Map map ? map : null;

		var xmlId = "Factur-X@"+profile;
		var xmlDoc = registry.documents().filter(doc -> doc.id().equals(xmlId)).findAny();
		if (xmlDoc.isEmpty()) return RenderError.of(ERR_DOCUMENT_NOT_FOUND,xmlId);

		var pdfDoc = registry.documents().filter(doc -> doc.name().equals(templateName)).findAny();
		if (pdfDoc.isEmpty()) return RenderError.of(ERR_DOCUMENT_NOT_FOUND,templateName);

		if (pdfDoc.get() instanceof WeasyDoc && pdfData != null) pdfData.put("additional_args",List.of("--pdf-variant","pdf/a-3b"));

		// Precursor rendering
		var xmlResult = xmlDoc.get().render(Map.of(KEY_DATA,docData));
		if (!(xmlResult instanceof Content xml)) return xmlResult;

		// Precursor rendering
		var pdfResult = pdfDoc.get().render(pdfData);
		if (!(pdfResult instanceof Content pdf)) return pdfResult;

		// combining
		try {
			var creator = pdfDoc.get() instanceof FileProcessor fileProcessor ? fileProcessor.software() : "SRSoftware Document Service";
			return new BytesContent(combine(pdf,xml,profile,creator,producer,docData));
		} catch (IOException e) {
			return RenderError.of(e.getMessage());
		}
	}
}
