/* © SRSoftware 2025 */
package de.srsoftware.document.mustang;




import static de.srsoftware.document.mustang.Constants.*;
import static org.apache.xmpbox.XmpConstants.PARSE_TYPE;
import static org.apache.xmpbox.XmpConstants.RESOURCE_NAME;
import static org.apache.xmpbox.type.PDFAPropertyType.*;

import org.apache.xmpbox.XMPMetadata;
import org.apache.xmpbox.XmpConstants;
import org.apache.xmpbox.schema.PDFAExtensionSchema;
import org.apache.xmpbox.type.ArrayProperty;
import org.apache.xmpbox.type.Attribute;
import org.apache.xmpbox.type.Cardinality;
import org.apache.xmpbox.type.ChoiceType;
import org.apache.xmpbox.type.DefinedStructuredType;
import org.apache.xmpbox.type.PDFASchemaType;
import org.apache.xmpbox.type.StructuredType;
import org.apache.xmpbox.type.TextType;


/**
 * Code adapted from Mustangproject's ZUGFeRD implementation
 * ZUGFeRD exporter helper class
 * Licensed under the APLv2

 * Additionally to adding a RDF namespace with an indication which file
 * attachment if Zugferd, this namespace has to be described in a PDFA Extension
 * Schema. I know there is a PDFAExtensionSchema in the context of PDFBox'
 * XMPBOX but I have been using PDFBox' JempBOX so far because I could not find
 * out how to write XMPBOX XMPMetadata to a PDF file. So this is my version of
 * PDFAExtensionSchema for PDFBox' jempbox XMPMetadata
 *
 * @author jstaerk, srichter
 */

@StructuredType(preferedPrefix = "pdfaExtension", namespace = "http://www.aiim.org/pdfa/ns/extension/")
public class XMPSchemaPDFAExtensions extends PDFAExtensionSchema {

	/**
	 * create new instance
	 * @param metadata the metadata add the extensions to
	 */
	public XMPSchemaPDFAExtensions(XMPMetadata metadata) {
		super(metadata);
		attachExtensions(metadata);
	}

	private void addProperty(ArrayProperty parent, String name, String description) {
		XMPMetadata metadata = getMetadata();

		DefinedStructuredType li = new DefinedStructuredType(metadata, getNamespace(), getPrefix(), XmpConstants.LIST_NAME);
		li.setAttribute(new Attribute(getNamespace(), PARSE_TYPE, RESOURCE_NAME));

		ChoiceType pdfa2 = new ChoiceType(metadata, XMLNS_PDFA_PROPERTY, PREFIX_PDFA_PROPERTY, NAME, name);
		li.addProperty(pdfa2);

		pdfa2 = new ChoiceType(metadata, XMLNS_PDFA_PROPERTY, PREFIX_PDFA_PROPERTY, VALUETYPE, "Text");
		li.addProperty(pdfa2);

		pdfa2 = new ChoiceType(metadata, XMLNS_PDFA_PROPERTY, PREFIX_PDFA_PROPERTY, CATEGORY, "external");
		li.addProperty(pdfa2);

		pdfa2 = new ChoiceType(metadata, XMLNS_PDFA_PROPERTY, PREFIX_PDFA_PROPERTY, DESCRIPTION, description);
		li.addProperty(pdfa2);

		parent.addProperty(li);
	}

	private void attachExtensions(XMPMetadata metadata) {
		addNamespace(XMLNS_PDFA_SCHEMA, PREFIX_PDFA_SCHEMA);
		addNamespace(XMLNS_PDFA_PROPERTY, PREFIX_PDFA_PROPERTY);

		ArrayProperty newBag = createArrayProperty(SCHEMAS, Cardinality.Bag);
		DefinedStructuredType li = new DefinedStructuredType(metadata, getNamespace(), getPrefix(), XmpConstants.LIST_NAME);
		li.setAttribute(new Attribute(getNamespace(), PARSE_TYPE, RESOURCE_NAME));
		newBag.addProperty(li);
		addProperty(newBag);
		{ // attach zugferd headers
			TextType pdfa1 = new TextType(metadata, XMLNS_PDFA_SCHEMA, PREFIX_PDFA_SCHEMA, PDFASchemaType.SCHEMA, "Factur-X PDFA Extension Schema");
			li.addProperty(pdfa1);

			pdfa1 = new TextType(metadata, XMLNS_PDFA_SCHEMA, PREFIX_PDFA_SCHEMA, PDFASchemaType.NAMESPACE_URI, NAMESPACE_FACTUR_X);
			li.addProperty(pdfa1);

			pdfa1 = new TextType(metadata, XMLNS_PDFA_SCHEMA, PREFIX_PDFA_SCHEMA, PDFASchemaType.PREFIX, PREFIX_FACTUR_X);
			li.addProperty(pdfa1);

			ArrayProperty newSeq = new ArrayProperty(metadata, XMLNS_PDFA_SCHEMA, PREFIX_PDFA_SCHEMA, PDFASchemaType.PROPERTY, Cardinality.Seq);
			li.addProperty(newSeq);

			// As of late 2022 the Delivery-X standard is not yet published. See specification:
			// Die digitale Ablösung des Papier-Lieferscheins, Version 1.1, April 2022
			// Chapter 7.1 XMP-Erweiterungsschema für PDF/A-3
			// http://docplayer.org/230301085-Der-digitale-lieferschein-dls.html
			addProperty(newSeq, DOC_FILENAME, "name of the embedded XML invoice file");
			addProperty(newSeq, DOC_TYPE, INVOICE);
			addProperty(newSeq, VERSION, "The actual version of the ZUGFeRD XML schema");
			addProperty(newSeq, CONFORMANCE_LEVEL, "The selected ZUGFeRD profile completeness");
		}
	}
}
