/* © SRSoftware 2025 */
package de.srsoftware.document.mustang;

import static de.srsoftware.document.mustang.Constants.*;
import static de.srsoftware.tools.Optionals.isSet;
import static java.lang.Thread.currentThread;
import static java.nio.charset.StandardCharsets.UTF_8;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.*;
import javax.xml.transform.TransformerException;
import org.apache.pdfbox.cos.*;
import org.apache.pdfbox.pdfwriter.compress.CompressParameters;
import org.apache.pdfbox.pdmodel.*;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.common.PDMetadata;
import org.apache.pdfbox.pdmodel.common.filespecification.PDComplexFileSpecification;
import org.apache.pdfbox.pdmodel.common.filespecification.PDEmbeddedFile;
import org.apache.pdfbox.pdmodel.documentinterchange.logicalstructure.PDMarkInfo;
import org.apache.pdfbox.pdmodel.documentinterchange.logicalstructure.PDStructureTreeRoot;
import org.apache.pdfbox.pdmodel.font.PDCIDFontType2;
import org.apache.pdfbox.pdmodel.font.PDType0Font;
import org.apache.pdfbox.pdmodel.graphics.color.PDOutputIntent;
import org.apache.xmpbox.XMPMetadata;
import org.apache.xmpbox.type.BadFieldValueException;
import org.apache.xmpbox.xml.DomXmpParser;
import org.apache.xmpbox.xml.XmpParsingException;
import org.apache.xmpbox.xml.XmpSerializer;

/**
 * Exporter for PDF-A3. Code adapted from mustangproject.
 * ZUGFeRD exporter helper class
 * Licensed under the APLv2
 * @author jstaerk, srichter
 **/
public class A3Exporter {
	private String     creator = DEFAULT_PRODUCER;
	private final PDDocument doc;
	private boolean    documentPrepared = false;
	private boolean    fileAttached = false;
	private String     profile = null;
	private String     producer = DEFAULT_PRODUCER;
	private String     author, subject, title;
	private byte[]     xml;

	/**
	 * Create a new instance.
	 * @param doc the PDF document to process
	 */
	public A3Exporter(PDDocument doc){
		this.doc = doc;
	}

	private void addSrgbOutputIntend() throws IOException {
		if (!doc.getDocumentCatalog().getOutputIntents().isEmpty()) return;

		var colorProfile = currentThread().getContextClassLoader().getResourceAsStream(INTENT_PROFILE_KEY);
		if (colorProfile != null) {
			var intent = new PDOutputIntent(doc, colorProfile);
			intent.setInfo(INTENT_INFO);
			intent.setOutputCondition(INTENT_OUTPUT_CONDITION);
			intent.setOutputConditionIdentifier(INTENT_OUTPUT_CONDITION_IDENTIFIER);
			intent.setRegistryName(INTENT_REGISTRY);
			doc.getDocumentCatalog().addOutputIntent(intent);
		}
	}

	private void addStructureTreeRoot() {
		var catalog = doc.getDocumentCatalog();
		if (catalog.getStructureTreeRoot() == null) catalog.setStructureTreeRoot(new PDStructureTreeRoot());
	}

	private void addXMP(XMPMetadata metadata) {
		metadata.addSchema(new ZugferdSchema(metadata, getXMPName(profile), getFilenameForVersion(profile)));
		metadata.addSchema(new XMPSchemaPDFAExtensions(metadata));
	}

	/**
	 * close the underlying document
	 * @throws IOException if closing fails
	 */
	public void close() throws IOException {
		if (doc != null) doc.close();
	}

	/**
	 * export the processed document
	 * @return the byte data for the document
	 * @throws IOException on precessing errors
	 */
	public byte[] export() throws IOException {
		if (!documentPrepared) prepareDocument();
		if (!fileAttached) throw new IOException("File must be attached (usually with setTransaction) before performing this operation");
		var bos = new ByteArrayOutputStream();
		doc.save(bos, CompressParameters.NO_COMPRESSION);
		close();
		return bos.toByteArray();
	}

	private String getFilenameForVersion(String profile) {
		if (profile.equals(PROFILE_XRECHNUNG)) return FILENAME_XRECHNUNG;
		return FILENAME_FACTUR_X;
	}

	private XMPMetadata getXmpMetadata() throws IOException {
		var meta = doc.getDocumentCatalog().getMetadata();
		if (meta != null && meta.getLength() > 0) try {
			return new DomXmpParser().parse(meta.toByteArray());
		} catch (XmpParsingException e) {
			throw new IOException(e);
		}
		return XMPMetadata.createXMPMetadata();
	}

	private String getXMPName(String profile) {
		return switch (profile){
			case PROFILE_BASIC_WL -> XMP_NAME_BASIC_WL;
			case PROFILE_EN16931 -> XMP_NAME_EN16931;
			default -> profile;
		};
	}

	private void pdfAttachGenericFile(PDDocument doc, String filename, String relationship, byte[] data) throws IOException {
		fileAttached = true;

		var fileSpec = new PDComplexFileSpecification();
		fileSpec.setFile(filename);

		var dict = fileSpec.getCOSObject();
		dict.setName("AFRelationship", relationship);
		dict.setString("UF", filename);
		dict.setString("Desc", XML_DESCRIPTION);

		var embeddedFile = new PDEmbeddedFile(doc, new ByteArrayInputStream(data));
		embeddedFile.setSubtype(MIME_XML);
		embeddedFile.setSize(data.length);
		embeddedFile.setCreationDate(new GregorianCalendar());

		embeddedFile.setModDate(Calendar.getInstance());

		fileSpec.setEmbeddedFile(embeddedFile);

		// In addition make sure the embedded file is set under /UF
		dict = fileSpec.getCOSObject();
		var efDict = (COSDictionary) dict.getDictionaryObject(COSName.EF);
		efDict.setItem(COSName.UF, efDict.getItem(COSName.F));

		// now add the entry to the embedded file tree and set in the document.
		var names = new PDDocumentNameDictionary(doc.getDocumentCatalog());
		var embeddedFiles = names.getEmbeddedFiles();
		if (embeddedFiles == null) embeddedFiles = new PDEmbeddedFilesNameTreeNode();

		var namesMap = new HashMap<String, PDComplexFileSpecification>();

		var oldNamesMap = embeddedFiles.getNames();
		if (oldNamesMap != null) {
			for (String key : oldNamesMap.keySet()) namesMap.put(key, oldNamesMap.get(key));
		}
		namesMap.put(filename, fileSpec);
		embeddedFiles.setNames(namesMap);

		names.setEmbeddedFiles(embeddedFiles);
		doc.getDocumentCatalog().setNames(names);

		// AF entry (Array) in catalog with the FileSpec
		switch (doc.getDocumentCatalog().getCOSObject().getItem("AF")) {
			case null -> {
				var cosArray = new COSArray();
				cosArray.add(fileSpec);
				doc.getDocumentCatalog().getCOSObject().setItem("AF", cosArray);
			}
			case COSArray cosArray -> {
				cosArray.add(fileSpec);
				doc.getDocumentCatalog().getCOSObject().setItem("AF", cosArray);
			}
			case COSObject cosObject when cosObject.getObject() instanceof COSArray cosArray -> cosArray.add(fileSpec);
			default -> throw new IOException("Unexpected object type for PDFDocument/Catalog/COSDictionary/Item(AF)");
		}
	}

	private void prepare() throws IOException {
		prepareDocument();
		// ZUGFeRD 2.1.1 Technical Supplement | Part A | 2.2.2. Data Relationship
		// See documentation ZUGFeRD211_EN/Documentation/ZUGFeRD-2.1.1 - Specification_TA_Part-A.pdf
		// https://www.ferd-net.de/standards/zugferd-2.1.1/index.html
		var relationship = Set.of("MINIMUM","BASICWL").contains(profile.toUpperCase()) ? "Data" : "Alternative";

		pdfAttachGenericFile(doc, getFilenameForVersion(profile), relationship, xml);
	}

	private void prepareDocument() throws IOException {
		var cat = doc.getDocumentCatalog();
		var metadata = new PDMetadata(doc);
		cat.setMetadata(metadata);
		removeCidSet(doc);
		var xmp = getXmpMetadata();
		writeAdobePDFSchema(xmp);
		writePDFAIdentificationSchema(xmp);
		writeDublinCoreSchema(xmp);
		writeXMLBasicSchema(xmp);
		writeDocumentInformation();
		addSrgbOutputIntend();
		setMarked();
		addStructureTreeRoot();
		addXMP(xmp);
		try {
			metadata.importXMPMetadata(serializeXmpMetadata(xmp));
		} catch (TransformerException e) {
			throw new RuntimeException("Could not export XmpMetadata", e);
		}
		documentPrepared = true;
	}

	private void removeCidSet(PDDocument doc) throws IOException {
		// https://github.com/ZUGFeRD/mustangproject/issues/249

		var cidSet = COSName.getPDFName("CIDSet");
		var resources = COSName.getPDFName("Resources");

		// iterate over all pdf pages

		for (var object : doc.getPages()) {
			if (object instanceof PDPage page) {

				var res = page.getResources();

				// Check for fonts in PDXObjects:
				for (var xObjectName : res.getXObjectNames()) {
					var dictionary = res.getXObject(xObjectName).getCOSObject().getCOSDictionary(resources);
					if (dictionary != null) removeCIDSetFromPDResources(cidSet, new PDResources(dictionary));
				}

				// Check for fonts in document-resources:
				removeCIDSetFromPDResources(cidSet, res);
			}
		}
	}

	private void removeCIDSetFromPDResources(COSName cidSet, PDResources res) throws IOException {
		for (var fontName : res.getFontNames()) {
			var pdFont = res.getFont(fontName);
			if (pdFont instanceof PDType0Font typedFont && typedFont.getDescendantFont() instanceof PDCIDFontType2) pdFont.getFontDescriptor().getCOSObject().removeItem(cidSet);
		}
	}

	private byte[] serializeXmpMetadata(XMPMetadata xmpMetadata) throws TransformerException {
		var buffer = new ByteArrayOutputStream();
		new XmpSerializer().serialize(xmpMetadata, buffer, true); // see https://github.com/ZUGFeRD/mustangproject/issues/44
		return buffer.toByteArray();
	}

	/**
	 * set the author data which will go into the PDF metadata
	 * @param newValue new value for author
	 * @return this Exporter instance
	 */
	public A3Exporter setAuthor(String newValue){
		author = newValue;
		return this;
	}

	/**
	 * set the creator data which will go into the PDF metadata
	 * @param newValue new value for creator
	 * @return this Exporter instance
	 */
	public A3Exporter setCreator(String newValue) {
		creator = newValue;
		return this;
	}

	private void setMarked() {
		var catalog = doc.getDocumentCatalog();
		if (catalog.getMarkInfo() == null) catalog.setMarkInfo(new PDMarkInfo(doc.getPages().getCOSObject()));
		catalog.getMarkInfo().setMarked(true);
	}

	/**
	 * set the producer data which will go into the PDF metadata
	 * @param newValue new value for producer
	 * @return this Exporter instance
	 */
	public A3Exporter setProducer(String newValue) {
		producer = newValue;
		return this;
	}

	/**
	 * set the profile data which will go into the PDF metadata
	 * @param newValue new value for profile
	 * @return this Exporter instance
	 */
	public A3Exporter setProfile(String newValue) {
		profile = newValue;
		return this;
	}

	/**
	 * set the title data which will go into the PDF metadata
	 * @param newValue new value for title
	 * @return this Exporter instance
	 */
	public A3Exporter setTitle(String newValue){
		title = newValue;
		return this;
	}

	/**
	 * set the zugferd data, that will be included in the pdf
	 * @param zugferdData xml data
	 * @return this exporter instance
	 * @throws IOException if any errors occur during processing
	 */
	public A3Exporter setXML(byte[] zugferdData) throws IOException {
		var zf = new String(zugferdData, UTF_8);
		/*
		 * rsm:CrossIndustry is ZF/FX/XR (CII 2016b),rsm:SCRDMCCBDACIOMessageStructure is Order-X (CIO 2021) and
		 *  SCRDMCCBDACIDAMessageStructure is Despatch Advice
		 */
		if ((!zf.contains("rsm:CrossIndustry")) && (!zf.contains("rsm:SCRDMCCBDACIOMessageStructure")) && (!zf.contains("SCRDMCCBDACIDAMessageStructure"))) {
			throw new RuntimeException("ZUGFeRD XML does not contain <rsm:CrossIndustry, <rsm:SCRDMCCBDACIOMessageStructure or SCRDMCCBDACIDAMessageStructure and can thus not be valid");
		}
		xml = zugferdData;
		prepare();
		return this;
	}

	private void writePDFAIdentificationSchema(XMPMetadata xmp) {
		try {
			var pdfaid = xmp.getPDFAIdentificationSchema();
			if (pdfaid != null) xmp.removeSchema(pdfaid);
			pdfaid = xmp.createAndAddPDFAIdentificationSchema();
			pdfaid.setConformance(CONFORMANCE_LEVEL_UNICODE);
			pdfaid.setPart(3);
		} catch (BadFieldValueException ex) {
			// This should be impossible, because it would occur only if an illegal
			// conformance level is supplied,
			// however the enum enforces that the conformance level is valid.
			throw new Error(ex);
		}
	}

	private void writeDocumentInformation() {
		var info = doc.getDocumentInformation();
		info.setCreationDate(Calendar.getInstance());
		info.setModificationDate(Calendar.getInstance());
		info.setAuthor(author);
		info.setProducer(producer);
		info.setCreator(creator);
		info.setTitle(title);
		info.setSubject(subject);
	}

	private void writeXMLBasicSchema(XMPMetadata xmp) {
		var xsb = xmp.getXMPBasicSchema();
		if (xsb != null) xmp.removeSchema(xsb);
		xsb = xmp.createAndAddXMPBasicSchema();
		xsb.setCreatorTool(creator);
		xsb.setCreateDate(Calendar.getInstance());
	}

	private void writeDublinCoreSchema(XMPMetadata xmp) {
		var dcSchema = xmp.getDublinCoreSchema();
		if (dcSchema != null)	xmp.removeSchema(dcSchema);
		dcSchema = xmp.createAndAddDublinCoreSchema();
		if (dcSchema.getFormat() == null) dcSchema.setFormat("application/pdf");
		if (creator != null) {
			dcSchema.addCreator(creator);
			dcSchema.addDate(Calendar.getInstance());
		}

		var titleProperty = dcSchema.getTitleProperty();
		if (titleProperty != null) {
			if (isSet(title)) {
				dcSchema.removeProperty(titleProperty);
				dcSchema.setTitle(title);
			} else if (titleProperty.getElementsAsString().stream().anyMatch("Untitled"::equalsIgnoreCase)) {
				// remove unfitting ghostscript default
				dcSchema.removeProperty(titleProperty);
			}
		} else if (isSet(title)) {
			dcSchema.setTitle(title);
		}
	}

	private void writeAdobePDFSchema(XMPMetadata xmp) {
		var pdf = xmp.getAdobePDFSchema();
		if (pdf != null) xmp.removeSchema(pdf);
		pdf = xmp.createAndAddAdobePDFSchema();
		pdf.setProducer(producer);
	}
}