/*
 * Decompiled with CFR 0.152.
 */
package de.sonallux.spotify.parser;

import de.sonallux.spotify.core.model.SpotifyWebApiCategory;
import de.sonallux.spotify.core.model.SpotifyWebApiEndpoint;
import de.sonallux.spotify.parser.ApiEndpointFixes;
import de.sonallux.spotify.parser.ApiParseException;
import de.sonallux.spotify.parser.Html2Markdown;
import de.sonallux.spotify.parser.ParseUtils;
import de.sonallux.spotify.parser.ResponseTypeMapper;
import java.io.IOException;
import java.nio.file.Path;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.Node;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ApiEndpointParser {
    private static final Logger log = LoggerFactory.getLogger(ApiEndpointParser.class);
    private final boolean isInteractive;
    private String documentationUrl;
    private String endpointUrl;
    private ResponseTypeMapper responseTypeMapper;

    SortedMap<String, SpotifyWebApiCategory> parseSpotifyApiCategories(List<Elements> sections, String documentationUrl, String endpointUrl, Path responseTypesFile) throws ApiParseException {
        this.documentationUrl = documentationUrl;
        this.endpointUrl = endpointUrl;
        try {
            this.responseTypeMapper = new ResponseTypeMapper(responseTypesFile);
        }
        catch (IOException | NoSuchAlgorithmException e) {
            throw new ApiParseException("Failed to initialize response type mapper", e);
        }
        TreeMap<String, SpotifyWebApiCategory> categories = new TreeMap<String, SpotifyWebApiCategory>();
        for (Elements element : sections.subList(1, sections.size() - 1)) {
            SpotifyWebApiCategory category = this.parseSpotifyApiCategory(element);
            if (categories.containsKey(category.getId())) {
                throw new ApiParseException("Category " + category.getId() + " is defined twice");
            }
            categories.put(category.getId(), category);
        }
        this.addResponseTypes(categories);
        ApiEndpointFixes.fixApiEndpoints(categories);
        return categories;
    }

    private void addResponseTypes(SortedMap<String, SpotifyWebApiCategory> categories) {
        try {
            if (this.isInteractive) {
                this.responseTypeMapper.update(new ArrayList<SpotifyWebApiCategory>(categories.values()));
            }
            for (SpotifyWebApiCategory category : categories.values()) {
                for (SpotifyWebApiEndpoint endpoint : category.getEndpointList()) {
                    List<SpotifyWebApiEndpoint.ResponseType> responseTypes = this.responseTypeMapper.getEndpointResponseTypes(category.getId(), endpoint);
                    if (responseTypes == null || responseTypes.isEmpty()) {
                        log.warn("Missing response type in {} for {} {} with response: \n{}\n", new Object[]{category.getId(), endpoint.getHttpMethod(), endpoint.getId(), endpoint.getResponseDescription()});
                        continue;
                    }
                    endpoint.setResponseTypes(responseTypes);
                }
            }
            this.responseTypeMapper.save();
        }
        catch (IOException e) {
            log.error("Failed to load missing response types", (Throwable)e);
        }
    }

    private SpotifyWebApiCategory parseSpotifyApiCategory(Elements elements) throws ApiParseException {
        Element header = elements.first();
        String id = header.attributes().get("id");
        String link = this.documentationUrl + "/#" + id;
        String name = header.text();
        TreeMap<String, SpotifyWebApiEndpoint> endpoints = new TreeMap<String, SpotifyWebApiEndpoint>();
        for (Elements element : ParseUtils.splitAt(elements, "h2")) {
            SpotifyWebApiEndpoint endpoint = this.parseSpotifyApiEndpoint(element);
            if (endpoints.containsKey(endpoint.getId())) {
                throw new ApiParseException("Endpoint " + endpoint.getId() + " already defined in category " + id);
            }
            endpoints.put(endpoint.getId(), endpoint);
        }
        return new SpotifyWebApiCategory(id, name, link, endpoints);
    }

    private SpotifyWebApiEndpoint parseSpotifyApiEndpoint(Elements elements) throws ApiParseException {
        String httpMethod;
        String[] parts;
        Element header = elements.first();
        String id = header.attributes().get("id");
        String link = this.documentationUrl + "/#" + id;
        String name = header.text();
        String description = Html2Markdown.convert((Node)elements.select("p").first());
        Elements codeBlocks = elements.select(".hidden-xs .highlighter-rouge code");
        if (codeBlocks.size() == 1) {
            parts = ((Element)codeBlocks.get(0)).text().split(" ");
            httpMethod = parts[0].toUpperCase();
            if (!parts[1].startsWith(this.endpointUrl)) {
                throw new ApiParseException("Found different base url for endpoint " + id + ": " + parts[1]);
            }
        } else {
            throw new ApiParseException("Found multiple urls for endpoint: " + id);
        }
        String path = parts[1].substring(this.endpointUrl.length());
        int queryParamStart = path.indexOf(63);
        if (queryParamStart != -1) {
            path = path.substring(0, queryParamStart);
        }
        String responseDescription = null;
        List<SpotifyWebApiEndpoint.Parameter> parameters = null;
        String notes = null;
        List<Elements> h5Sections = ParseUtils.splitAt(elements, "h5");
        block10: for (Elements h5Section : h5Sections) {
            switch (((Element)h5Section.get(0)).text()) {
                case "Request": {
                    parameters = this.parseRequestParameters(h5Section);
                    continue block10;
                }
                case "Response": {
                    responseDescription = this.parseResponseParameters(h5Section.subList(1, h5Section.size()));
                    continue block10;
                }
                case "Notes": {
                    notes = this.parseNotes(h5Section.subList(1, h5Section.size()));
                    continue block10;
                }
            }
            log.warn("Unknown h5 section in endpoint " + id + ": " + ((Element)h5Section.get(0)).text());
        }
        if (parameters == null) {
            throw new ApiParseException("Can not find request parameters for endpoint " + id);
        }
        if (responseDescription == null) {
            throw new ApiParseException("Can not find response description for endpoint " + id);
        }
        List<String> scopes = this.extractScopes(id, parameters);
        return new SpotifyWebApiEndpoint(id, name, link, description, httpMethod, path, parameters, responseDescription, scopes, notes);
    }

    private List<SpotifyWebApiEndpoint.Parameter> parseRequestParameters(Elements elements) {
        ArrayList<SpotifyWebApiEndpoint.Parameter> parameters = new ArrayList<SpotifyWebApiEndpoint.Parameter>();
        for (Element table : elements.select("table")) {
            String locationString = table.selectFirst("thead > tr > th").text();
            SpotifyWebApiEndpoint.ParameterLocation location = this.parse(locationString);
            for (Element entry : table.select("tbody > tr")) {
                Element descriptionElement;
                String name = entry.selectFirst("code").text();
                if (location == SpotifyWebApiEndpoint.ParameterLocation.PATH) {
                    name = name.substring(1, name.length() - 1);
                }
                String description = (descriptionElement = entry.selectFirst("small")) == null ? "" : Html2Markdown.convert((Node)descriptionElement);
                String type = entry.child(1).text();
                String requiredText = entry.child(2).text();
                parameters.add(new SpotifyWebApiEndpoint.Parameter(location, name, description, type, "required".equalsIgnoreCase(requiredText)));
            }
        }
        return parameters;
    }

    private SpotifyWebApiEndpoint.ParameterLocation parse(String text) {
        switch (text) {
            case "Header": {
                return SpotifyWebApiEndpoint.ParameterLocation.HEADER;
            }
            case "Path Parameter": {
                return SpotifyWebApiEndpoint.ParameterLocation.PATH;
            }
            case "Query Parameter": {
                return SpotifyWebApiEndpoint.ParameterLocation.QUERY;
            }
            case "JSON Body Parameter": {
                return SpotifyWebApiEndpoint.ParameterLocation.BODY;
            }
        }
        throw new IllegalArgumentException("Unknown parameter location: " + text);
    }

    private String parseResponseParameters(List<Element> elements) {
        elements.removeIf(e -> "Try in our Web Console".equalsIgnoreCase(e.text()));
        return Html2Markdown.convert(elements);
    }

    private String parseNotes(List<Element> elements) {
        elements.removeIf(e -> "Try in our Web Console".equalsIgnoreCase(e.text()));
        return Html2Markdown.convert(elements);
    }

    private List<String> extractScopes(String id, List<SpotifyWebApiEndpoint.Parameter> parameters) {
        Optional<SpotifyWebApiEndpoint.Parameter> authHeader = parameters.stream().filter(p -> "Authorization".equals(p.getName())).findFirst();
        if (authHeader.isEmpty()) {
            log.warn("Endpoint {} has no Authorization header", (Object)id);
            return new ArrayList<String>();
        }
        Pattern codePattern = Pattern.compile("`([a-z-]+)`");
        Matcher matcher = codePattern.matcher(authHeader.get().getDescription());
        ArrayList<String> scopes = new ArrayList<String>();
        while (matcher.find()) {
            scopes.add(matcher.group(1));
        }
        return scopes;
    }

    public ApiEndpointParser(boolean isInteractive) {
        this.isInteractive = isInteractive;
    }
}

