/*
 * Decompiled with CFR 0.152.
 */
package de.sonallux.spotify.generator.openapi;

import com.google.common.base.Strings;
import de.sonallux.spotify.core.SpotifyWebApiUtils;
import de.sonallux.spotify.core.model.SpotifyAuthorizationScopes;
import de.sonallux.spotify.core.model.SpotifyScope;
import de.sonallux.spotify.core.model.SpotifyWebApi;
import de.sonallux.spotify.core.model.SpotifyWebApiCategory;
import de.sonallux.spotify.core.model.SpotifyWebApiEndpoint;
import de.sonallux.spotify.core.model.SpotifyWebApiObject;
import de.sonallux.spotify.generator.openapi.CloneHelper;
import de.sonallux.spotify.generator.openapi.VersionProvider;
import io.swagger.v3.oas.models.Components;
import io.swagger.v3.oas.models.ExternalDocumentation;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.Operation;
import io.swagger.v3.oas.models.PathItem;
import io.swagger.v3.oas.models.Paths;
import io.swagger.v3.oas.models.info.Contact;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.media.ArraySchema;
import io.swagger.v3.oas.models.media.BooleanSchema;
import io.swagger.v3.oas.models.media.ComposedSchema;
import io.swagger.v3.oas.models.media.Content;
import io.swagger.v3.oas.models.media.DateTimeSchema;
import io.swagger.v3.oas.models.media.IntegerSchema;
import io.swagger.v3.oas.models.media.MediaType;
import io.swagger.v3.oas.models.media.NumberSchema;
import io.swagger.v3.oas.models.media.ObjectSchema;
import io.swagger.v3.oas.models.media.Schema;
import io.swagger.v3.oas.models.media.StringSchema;
import io.swagger.v3.oas.models.parameters.Parameter;
import io.swagger.v3.oas.models.parameters.RequestBody;
import io.swagger.v3.oas.models.responses.ApiResponse;
import io.swagger.v3.oas.models.responses.ApiResponses;
import io.swagger.v3.oas.models.security.OAuthFlow;
import io.swagger.v3.oas.models.security.OAuthFlows;
import io.swagger.v3.oas.models.security.Scopes;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import io.swagger.v3.oas.models.servers.Server;
import io.swagger.v3.oas.models.tags.Tag;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OpenApiGenerator {
    private static final Logger log = LoggerFactory.getLogger(OpenApiGenerator.class);
    private static final String MEDIA_TYPE_JSON = "application/json";
    private static final String SPOTIFY_SECURITY_SCHEME = "spotify_auth";
    private OpenAPI openAPI;
    private final CloneHelper cloneHelper = new CloneHelper();

    public OpenAPI generate(SpotifyWebApi apiDocumentation) {
        this.openAPI = new OpenAPI();
        this.openAPI.externalDocs(this.generateExternalDocumentation(apiDocumentation.getApiDocumentationUrl())).info(new Info().title("Spotify Web API").version(VersionProvider.getVersion()).contact(new Contact().name("sonallux").url("https://github.com/sonallux/spotify-web-api"))).servers(List.of(new Server().url(apiDocumentation.getEndpointUrl()))).components(new Components().schemas(this.generateSchemaObjects(apiDocumentation.getObjectList())).addResponses("ErrorResponse", this.getDefaultErrorResponse()).securitySchemes(Map.of(SPOTIFY_SECURITY_SCHEME, this.getSpotifySecurityScheme(apiDocumentation.getScopes())))).tags(this.generateTags(apiDocumentation.getCategoryList())).paths(this.generatePaths(apiDocumentation.getCategoryList()));
        return this.openAPI;
    }

    private SecurityScheme getSpotifySecurityScheme(SpotifyAuthorizationScopes scopes) {
        Scopes openApiScopes = new Scopes();
        openApiScopes.putAll(scopes.getScopeList().stream().collect(Collectors.toMap(SpotifyScope::getId, SpotifyScope::getDescription)));
        return new SecurityScheme().type(SecurityScheme.Type.OAUTH2).flows(new OAuthFlows().authorizationCode(new OAuthFlow().authorizationUrl("https://accounts.spotify.com/authorize").tokenUrl("https://accounts.spotify.com/api/token").scopes(openApiScopes)));
    }

    private Paths generatePaths(Collection<SpotifyWebApiCategory> categories) {
        Paths paths = new Paths();
        for (SpotifyWebApiCategory category : categories) {
            block13: for (SpotifyWebApiEndpoint endpoint : category.getEndpointList()) {
                PathItem path = (PathItem)paths.computeIfAbsent((Object)endpoint.getPath(), s -> new PathItem());
                Operation operation = this.generateOperation(endpoint);
                operation.addTagsItem(category.getId());
                switch (endpoint.getHttpMethod()) {
                    case "GET": {
                        path.setGet(operation);
                        continue block13;
                    }
                    case "PUT": {
                        path.setPut(operation);
                        continue block13;
                    }
                    case "POST": {
                        path.setPost(operation);
                        continue block13;
                    }
                    case "DELETE": {
                        path.setDelete(operation);
                        continue block13;
                    }
                }
                log.warn("Unknown http method at endpoint " + endpoint.getId() + ": " + endpoint.getHttpMethod());
            }
        }
        return paths;
    }

    private Operation generateOperation(SpotifyWebApiEndpoint endpoint) {
        List parameters = endpoint.getParameters().stream().map(this::generateParameter).filter(Objects::nonNull).collect(Collectors.toList());
        RequestBody requestBody = this.generateRequestBody(endpoint);
        ApiResponses apiResponses = new ApiResponses()._default(new ApiResponse().$ref("#/components/responses/ErrorResponse"));
        for (SpotifyWebApiEndpoint.ResponseType responseType : endpoint.getResponseTypes()) {
            ApiResponse response = this.getApiResponse(responseType);
            if (response == null) continue;
            response.description(endpoint.getResponseDescription());
            apiResponses.put((Object)String.valueOf(responseType.getStatus()), (Object)response);
        }
        return new Operation().operationId(endpoint.getId()).summary(endpoint.getName()).description(endpoint.getDescription()).externalDocs(this.generateExternalDocumentation(endpoint.getLink())).parameters(parameters).requestBody(requestBody).responses(apiResponses).security(List.of(new SecurityRequirement().addList(SPOTIFY_SECURITY_SCHEME, endpoint.getScopes())));
    }

    private ApiResponse getApiResponse(SpotifyWebApiEndpoint.ResponseType responseType) {
        if ("Void".equals(responseType.getType())) {
            return new ApiResponse();
        }
        Schema responseSchema = this.getSchema(responseType.getType(), this.openAPI.getComponents().getSchemas());
        if (responseSchema == null) {
            return null;
        }
        Content content = new Content().addMediaType(MEDIA_TYPE_JSON, new MediaType().schema(responseSchema));
        return new ApiResponse().content(content);
    }

    private Parameter generateParameter(SpotifyWebApiEndpoint.Parameter param) {
        Parameter parameter = new Parameter().name(param.getName()).description(param.getDescription()).required(Boolean.valueOf(param.isRequired())).schema(this.getSchema(param.getType(), Map.of()));
        switch (param.getLocation()) {
            case HEADER: {
                parameter.in("header");
                break;
            }
            case PATH: {
                parameter.in("path");
                break;
            }
            case QUERY: {
                parameter.in("query");
                break;
            }
            case BODY: {
                return null;
            }
            default: {
                log.warn("Parameter " + param.getName() + " has unknown location: " + param.getLocation());
                return null;
            }
        }
        return parameter;
    }

    public RequestBody generateRequestBody(SpotifyWebApiEndpoint endpoint) {
        SpotifyWebApiEndpoint.RequestBody requestBody = endpoint.getRequestBody();
        if (requestBody == null) {
            return null;
        }
        if (requestBody instanceof SpotifyWebApiEndpoint.JsonRequestBody) {
            return this.generateJsonRequestBody((SpotifyWebApiEndpoint.JsonRequestBody)requestBody);
        }
        if (requestBody instanceof SpotifyWebApiEndpoint.Base64ImageRequestBody) {
            return this.generateBase64ImageRequestBody((SpotifyWebApiEndpoint.Base64ImageRequestBody)requestBody);
        }
        log.warn("Unsupported RequestBody type: " + requestBody.getClass().getName());
        return null;
    }

    private RequestBody generateJsonRequestBody(SpotifyWebApiEndpoint.JsonRequestBody requestBody) {
        ArrayList<String> requiredProps = new ArrayList<String>();
        LinkedHashMap<String, Schema> props = new LinkedHashMap<String, Schema>();
        for (SpotifyWebApiEndpoint.Parameter param : requestBody.getParameters()) {
            Schema paramSchema = this.getSchema(param.getType(), Map.of());
            if (paramSchema == null) continue;
            paramSchema.description(param.getDescription());
            props.put(param.getName(), paramSchema);
            if (!param.isRequired()) continue;
            requiredProps.add(param.getName());
        }
        Schema schema = new ObjectSchema().properties(props);
        if (requiredProps.size() != 0) {
            schema.required(requiredProps);
        }
        return new RequestBody().description(Strings.isNullOrEmpty((String)requestBody.getDescription()) ? null : requestBody.getDescription()).content(new Content().addMediaType(requestBody.getContentType(), new MediaType().schema(schema))).required(Boolean.valueOf(requiredProps.size() != 0));
    }

    private RequestBody generateBase64ImageRequestBody(SpotifyWebApiEndpoint.Base64ImageRequestBody requestBody) {
        return new RequestBody().description(requestBody.getDescription()).content(new Content().addMediaType(requestBody.getContentType(), new MediaType().schema(new StringSchema().format("base64")))).required(Boolean.valueOf(true));
    }

    public ApiResponse getDefaultErrorResponse() {
        return new ApiResponse().description("Unexpected error").content(new Content().addMediaType(MEDIA_TYPE_JSON, new MediaType().schema(new Schema().$ref("#/components/schemas/ErrorResponseObject"))));
    }

    public List<Tag> generateTags(Collection<SpotifyWebApiCategory> categories) {
        return categories.stream().map(c -> new Tag().name(c.getId()).description(c.getName()).externalDocs(this.generateExternalDocumentation(c.getLink()))).sorted(Comparator.comparing(Tag::getName)).collect(Collectors.toList());
    }

    private ExternalDocumentation generateExternalDocumentation(String link) {
        return new ExternalDocumentation().url(link).description("Find more info on the official Spotify Web API Reference");
    }

    private Map<String, Schema> generateSchemaObjects(Collection<SpotifyWebApiObject> objects) {
        LinkedHashMap<String, Schema> schemas = new LinkedHashMap<String, Schema>();
        SpotifyWebApiObject pagingObject = null;
        SpotifyWebApiObject cursorPagingObject = null;
        for (SpotifyWebApiObject object : objects) {
            if ("PagingObject".equals(object.getName())) {
                pagingObject = object;
            } else if ("CursorPagingObject".equals(object.getName())) {
                cursorPagingObject = object;
            }
            ObjectSchema objectSchema = new ObjectSchema();
            if (!Strings.isNullOrEmpty((String)object.getLink())) {
                objectSchema.externalDocs(this.generateExternalDocumentation(object.getLink()));
            }
            schemas.put(object.getName(), (Schema)objectSchema);
        }
        if (pagingObject == null || cursorPagingObject == null) {
            log.warn("Can not find PagingObject or CursorPagingObject");
        } else {
            ((Schema)schemas.get(pagingObject.getName())).properties(this.generateObjectProperties(pagingObject, schemas));
            schemas.get(cursorPagingObject.getName()).properties(this.generateObjectProperties(cursorPagingObject, schemas));
        }
        for (SpotifyWebApiObject object : objects) {
            schemas.get(object.getName()).properties(this.generateObjectProperties(object, schemas));
        }
        return schemas;
    }

    private Map<String, Schema> generateObjectProperties(SpotifyWebApiObject object, Map<String, Schema> customSchemas) {
        LinkedHashMap<String, Schema> properties = new LinkedHashMap<String, Schema>();
        for (SpotifyWebApiObject.Property prop : object.getProperties()) {
            Schema schema = this.getSchema(prop.getType(), customSchemas);
            if (schema == null) continue;
            schema.description(prop.getDescription());
            properties.put(prop.getName(), schema);
        }
        return properties;
    }

    private Schema getSchema(String type, Map<String, Schema> customSchemas) {
        if ("String".equals(type)) {
            return new StringSchema();
        }
        if ("Integer".equals(type)) {
            return new IntegerSchema();
        }
        if ("Float".equals(type) || "Number".equals(type)) {
            return new NumberSchema();
        }
        if ("Boolean".equals(type)) {
            return new BooleanSchema();
        }
        if ("Timestamp".equals(type)) {
            return new DateTimeSchema();
        }
        if ("Object".equals(type)) {
            return new ObjectSchema();
        }
        if ("Void".equals(type)) {
            return null;
        }
        if (customSchemas.containsKey(type)) {
            return new Schema().$ref("#/components/schemas/" + type);
        }
        Matcher matcher = SpotifyWebApiUtils.ARRAY_TYPE_PATTERN.matcher(type);
        if (matcher.matches()) {
            Schema arrayItemSchema = this.getSchema(matcher.group(1), customSchemas);
            return new ArraySchema().items(arrayItemSchema);
        }
        matcher = SpotifyWebApiUtils.PAGING_OBJECT_TYPE_PATTERN.matcher(type);
        if (matcher.matches()) {
            Schema arrayItemSchema = this.getSchema(matcher.group(1), customSchemas);
            Schema pagingObjectSchema = this.cloneHelper.cloneSchema(customSchemas.get("PagingObject"));
            ArraySchema pagingItemsSchema = (ArraySchema)pagingObjectSchema.getProperties().get("items");
            pagingItemsSchema.items(arrayItemSchema);
            return pagingObjectSchema;
        }
        matcher = SpotifyWebApiUtils.CURSOR_PAGING_OBJECT_TYPE_PATTERN.matcher(type);
        if (matcher.matches()) {
            Schema arrayItemSchema = this.getSchema(matcher.group(1), customSchemas);
            Schema pagingObjectSchema = this.cloneHelper.cloneSchema(customSchemas.get("CursorPagingObject"));
            ArraySchema pagingItemsSchema = (ArraySchema)pagingObjectSchema.getProperties().get("items");
            pagingItemsSchema.items(arrayItemSchema);
            return pagingObjectSchema;
        }
        if (type.contains(" | ")) {
            List types = Arrays.stream(type.split(" \\| ")).map(t -> this.getSchema((String)t, customSchemas)).collect(Collectors.toList());
            return new ComposedSchema().oneOf(types);
        }
        throw new RuntimeException("Missing type: " + type);
    }
}

