/*
 * Decompiled with CFR 0.152.
 */
package de.softwareforge.testing.postgres.embedded;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.io.CharStreams;
import com.google.common.io.Closeables;
import de.softwareforge.testing.postgres.embedded.DatabaseInfo;
import de.softwareforge.testing.postgres.embedded.EmbeddedUtil;
import de.softwareforge.testing.postgres.embedded.PgDirectoryResolver;
import de.softwareforge.testing.postgres.embedded.ProcessOutputLogger;
import de.softwareforge.testing.postgres.embedded.UncompressBundleDirectoryResolver;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.sql.SQLException;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.sql.DataSource;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.postgresql.ds.PGSimpleDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class EmbeddedPostgres
implements AutoCloseable {
    static final String[] LOCALHOST_SERVERNAMES = new String[]{"localhost"};
    static final String JDBC_FORMAT = "jdbc:postgresql://localhost:%d/%s?user=%s";
    private static final String PG_TEMPLATE_DB = "template1";
    @VisibleForTesting
    public static final Duration DEFAULT_PG_STARTUP_WAIT = Duration.ofSeconds(10L);
    private static final long MINIMUM_AGE_IN_MS = Duration.ofMinutes(10L).toMillis();
    private static final String DATA_DIRECTORY_PREFIX = "epd-";
    private static final String PG_STOP_MODE = "fast";
    private static final String PG_STOP_WAIT_S = "5";
    static final String LOCK_FILE_NAME = "epg-lock";
    private final Logger logger;
    private final String instanceId;
    private final File pgDir;
    private final File dataDirectory;
    private final Duration serverStartupWait;
    private final int port;
    private final AtomicBoolean started = new AtomicBoolean();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final ImmutableMap<String, String> serverConfiguration;
    private final ImmutableMap<String, String> localeConfiguration;
    private final ImmutableMap<String, String> connectionProperties;
    private final File lockFile;
    private volatile FileOutputStream lockStream;
    private volatile FileLock lock;
    private final boolean removeDataOnShutdown;
    private final ProcessBuilder.Redirect errorRedirector;
    private final ProcessBuilder.Redirect outputRedirector;

    public static EmbeddedPostgres defaultInstance() throws IOException {
        return EmbeddedPostgres.builderWithDefaults().build();
    }

    public static Builder builderWithDefaults() {
        return new Builder().withDefaults();
    }

    public static Builder builder() {
        return new Builder();
    }

    private EmbeddedPostgres(String instanceId, File postgresInstallDirectory, File dataDirectory, boolean removeDataOnShutdown, Map<String, String> serverConfiguration, Map<String, String> localeConfiguration, Map<String, String> connectionProperties, int port, ProcessBuilder.Redirect errorRedirector, ProcessBuilder.Redirect outputRedirector, Duration serverStartupWait) {
        this.instanceId = (String)Preconditions.checkNotNull((Object)instanceId, (Object)"instanceId is null");
        this.logger = LoggerFactory.getLogger((String)this.toString());
        this.pgDir = (File)Preconditions.checkNotNull((Object)postgresInstallDirectory, (Object)"postgresInstallDirectory is null");
        this.dataDirectory = (File)Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
        this.removeDataOnShutdown = removeDataOnShutdown;
        this.serverConfiguration = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(serverConfiguration, (Object)"serverConfiguration is null")));
        this.localeConfiguration = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(localeConfiguration, (Object)"localeConfiguration is null")));
        this.connectionProperties = ImmutableMap.copyOf((Map)((Map)Preconditions.checkNotNull(connectionProperties, (Object)"connectionProperties is null")));
        this.port = port;
        this.errorRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)errorRedirector, (Object)"errorRedirector is null");
        this.outputRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)outputRedirector, (Object)"outputRedirector is null");
        this.serverStartupWait = (Duration)Preconditions.checkNotNull((Object)serverStartupWait, (Object)"serverStartupWait is null");
        this.lockFile = new File(this.dataDirectory, LOCK_FILE_NAME);
        this.logger.debug(String.format("data dir is %s, install dir is %s", this.dataDirectory, this.pgDir));
    }

    public DatabaseInfo createDefaultDatabaseInfo() {
        return DatabaseInfo.builder().port(this.getPort()).properties(this.connectionProperties).build();
    }

    public DataSource createTemplateDataSource() throws SQLException {
        return EmbeddedPostgres.createDataSource("postgres", PG_TEMPLATE_DB, this.getPort(), this.connectionProperties);
    }

    public DataSource createDefaultDataSource() throws SQLException {
        return EmbeddedPostgres.createDataSource("postgres", "postgres", this.getPort(), this.connectionProperties);
    }

    public DataSource createDataSource(String user, String databaseName) throws SQLException {
        return EmbeddedPostgres.createDataSource(user, databaseName, this.getPort(), this.connectionProperties);
    }

    static DataSource createDataSource(String user, String databaseName, int port, Map<String, String> properties) throws SQLException {
        Preconditions.checkNotNull((Object)user, (Object)"user is null");
        Preconditions.checkNotNull((Object)databaseName, (Object)"databaseName is null");
        Preconditions.checkNotNull(properties, (Object)"properties is null");
        PGSimpleDataSource ds = new PGSimpleDataSource();
        ds.setServerNames(LOCALHOST_SERVERNAMES);
        ds.setPortNumbers(new int[]{port});
        ds.setDatabaseName(databaseName);
        ds.setUser(user);
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            ds.setProperty(entry.getKey(), entry.getValue());
        }
        return ds;
    }

    public int getPort() {
        return this.port;
    }

    ImmutableMap<String, String> getConnectionProperties() {
        return this.connectionProperties;
    }

    public String instanceId() {
        return this.instanceId;
    }

    public String toString() {
        Preconditions.checkNotNull((Object)this.instanceId, (Object)"instanceId is null");
        return this.getClass().getName() + "$" + this.instanceId;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        EmbeddedPostgres that = (EmbeddedPostgres)o;
        return this.instanceId.equals(that.instanceId);
    }

    public int hashCode() {
        return Objects.hash(this.instanceId);
    }

    private void boot() throws IOException {
        EmbeddedUtil.mkdirs(this.dataDirectory);
        if (this.removeDataOnShutdown || !new File(this.dataDirectory, "postgresql.conf").exists()) {
            this.initDatabase();
        }
        this.lock();
        this.startDatabase();
    }

    private synchronized void lock() throws IOException {
        this.lockStream = new FileOutputStream(this.lockFile);
        this.lock = this.lockStream.getChannel().tryLock();
        Preconditions.checkState((this.lock != null ? 1 : 0) != 0, (String)"could not lock %s", (Object)this.lockFile);
    }

    private synchronized void unlock() throws IOException {
        if (this.lock != null) {
            this.lock.release();
        }
        Closeables.close((Closeable)this.lockStream, (boolean)true);
    }

    private void initDatabase() throws IOException {
        ImmutableList.Builder commandBuilder = ImmutableList.builder();
        commandBuilder.add((Object)this.pgBin("initdb")).addAll(this.createLocaleOptions()).add((Object[])new String[]{"-A", "trust", "-U", "postgres", "-D", this.dataDirectory.getPath(), "-E", "UTF-8"});
        Stopwatch watch = this.system((List<String>)commandBuilder.build(), true);
        this.logger.debug(String.format("initdb completed in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
    }

    private void startDatabase() throws IOException {
        Preconditions.checkState((!this.started.getAndSet(true) ? 1 : 0) != 0, (Object)"database already started!");
        ImmutableList.Builder commandBuilder = ImmutableList.builder();
        commandBuilder.add((Object[])new String[]{this.pgBin("pg_ctl"), "-D", this.dataDirectory.getPath(), "-o", this.createInitOptions().stream().collect(Collectors.joining(" ")), "start"});
        Stopwatch watch = Stopwatch.createStarted();
        Process postmaster = this.spawn("pg", (List<String>)commandBuilder.build(), true);
        this.logger.info(String.format("started as pid %d on port %d", postmaster.pid(), this.port));
        this.logger.debug(String.format("Waiting up to %s for server startup to finish", EmbeddedUtil.formatDuration(this.serverStartupWait)));
        Runtime.getRuntime().addShutdownHook(this.newCloserThread());
        Preconditions.checkState((boolean)this.waitForServerStartup(), (Object)"Could not start pg, interrupted?");
        this.logger.debug(String.format("startup complete in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
    }

    private void stopDatabase(File dataDirectory) throws IOException {
        ImmutableList.Builder commandBuilder = ImmutableList.builder();
        commandBuilder.add((Object[])new String[]{this.pgBin("pg_ctl"), "-D", dataDirectory.getPath(), "stop", "-m", PG_STOP_MODE, "-t", PG_STOP_WAIT_S, "-w"});
        Stopwatch watch = this.system((List<String>)commandBuilder.build(), true);
        this.logger.debug(String.format("shutdown complete in %s", EmbeddedUtil.formatDuration(watch.elapsed())));
    }

    private List<String> createInitOptions() {
        ImmutableList.Builder initOptions = ImmutableList.builder();
        initOptions.add((Object[])new String[]{"-p", Integer.toString(this.port), "-F"});
        this.serverConfiguration.forEach((k, v) -> {
            initOptions.add((Object)"-c");
            initOptions.add((Object)(k + "=" + v));
        });
        return initOptions.build();
    }

    private List<String> createLocaleOptions() {
        ImmutableList.Builder localeOptions = ImmutableList.builder();
        this.localeConfiguration.forEach((key, value) -> {
            if (SystemUtils.IS_OS_WINDOWS) {
                localeOptions.add((Object)("--" + key + "=" + value));
            } else {
                localeOptions.add((Object[])new String[]{"--" + key, value});
            }
        });
        return localeOptions.build();
    }

    private boolean waitForServerStartup() throws IOException {
        SQLException lastCause = null;
        long start = System.nanoTime();
        long maxWaitNs = TimeUnit.NANOSECONDS.convert(this.serverStartupWait.toMillis(), TimeUnit.MILLISECONDS);
        while (System.nanoTime() - start < maxWaitNs) {
            try {
                if (this.verifyReady()) {
                    return true;
                }
            }
            catch (SQLException e) {
                lastCause = e;
                this.logger.trace("while waiting for server startup:", (Throwable)e);
            }
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
        }
        throw new IOException("Gave up waiting for server to start after " + this.serverStartupWait.toMillis() + "ms", lastCause);
    }

    /*
     * Exception decompiling
     */
    private boolean verifyReady() throws IOException, SQLException {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    private Thread newCloserThread() {
        Thread closeThread = new Thread(() -> {
            try {
                this.close();
            }
            catch (IOException e) {
                this.logger.trace("while closing instance:", (Throwable)e);
            }
        });
        closeThread.setName("pg-closer");
        return closeThread;
    }

    @Override
    public void close() throws IOException {
        if (this.closed.getAndSet(true)) {
            return;
        }
        try {
            this.stopDatabase(this.dataDirectory);
        }
        catch (Exception e) {
            this.logger.error("could not stop pg:", (Throwable)e);
        }
        this.unlock();
        if (this.removeDataOnShutdown) {
            try {
                EmbeddedUtil.rmdirs(this.dataDirectory);
            }
            catch (Exception e) {
                this.logger.error(String.format("Could not clean up directory %s:", this.dataDirectory.getAbsolutePath()), (Throwable)e);
            }
        } else {
            this.logger.debug(String.format("preserved data directory %s", this.dataDirectory.getAbsolutePath()));
        }
    }

    @VisibleForTesting
    File getDataDirectory() {
        return this.dataDirectory;
    }

    @VisibleForTesting
    Map<String, String> getLocaleConfiguration() {
        return this.localeConfiguration;
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_OF_NONNULL_VALUE"})
    private void cleanOldDataDirectories(File parentDirectory) {
        File[] children = parentDirectory.listFiles();
        if (children == null) {
            return;
        }
        for (File dir : children) {
            File lockFile;
            if (!dir.isDirectory() || !dir.getName().startsWith(DATA_DIRECTORY_PREFIX) || !(lockFile = new File(dir, LOCK_FILE_NAME)).exists() || System.currentTimeMillis() - lockFile.lastModified() < MINIMUM_AGE_IN_MS) continue;
            try (FileOutputStream fos = new FileOutputStream(lockFile);
                 FileLock lock = fos.getChannel().tryLock();){
                if (lock == null) continue;
                this.logger.debug(String.format("found stale data directory %s", dir));
                if (new File(dir, "postmaster.pid").exists()) {
                    try {
                        this.stopDatabase(dir);
                        this.logger.debug("shut down orphaned database!");
                    }
                    catch (Exception e) {
                        this.logger.warn(String.format("failed to orphaned database in %s:", dir), (Throwable)e);
                    }
                }
                EmbeddedUtil.rmdirs(dir);
            }
            catch (OverlappingFileLockException e) {
                this.logger.trace("while cleaning old data directories:", (Throwable)e);
            }
            catch (Exception e) {
                this.logger.warn("while cleaning old data directories:", (Throwable)e);
            }
        }
    }

    private String pgBin(String binaryName) {
        String extension = SystemUtils.IS_OS_WINDOWS ? ".exe" : "";
        return new File(this.pgDir, "bin/" + binaryName + extension).getPath();
    }

    private Process spawn(@Nullable String processName, List<String> commandAndArgs, boolean debug) throws IOException {
        ProcessBuilder builder = new ProcessBuilder(commandAndArgs);
        builder.redirectErrorStream(true);
        builder.redirectError(this.errorRedirector);
        builder.redirectOutput(this.outputRedirector);
        Process process = builder.start();
        processName = processName != null ? processName : process.info().command().map(EmbeddedUtil::getFileBaseName).orElse("<unknown>");
        String name = String.format("%s (%d)", processName, process.pid());
        ProcessOutputLogger.logOutput(debug, this.logger, name, process);
        return process;
    }

    private Stopwatch system(List<String> commandAndArgs, boolean debug) throws IOException {
        Stopwatch watch;
        block6: {
            Preconditions.checkArgument((commandAndArgs.size() > 0 ? 1 : 0) != 0, (Object)"No commandAndArgs given!");
            String prefix = EmbeddedUtil.getFileBaseName(commandAndArgs.get(0));
            watch = Stopwatch.createStarted();
            Process process = this.spawn(prefix, commandAndArgs, debug);
            try {
                if (process.waitFor() == 0) break block6;
                InputStreamReader reader = new InputStreamReader(process.getErrorStream(), StandardCharsets.UTF_8);
                try {
                    throw new IllegalStateException(String.format("Process %s failed%n%s", commandAndArgs, CharStreams.toString((Readable)reader)));
                }
                catch (Throwable throwable) {
                    try {
                        reader.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        return watch;
    }

    public static class Builder {
        private File installationDirectory = null;
        private File dataDirectory = null;
        private final Map<String, String> serverConfiguration = new HashMap<String, String>();
        private final Map<String, String> localeConfiguration = new HashMap<String, String>();
        private boolean removeDataOnShutdown = true;
        private int port = 0;
        private final Map<String, String> connectionProperties = new HashMap<String, String>();
        private PgDirectoryResolver directoryResolver = UncompressBundleDirectoryResolver.getDefault();
        private Duration serverStartupWait = DEFAULT_PG_STARTUP_WAIT;
        private ProcessBuilder.Redirect errRedirector = ProcessBuilder.Redirect.PIPE;
        private ProcessBuilder.Redirect outRedirector = ProcessBuilder.Redirect.PIPE;

        Builder() {
        }

        public Builder withDefaults() {
            this.serverConfiguration.put("timezone", "UTC");
            this.serverConfiguration.put("synchronous_commit", "off");
            this.serverConfiguration.put("max_connections", "300");
            return this;
        }

        public Builder setServerStartupWait(Duration serverStartupWait) {
            Preconditions.checkNotNull((Object)serverStartupWait, (Object)"serverStartupWait is null");
            Preconditions.checkArgument((!serverStartupWait.isNegative() ? 1 : 0) != 0, (Object)"Negative durations are not permitted.");
            this.serverStartupWait = serverStartupWait;
            return this;
        }

        public Builder setRemoveDataOnShutdown(boolean removeDataOnShutdown) {
            this.removeDataOnShutdown = removeDataOnShutdown;
            return this;
        }

        public Builder setDataDirectory(Path dataDirectory) {
            Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this.setDataDirectory(dataDirectory.toFile());
        }

        public Builder setDataDirectory(String dataDirectory) {
            Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this.setDataDirectory(new File(dataDirectory));
        }

        public Builder setDataDirectory(File dataDirectory) {
            this.dataDirectory = (File)Preconditions.checkNotNull((Object)dataDirectory, (Object)"dataDirectory is null");
            return this;
        }

        public Builder addServerConfiguration(String key, String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.serverConfiguration.put(key, value);
            return this;
        }

        public Builder addLocaleConfiguration(String key, String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.localeConfiguration.put(key, value);
            return this;
        }

        public Builder addConnectionProperty(String key, String value) {
            Preconditions.checkNotNull((Object)key, (Object)"key is null");
            Preconditions.checkNotNull((Object)value, (Object)"value is null");
            this.connectionProperties.put(key, value);
            return this;
        }

        public Builder setInstallationDirectory(File installationDirectory) {
            Preconditions.checkNotNull((Object)installationDirectory, (Object)"workingDirectory is null");
            this.installationDirectory = installationDirectory;
            return this;
        }

        public Builder setPort(int port) {
            this.port = port;
            return this;
        }

        public Builder setErrorRedirector(ProcessBuilder.Redirect errRedirector) {
            this.errRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)errRedirector, (Object)"errRedirector is null");
            return this;
        }

        public Builder setOutputRedirector(ProcessBuilder.Redirect outRedirector) {
            this.outRedirector = (ProcessBuilder.Redirect)Preconditions.checkNotNull((Object)outRedirector, (Object)"outRedirector is null");
            return this;
        }

        public Builder setPostgresDirectoryResolver(PgDirectoryResolver directoryResolver) {
            this.directoryResolver = (PgDirectoryResolver)Preconditions.checkNotNull((Object)directoryResolver, (Object)"directoryResolver is null");
            return this;
        }

        public Builder setPostgresBinaryDirectory(File directory) {
            Preconditions.checkNotNull((Object)directory, (Object)"directory is null");
            return this.setPostgresDirectoryResolver(x -> directory);
        }

        public EmbeddedPostgres build() throws IOException {
            String instanceId = RandomStringUtils.randomAlphanumeric((int)16);
            int port = this.port != 0 ? this.port : EmbeddedUtil.allocatePort();
            File parentDirectory = EmbeddedUtil.getWorkingDirectory();
            EmbeddedUtil.mkdirs(parentDirectory);
            File installationDirectory = (File)MoreObjects.firstNonNull((Object)this.installationDirectory, (Object)parentDirectory);
            File postgresInstallDirectory = this.directoryResolver.getDirectory(installationDirectory);
            File dataDirectory = this.dataDirectory != null ? this.dataDirectory : new File(parentDirectory, EmbeddedPostgres.DATA_DIRECTORY_PREFIX + instanceId);
            EmbeddedPostgres embeddedPostgres = new EmbeddedPostgres(instanceId, postgresInstallDirectory, dataDirectory, this.removeDataOnShutdown, this.serverConfiguration, this.localeConfiguration, this.connectionProperties, port, this.errRedirector, this.outRedirector, this.serverStartupWait);
            embeddedPostgres.cleanOldDataDirectories(parentDirectory);
            embeddedPostgres.boot();
            return embeddedPostgres;
        }
    }

    @FunctionalInterface
    public static interface BuilderCustomizer {
        public void customize(Builder var1) throws SQLException;
    }
}

