/*
 * Decompiled with CFR 0.152.
 */
package de.softwareforge.jsonschema;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.reflect.TypeToken;
import de.softwareforge.jsonschema.AttributeHolder;
import de.softwareforge.jsonschema.JsonSchemaGeneratorConfiguration;
import de.softwareforge.jsonschema.SimpleTypeMappings;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;

public final class JsonSchemaGenerator {
    private final JsonNodeFactory nodeFactory;
    private final JsonSchemaGeneratorConfiguration config;
    private final Set<Type> dictionary = new HashSet<Type>();

    JsonSchemaGenerator(JsonSchemaGeneratorConfiguration config) {
        this.nodeFactory = config.nodeFactory();
        this.config = config;
    }

    private static Optional<AttributeHolder> acceptMethod(Method method) {
        int modifiers = method.getModifiers();
        if (method.isBridge() || method.isSynthetic() || method.isDefault() || Modifier.isStatic(modifiers)) {
            return Optional.empty();
        }
        return AttributeHolder.locate(method);
    }

    private static Optional<AttributeHolder> acceptField(Field field) {
        int modifiers = field.getModifiers();
        if (field.isEnumConstant() || field.isSynthetic() || Modifier.isTransient(modifiers) || Modifier.isStatic(modifiers)) {
            return Optional.empty();
        }
        return AttributeHolder.locate(field);
    }

    public <T> ObjectNode generateSchema(Class<T> type) {
        TypeToken typeToken = TypeToken.of(type);
        Optional<AttributeHolder> rootAttributes = AttributeHolder.locate(typeToken.getRawType());
        ObjectNode schema = this.nodeFactory.objectNode();
        if (this.config.addSchemaVersion()) {
            schema.put("$schema", "http://json-schema.org/draft-04/schema#");
        }
        this.createSchemaForType(schema, type, rootAttributes);
        return schema;
    }

    private <T> void createSchemaForType(ObjectNode schema, Type type, Optional<AttributeHolder> attributes) {
        if (this.dictionary.contains(type)) {
            throw new IllegalStateException("Recursion detected, not supported!");
        }
        Optional<String> overriddenType = attributes.isPresent() ? attributes.get().type() : Optional.empty();
        Optional<String> s = SimpleTypeMappings.forClass(type);
        if (s.isPresent()) {
            this.addTypeToSchema(schema, overriddenType.orElse(s.get()));
            SimpleTypeMappings.formatHint(type).ifPresent(formatHint -> schema.put("format", formatHint));
        } else if (SimpleTypeMappings.isCollectionLike(type)) {
            this.augmentSchemaWithCollection(schema, type);
        } else if (type == Void.class || type == Void.TYPE) {
            this.addTypeToSchema(schema, overriddenType.orElse("null"));
        } else if (this.isEnum(type, attributes)) {
            this.augmentSchemaWithEnum((Class)type, schema);
        } else {
            this.dictionary.add(type);
            this.augmentSchemaWithCustomType(schema, type, attributes);
            this.dictionary.remove(type);
        }
        attributes.ifPresent(schemaAttributes -> this.augmentAttributes(schema, type, (AttributeHolder)schemaAttributes));
    }

    private <T> void augmentSchemaWithEnum(Class<T> type, ObjectNode schema) {
        ArrayNode enumArray = schema.putArray("enum");
        for (T constant : type.getEnumConstants()) {
            String value = constant.toString();
            try {
                Long integer = Long.parseLong(value);
                enumArray.add(integer);
            }
            catch (NumberFormatException e) {
                try {
                    BigDecimal number = new BigDecimal(value);
                    enumArray.add(number);
                }
                catch (NumberFormatException e1) {
                    enumArray.add(value);
                }
            }
        }
    }

    private <T> void augmentSchemaWithCustomType(ObjectNode schema, Type type, Optional<AttributeHolder> attributeHolder) {
        this.addTypeToSchema(schema, "object");
        if (attributeHolder.isPresent() && attributeHolder.get().ignoredProperties()) {
            return;
        }
        if (this.config.processProperties()) {
            this.findSchemaPropertiesFromMethods(type, schema).forEach((propertyName, objectNode) -> this.addToProperties(schema, (String)propertyName, (ObjectNode)objectNode));
        }
        if (this.config.processFields()) {
            this.findSchemaPropertiesFromFields(type, schema).forEach((propertyName, objectNode) -> this.addToProperties(schema, (String)propertyName, (ObjectNode)objectNode));
        }
    }

    private Map<String, ObjectNode> findSchemaPropertiesFromMethods(Type type, ObjectNode parent) {
        TreeMap<String, ObjectNode> propertyMap = this.config.sortSchemaProperties() ? new TreeMap() : new LinkedHashMap();
        TypeToken typeToken = TypeToken.of((Type)type);
        for (TypeToken implementingTypeToken : typeToken.getTypes()) {
            Method[] methods;
            Class clazz = implementingTypeToken.getRawType();
            for (Method method : methods = clazz.getDeclaredMethods()) {
                Optional<AttributeHolder> attributeHolder = JsonSchemaGenerator.acceptMethod(method);
                if (!attributeHolder.isPresent()) continue;
                AttributeHolder attributes = attributeHolder.get();
                String propertyName = attributes.named().orElseGet(() -> this.propertyName(method));
                if (propertyMap.containsKey(propertyName)) {
                    throw new IllegalStateException(String.format(Locale.ENGLISH, "Property %s defined multiple times (saw %s)", propertyName, clazz.getSimpleName()));
                }
                if (attributes.required()) {
                    this.addToRequired(parent, propertyName);
                }
                if (attributes.ignored()) continue;
                TypeToken returnType = implementingTypeToken.resolveType(method.getGenericReturnType());
                ObjectNode propertyNode = this.nodeFactory.objectNode();
                this.createSchemaForType(propertyNode, returnType.getType(), Optional.of(attributes));
                propertyMap.put(propertyName, propertyNode);
            }
        }
        return propertyMap;
    }

    private Map<String, ObjectNode> findSchemaPropertiesFromFields(Type type, ObjectNode parent) {
        TreeMap<String, ObjectNode> propertyMap = this.config.sortSchemaProperties() ? new TreeMap() : new LinkedHashMap();
        TypeToken typeToken = TypeToken.of((Type)type);
        for (TypeToken implementingTypeToken : typeToken.getTypes()) {
            Field[] fields;
            Class clazz = implementingTypeToken.getRawType();
            for (Field field : fields = clazz.getDeclaredFields()) {
                Optional<AttributeHolder> attributeHolder = JsonSchemaGenerator.acceptField(field);
                if (!attributeHolder.isPresent()) continue;
                AttributeHolder attributes = attributeHolder.get();
                String propertyName = attributes.named().orElse(this.propertyName(field));
                if (propertyMap.containsKey(propertyName)) {
                    throw new IllegalStateException(String.format(Locale.ENGLISH, "Property %s defined multiple times (saw %s)", propertyName, field.getName()));
                }
                if (attributes.required()) {
                    this.addToRequired(parent, propertyName);
                }
                if (attributes.ignored()) continue;
                TypeToken fieldType = implementingTypeToken.resolveType(field.getGenericType());
                ObjectNode propertyNode = this.nodeFactory.objectNode();
                this.createSchemaForType(propertyNode, fieldType.getType(), Optional.of(attributes));
                propertyMap.put(propertyName, propertyNode);
            }
        }
        return propertyMap;
    }

    private void augmentAttributes(ObjectNode schema, Type type, AttributeHolder schemaAttributes) {
        schemaAttributes.augmentCommonAttributes(schema);
        schemaAttributes.$ref().ifPresent($ref -> schema.put("$ref", $ref));
        if (!schemaAttributes.additionalProperties()) {
            schema.put("additionalProperties", false);
        }
        if (schemaAttributes.nullable()) {
            if (this.isEnum(type, Optional.of(schemaAttributes))) {
                ((ArrayNode)schema.get("enum")).addNull();
            }
            this.addTypeToSchema(schema, "null");
        }
    }

    private boolean isEnum(Type type, Optional<AttributeHolder> schemaAttributes) {
        if (schemaAttributes.isPresent() && !schemaAttributes.get().enums().isEmpty()) {
            return true;
        }
        return type instanceof Class && ((Class)type).isEnum();
    }

    private void augmentSchemaWithCollection(ObjectNode schema, Type type) {
        this.addTypeToSchema(schema, "array");
        TypeToken typeToken = TypeToken.of((Type)type);
        if (typeToken.isArray()) {
            this.augmentItems(schema, typeToken.getComponentType().getType());
        } else {
            Class clazz = typeToken.getRawType();
            Preconditions.checkState((clazz.getTypeParameters().length > 0 ? 1 : 0) != 0, (Object)"No type arguments in return type found!");
            Type itemType = typeToken.resolveType(clazz.getTypeParameters()[0]).getType();
            this.augmentItems(schema, itemType);
        }
    }

    public void augmentItems(ObjectNode schema, Type itemType) {
        ObjectNode itemNode = this.nodeFactory.objectNode();
        TypeToken typeToken = TypeToken.of((Type)itemType);
        Optional<AttributeHolder> itemAttributes = AttributeHolder.locate(typeToken.getRawType());
        this.createSchemaForType(itemNode, itemType, itemAttributes);
        schema.set("items", (JsonNode)itemNode);
    }

    private void addToRequired(ObjectNode schema, String name) {
        ArrayNode requiredNode = schema.has("required") ? (ArrayNode)schema.get("required") : schema.putArray("required");
        requiredNode.add(name);
    }

    private void addToProperties(ObjectNode schema, String name, ObjectNode property) {
        ObjectNode propertiesNode = schema.has("properties") ? (ObjectNode)schema.get("properties") : schema.putObject("properties");
        propertiesNode.set(name, (JsonNode)property);
    }

    private String propertyName(AnnotatedElement element) {
        if (element instanceof Field) {
            return ((Field)element).getName();
        }
        if (element instanceof Method) {
            Method method = (Method)element;
            Class<?> clazz = method.getDeclaringClass();
            try {
                PropertyDescriptor[] props;
                BeanInfo info = Introspector.getBeanInfo(clazz);
                for (PropertyDescriptor pd : props = info.getPropertyDescriptors()) {
                    if (!method.equals(pd.getWriteMethod()) && !method.equals(pd.getReadMethod())) continue;
                    return pd.getName();
                }
            }
            catch (IntrospectionException e) {
                throw new IllegalStateException(String.format(Locale.ENGLISH, "Could not locate property name for %s", method.getName()), e);
            }
            if (method.getParameterCount() == 0 && method.getReturnType() != Void.TYPE) {
                return method.getName();
            }
            throw new IllegalStateException(String.format(Locale.ENGLISH, "Could not locate property name for %s", method.getName()));
        }
        throw new IllegalArgumentException(String.format(Locale.ENGLISH, "%s is not a field or method", element));
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void addTypeToSchema(ObjectNode schema, String type) {
        if (schema.has("type")) {
            JsonNode typeNode = schema.get("type");
            if (typeNode.isArray()) {
                ArrayNode arrayNode = (ArrayNode)typeNode;
                arrayNode.add(type);
                return;
            } else {
                if (!typeNode.isTextual()) throw new IllegalStateException("Return type is not nullable");
                ArrayNode typeArray = schema.putArray("type");
                typeArray.add(typeNode);
                typeArray.add(type);
                schema.replace("type", (JsonNode)typeArray);
            }
            return;
        } else {
            schema.put("type", type);
        }
    }
}

