package de.sldw.composeonboarding

import androidx.compose.animation.expandHorizontally
import androidx.compose.animation.fadeIn
import androidx.compose.animation.fadeOut
import androidx.compose.animation.shrinkHorizontally
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.pager.HorizontalPager
import androidx.compose.foundation.pager.PagerState
import androidx.compose.foundation.pager.rememberPagerState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material3.BottomAppBar
import androidx.compose.material3.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.rememberCoroutineScope
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.unit.dp
import de.sldw.composeonboarding.indicator.IndicatorType
import kotlinx.coroutines.launch
import kotlin.math.min

/**
 * Entrypoint for compose onboarding
 *
 * @param pages List of OnboardingPage or PolicyOnboardingPage elements to be shown to the user. Cannot be empty
 * @param indicatorType Use displayed indicator. Choose between ProgressIndicator, DotIndicator and TextIndicator
 * @param onFinishPressed Callback for the next button on the last page
 * @param modifier Optional parameter providing modifier to the onboarding scaffold
 */

@Composable
fun ComposeOnboarding(
    pages: List<OnboardingPage>,
    indicatorType: IndicatorType,
    onFinishPressed: () -> Unit,
    modifier: Modifier = Modifier,
    settings: OnboardingSettings = OnboardingSettings(),
) {
    if (pages.isEmpty()) {
        return
    }

    val pagerState = rememberPagerState(
        pageCount = {
            min(
                pages
                    .takeWhile { it.mCanNavigateNext }
                    .count() + 1,
                pages.size
            )
        }
    )

    PlatformBackHandler()

    Scaffold(
        modifier = Modifier
            .fillMaxSize()
            .then(modifier),
        bottomBar = {
            BottomAppBar(
                modifier = Modifier
                    .fillMaxWidth()
                    .clip(RoundedCornerShape(16.dp, 16.dp, 0.dp, 0.dp))
            ) {
                Row(
                    modifier = Modifier
                        .fillMaxHeight()
                        .padding(4.dp),
                    verticalAlignment = Alignment.CenterVertically,
                ) {
                    OnboardingBackButton(
                        pagerState = pagerState,
                        settings = settings
                    )

                    Box(
                        modifier = Modifier.weight(1f),
                        contentAlignment = Alignment.Center
                    ) {
                        indicatorType.Content(
                            currentPage = pagerState.currentPage,
                            maxPages = pages.size
                        )
                    }

                    OnboardingNextButton(
                        pagerState = pagerState,
                        pages = pages,
                        settings = settings,
                        onFinishPressed = onFinishPressed
                    )
                }
            }
        }
    ) { padding ->
        HorizontalPager(
            modifier = Modifier.fillMaxSize(),
            state = pagerState
        ) {
            Box(
                modifier = Modifier.fillMaxSize(),
                content = {
                    DisposableEffect(pagerState.currentPage == it) {
                        if (pagerState.currentPage == it) {
                            pages[it].mOnAppear()
                        } else {
                            pages[it].mOnDisappear()
                        }

                        onDispose {
                            if (pagerState.currentPage == it) {
                                pages[it].mOnDisappear()
                            }
                        }
                    }

                    pages[it].Content(padding)
                }
            )
        }
    }
}



@Composable
internal fun OnboardingBackButton(
    pagerState: PagerState,
    settings: OnboardingSettings
) {
    val coroutine = rememberCoroutineScope()

    AnimatedVisibilityBlockingSpace(
        visible = pagerState.canScrollBackward,
        enter = fadeIn() + expandHorizontally(expandFrom = Alignment.Start),
        exit = fadeOut() + shrinkHorizontally(shrinkTowards = Alignment.Start)
    ) {
        settings.previousButton(
            OnboardingButtonParams(
                isEnabled = it,
                onClick = {
                    coroutine.launch {
                        pagerState.animateScrollToPage(pagerState.currentPage - 1)
                    }
                }
            )
        )
    }
}

@Composable
internal fun OnboardingNextButton(
    pagerState: PagerState,
    pages: List<OnboardingPage>,
    settings: OnboardingSettings,
    onFinishPressed: () -> Unit
) {
    val coroutine = rememberCoroutineScope()

    val isLastPage = ((pagerState.currentPage + 1) >= pages.size)
    Box(modifier = Modifier.width(IntrinsicSize.Min)) {
        AnimatedVisibilityBlockingSpace(
            visible = !isLastPage,
            enter = fadeIn() + expandHorizontally(expandFrom = Alignment.End),
            exit = fadeOut() + shrinkHorizontally(shrinkTowards = Alignment.End),
            zIndex = if (isLastPage) -100f else 100f
        ) {
            settings.nextButton(
                OnboardingButtonParams(
                    isEnabled = it && pages[pagerState.currentPage].mCanNavigateNext,
                    onClick = {
                        coroutine.launch {
                            pagerState.animateScrollToPage(pagerState.currentPage + 1)
                        }
                    }
                )
            )
        }

        AnimatedVisibilityBlockingSpace(
            visible = isLastPage,
            enter = fadeIn() + expandHorizontally(expandFrom = Alignment.End),
            exit = fadeOut() + shrinkHorizontally(shrinkTowards = Alignment.End),
            zIndex = if (isLastPage) 100f else -100f
        ) {
            settings.finishButton(
                OnboardingButtonParams(
                    isEnabled = it && pages[pagerState.currentPage].mCanNavigateNext,
                    onClick = { onFinishPressed() }
                )
            )
        }
    }
}
