package de.skuzzle.test.snapshots;

import java.nio.file.Path;

import de.skuzzle.test.snapshots.SnapshotTestResult.SnapshotStatus;
import de.skuzzle.test.snapshots.io.UncheckedIO;

import org.apiguardian.api.API;
import org.apiguardian.api.API.Status;

/**
 * Holds the paths to all files that might have been generated by a single snapshot
 * assertion.
 * <p>
 * Depending on the configuration (i.e. SnapshotTestOptions )
 *
 * @author Simon Taddiken
 * @since 1.8.0
 */
@API(status = Status.EXPERIMENTAL, since = "1.8.0")
public final class ContextFiles {

    private final Path snapshotFile;
    private final Path actualResultFile;
    private final Path rawActualResultFile;

    private ContextFiles(Path snapshotFile, Path actualResultFile, Path rawActualResultFile) {
        this.snapshotFile = snapshotFile;
        this.actualResultFile = actualResultFile;
        this.rawActualResultFile = rawActualResultFile;
    }

    public static ContextFiles of(Path snapshotFile, Path actualResultFile, Path rawActualResultFile) {
        return new ContextFiles(snapshotFile, actualResultFile, rawActualResultFile);
    }

    /**
     * The snapshot file. Note that it is possible that the file does not exist in case
     * that {@link SnapshotTestResult#status()} is {@link SnapshotStatus#DISABLED}.
     *
     * @return The snapshot file.
     * @see #actualResultFile()
     * @see #rawActualResultFile()
     */
    public Path snapshotFile() {
        return snapshotFile;
    }

    /**
     * Path to the file in which the latest actual result will be stored. The file will
     * only exist if the recent snapshot assertion was executed with
     * {@link SnapshotTestOptions#alwaysPersistActualResult()} being true.
     *
     * @return The path to the file with the latest actual result file.
     * @see #snapshotFile()
     * @see #rawActualResultFile()
     * @see SnapshotTestOptions#alwaysPersistActualResult()
     */
    public Path actualResultFile() {
        return actualResultFile;
    }

    /**
     * Path to the file in which the latest raw actual result will be stored (without the
     * snapshot header). The file will only exist if the recent snapshot assertion was
     * executed with {@link SnapshotTestOptions#alwaysPersistRawResult()} being true.
     *
     * @return The path to the file with the latest raw actual result file.
     * @see #snapshotFile()
     * @see #actualResultFile()
     * @see SnapshotTestOptions#alwaysPersistRawResult()
     */
    public Path rawActualResultFile() {
        return rawActualResultFile;
    }

    /**
     * Deletes all the context files that are existing.
     */
    public void deleteFiles() {
        UncheckedIO.deleteIfExists(snapshotFile);
        UncheckedIO.deleteIfExists(actualResultFile);
        UncheckedIO.deleteIfExists(rawActualResultFile);
    }
}
