// $Id: Whirlpool1.java 214 2010-06-03 17:25:08Z tp $

package fr.cryptohash;

/**
 * <p>This class implements the Whirlpool-1 digest algorithm under the
 * {@link Digest} API. This is the second variant of Whirlpool, created
 * in 2001 and superseded in 2003 by the third variant (Whirlpool).</p>
 *
 * <pre>
 * ==========================(LICENSE BEGIN)============================
 *
 * Copyright (c) 2007-2010  Projet RNRT SAPHIR
 * 
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * ===========================(LICENSE END)=============================
 * </pre>
 *
 * @version   $Revision: 214 $
 * @author    Thomas Pornin &lt;thomas.pornin@cryptolog.com&gt;
 */

public class Whirlpool1 extends WhirlpoolCore {

	/**
	 * Create the object.
	 */
	public Whirlpool1()
	{
		super(T0, T1, T2, T3, T4, T5, T6, T7, RC);
	}

	/** @see Digest */
	public Digest copy()
	{
		return copyState(new Whirlpool1());
	}

	private static final long[] T0 = {
		0x78D8C07818281818L, 0xAF2605AF23652323L,
		0xF9B87EF9C657C6C6L, 0x6FFB136FE825E8E8L,
		0xA1CB4CA187948787L, 0x6211A962B8D5B8B8L,
		0x0509080501030101L, 0x6E0D426E4FD14F4FL,
		0xEE9BADEE365A3636L, 0x04FF5904A6F7A6A6L,
		0xBD0CDEBDD26BD2D2L, 0x060EFB06F502F5F5L,
		0x8096EF80798B7979L, 0xCE305FCE6FB16F6FL,
		0xEF6DFCEF91AE9191L, 0x07F8AA0752F65252L,
		0xFD4727FD60A06060L, 0x76358976BCD9BCBCL,
		0xCD37ACCD9BB09B9BL, 0x8C8A048C8E8F8E8EL,
		0x15D27115A3F8A3A3L, 0x3C6C603C0C140C0CL,
		0x8A84FF8A7B8D7B7BL, 0xE180B5E1355F3535L,
		0x69F5E8691D271D1DL, 0x47B35347E03DE0E0L,
		0xAC21F6ACD764D7D7L, 0xED9C5EEDC25BC2C2L,
		0x96436D962E722E2EL, 0x7A29627A4BDD4B4BL,
		0x215DA321FE1FFEFEL, 0x16D5821657F95757L,
		0x41BDA841153F1515L, 0xB6E89FB677997777L,
		0xEB92A5EB37593737L, 0x569E7B56E532E5E5L,
		0xD9138CD99FBC9F9FL, 0x1723D317F00DF0F0L,
		0x7F206A7F4ADE4A4AL, 0x95449E95DA73DADAL,
		0x25A2FA2558E85858L, 0xCACF06CAC946C9C9L,
		0x8D7C558D297B2929L, 0x225A50220A1E0A0AL,
		0x4F50E14FB1CEB1B1L, 0x1AC9691AA0FDA0A0L,
		0xDA147FDA6BBD6B6BL, 0xABD95CAB85928585L,
		0x733C8173BDDABDBDL, 0x348FD2345DE75D5DL,
		0x5090805010301010L, 0x0307F303F401F4F4L,
		0xC0DD16C0CB40CBCBL, 0xC6D3EDC63E423E3EL,
		0x112D2811050F0505L, 0xE6781FE667A96767L,
		0x53977353E431E4E4L, 0xBB0225BB27692727L,
		0x5873325841C34141L, 0x9DA72C9D8B808B8BL,
		0x01F65101A7F4A7A7L, 0x94B2CF947D877D7DL,
		0xFB49DCFB95A29595L, 0x9F568E9FD875D8D8L,
		0x30708B30FB10FBFBL, 0x71CD2371EE2FEEEEL,
		0x91BBC7917C847C7CL, 0xE37117E366AA6666L,
		0x8E7BA68EDD7ADDDDL, 0x4BAFB84B17391717L,
		0x4645024647C94747L, 0xDC1A84DC9EBF9E9EL,
		0xC5D41EC5CA43CACAL, 0x995875992D772D2DL,
		0x792E9179BFDCBFBFL, 0x1B3F381B07090707L,
		0x23AC0123ADEAADADL, 0x2FB0EA2F5AEE5A5AL,
		0xB5EF6CB583988383L, 0xFFB685FF33553333L,
		0xF25C3FF263A56363L, 0x0A12100A02060202L,
		0x38933938AAE3AAAAL, 0xA8DEAFA871937171L,
		0xCFC60ECFC845C8C8L, 0x7DD1C87D192B1919L,
		0x703B727049DB4949L, 0x9A5F869AD976D9D9L,
		0x1D31C31DF20BF2F2L, 0x48A84B48E338E3E3L,
		0x2AB9E22A5BED5B5BL, 0x92BC349288858888L,
		0xC83EA4C89AB39A9AL, 0xBE0B2DBE266A2626L,
		0xFABF8DFA32563232L, 0x4A59E94AB0CDB0B0L,
		0x6AF21B6AE926E9E9L, 0x337778330F110F0FL,
		0xA633E6A6D562D5D5L, 0xBAF474BA809D8080L,
		0x7C27997CBEDFBEBEL, 0xDEEB26DECD4ACDCDL,
		0xE489BDE4345C3434L, 0x75327A7548D84848L,
		0x2454AB24FF1CFFFFL, 0x8F8DF78F7A8E7A7AL,
		0xEA64F4EA90AD9090L, 0x3E9DC23E5FE15F5FL,
		0xA03D1DA020602020L, 0xD50F67D568B86868L,
		0x72CAD0721A2E1A1AL, 0x2CB7192CAEEFAEAEL,
		0x5E7DC95EB4C1B4B4L, 0x19CE9A1954FC5454L,
		0xE57FECE593A89393L, 0xAA2F0DAA22662222L,
		0xE96307E964AC6464L, 0x122ADB12F10EF1F1L,
		0xA2CCBFA273957373L, 0x5A82905A12361212L,
		0x5D7A3A5D40C04040L, 0x2848402808180808L,
		0xE89556E8C358C3C3L, 0x7BDF337BEC29ECECL,
		0x904D9690DB70DBDBL, 0x1FC0611FA1FEA1A1L,
		0x83911C838D8A8D8DL, 0xC9C8F5C93D473D3DL,
		0xF15BCCF197A49797L, 0x0000000000000000L,
		0xD4F936D4CF4CCFCFL, 0x876E45872B7D2B2BL,
		0xB3E197B3769A7676L, 0xB0E664B0829B8282L,
		0xA928FEA9D667D6D6L, 0x77C3D8771B2D1B1BL,
		0x5B74C15BB5C2B5B5L, 0x29BE1129AFECAFAFL,
		0xDF1D77DF6ABE6A6AL, 0x0DEABA0D50F05050L,
		0x4C57124C45CF4545L, 0x1838CB18F308F3F3L,
		0xF0AD9DF030503030L, 0x74C42B74EF2CEFEFL,
		0xC3DAE5C33F413F3FL, 0x1CC7921C55FF5555L,
		0x10DB7910A2FBA2A2L, 0x65E90365EA23EAEAL,
		0xEC6A0FEC65AF6565L, 0x6803B968BAD3BABAL,
		0x934A65932F712F2FL, 0xE78E4EE7C05DC0C0L,
		0x8160BE81DE7FDEDEL, 0x6CFCE06C1C241C1CL,
		0x2E46BB2EFD1AFDFDL, 0x641F52644DD74D4DL,
		0xE076E4E092AB9292L, 0xBCFA8FBC759F7575L,
		0x1E36301E060A0606L, 0x98AE24988A838A8AL,
		0x404BF940B2CBB2B2L, 0x59856359E637E6E6L,
		0x367E70360E120E0EL, 0x63E7F8631F211F1FL,
		0xF75537F762A66262L, 0xA33AEEA3D461D4D4L,
		0x32812932A8E5A8A8L, 0xF452C4F496A79696L,
		0x3A629B3AF916F9F9L, 0xF6A366F6C552C5C5L,
		0xB11035B1256F2525L, 0x20ABF22059EB5959L,
		0xAED054AE84918484L, 0xA7C5B7A772967272L,
		0xDDECD5DD394B3939L, 0x61165A614CD44C4CL,
		0x3B94CA3B5EE25E5EL, 0x859FE78578887878L,
		0xD8E5DDD838483838L, 0x869814868C898C8CL,
		0xB217C6B2D16ED1D1L, 0x0BE4410BA5F2A5A5L,
		0x4DA1434DE23BE2E2L, 0xF84E2FF861A36161L,
		0x4542F145B3C8B3B3L, 0xA53415A521632121L,
		0xD60894D69CB99C9CL, 0x66EEF0661E221E1EL,
		0x5261225243C54343L, 0xFCB176FCC754C7C7L,
		0x2B4FB32BFC19FCFCL, 0x14242014040C0404L,
		0x08E3B20851F35151L, 0xC725BCC799B69999L,
		0xC4224FC46DB76D6DL, 0x396568390D170D0DL,
		0x35798335FA13FAFAL, 0x8469B684DF7CDFDFL,
		0x9BA9D79B7E827E7EL, 0xB4193DB4246C2424L,
		0xD7FEC5D73B4D3B3BL, 0x3D9A313DABE0ABABL,
		0xD1F03ED1CE4FCECEL, 0x5599885511331111L,
		0x89830C898F8C8F8FL, 0x6B044A6B4ED24E4EL,
		0x5166D151B7C4B7B7L, 0x60E00B60EB20EBEBL,
		0xCCC1FDCC3C443C3CL, 0xBFFD7CBF819E8181L,
		0xFE40D4FE94A19494L, 0x0C1CEB0CF704F7F7L,
		0x6718A167B9D6B9B9L, 0x5F8B985F13351313L,
		0x9C517D9C2C742C2CL, 0xB805D6B8D368D3D3L,
		0x5C8C6B5CE734E7E7L, 0xCB3957CB6EB26E6EL,
		0xF3AA6EF3C451C4C4L, 0x0F1B180F03050303L,
		0x13DC8A1356FA5656L, 0x495E1A4944CC4444L,
		0x9EA0DF9E7F817F7FL, 0x37882137A9E6A9A9L,
		0x82674D822A7E2A2AL, 0x6D0AB16DBBD0BBBBL,
		0xE28746E2C15EC1C1L, 0x02F1A20253F55353L,
		0x8B72AE8BDC79DCDCL, 0x275358270B1D0B0BL,
		0xD3019CD39DBA9D9DL, 0xC12B47C16CB46C6CL,
		0xF5A495F531533131L, 0xB9F387B9749C7474L,
		0x0915E309F607F6F6L, 0x434C0A4346CA4646L,
		0x26A50926ACE9ACACL, 0x97B53C9789868989L,
		0x44B4A044143C1414L, 0x42BA5B42E13EE1E1L,
		0x4EA6B04E163A1616L, 0xD2F7CDD23A4E3A3AL,
		0xD0066FD069BB6969L, 0x2D41482D091B0909L,
		0xADD7A7AD70907070L, 0x546FD954B6C7B6B6L,
		0xB71ECEB7D06DD0D0L, 0x7ED63B7EED2AEDEDL,
		0xDBE22EDBCC49CCCCL, 0x57682A5742C64242L,
		0xC22CB4C298B59898L, 0x0EED490EA4F1A4A4L,
		0x88755D8828782828L, 0x3186DA315CE45C5CL,
		0x3F6B933FF815F8F8L, 0xA4C244A486978686L
	};

	private static final long[] T1 = {
		0xD8C0781828181878L, 0x2605AF23652323AFL,
		0xB87EF9C657C6C6F9L, 0xFB136FE825E8E86FL,
		0xCB4CA187948787A1L, 0x11A962B8D5B8B862L,
		0x0908050103010105L, 0x0D426E4FD14F4F6EL,
		0x9BADEE365A3636EEL, 0xFF5904A6F7A6A604L,
		0x0CDEBDD26BD2D2BDL, 0x0EFB06F502F5F506L,
		0x96EF80798B797980L, 0x305FCE6FB16F6FCEL,
		0x6DFCEF91AE9191EFL, 0xF8AA0752F6525207L,
		0x4727FD60A06060FDL, 0x358976BCD9BCBC76L,
		0x37ACCD9BB09B9BCDL, 0x8A048C8E8F8E8E8CL,
		0xD27115A3F8A3A315L, 0x6C603C0C140C0C3CL,
		0x84FF8A7B8D7B7B8AL, 0x80B5E1355F3535E1L,
		0xF5E8691D271D1D69L, 0xB35347E03DE0E047L,
		0x21F6ACD764D7D7ACL, 0x9C5EEDC25BC2C2EDL,
		0x436D962E722E2E96L, 0x29627A4BDD4B4B7AL,
		0x5DA321FE1FFEFE21L, 0xD5821657F9575716L,
		0xBDA841153F151541L, 0xE89FB677997777B6L,
		0x92A5EB37593737EBL, 0x9E7B56E532E5E556L,
		0x138CD99FBC9F9FD9L, 0x23D317F00DF0F017L,
		0x206A7F4ADE4A4A7FL, 0x449E95DA73DADA95L,
		0xA2FA2558E8585825L, 0xCF06CAC946C9C9CAL,
		0x7C558D297B29298DL, 0x5A50220A1E0A0A22L,
		0x50E14FB1CEB1B14FL, 0xC9691AA0FDA0A01AL,
		0x147FDA6BBD6B6BDAL, 0xD95CAB85928585ABL,
		0x3C8173BDDABDBD73L, 0x8FD2345DE75D5D34L,
		0x9080501030101050L, 0x07F303F401F4F403L,
		0xDD16C0CB40CBCBC0L, 0xD3EDC63E423E3EC6L,
		0x2D2811050F050511L, 0x781FE667A96767E6L,
		0x977353E431E4E453L, 0x0225BB27692727BBL,
		0x73325841C3414158L, 0xA72C9D8B808B8B9DL,
		0xF65101A7F4A7A701L, 0xB2CF947D877D7D94L,
		0x49DCFB95A29595FBL, 0x568E9FD875D8D89FL,
		0x708B30FB10FBFB30L, 0xCD2371EE2FEEEE71L,
		0xBBC7917C847C7C91L, 0x7117E366AA6666E3L,
		0x7BA68EDD7ADDDD8EL, 0xAFB84B173917174BL,
		0x45024647C9474746L, 0x1A84DC9EBF9E9EDCL,
		0xD41EC5CA43CACAC5L, 0x5875992D772D2D99L,
		0x2E9179BFDCBFBF79L, 0x3F381B070907071BL,
		0xAC0123ADEAADAD23L, 0xB0EA2F5AEE5A5A2FL,
		0xEF6CB583988383B5L, 0xB685FF33553333FFL,
		0x5C3FF263A56363F2L, 0x12100A020602020AL,
		0x933938AAE3AAAA38L, 0xDEAFA871937171A8L,
		0xC60ECFC845C8C8CFL, 0xD1C87D192B19197DL,
		0x3B727049DB494970L, 0x5F869AD976D9D99AL,
		0x31C31DF20BF2F21DL, 0xA84B48E338E3E348L,
		0xB9E22A5BED5B5B2AL, 0xBC34928885888892L,
		0x3EA4C89AB39A9AC8L, 0x0B2DBE266A2626BEL,
		0xBF8DFA32563232FAL, 0x59E94AB0CDB0B04AL,
		0xF21B6AE926E9E96AL, 0x7778330F110F0F33L,
		0x33E6A6D562D5D5A6L, 0xF474BA809D8080BAL,
		0x27997CBEDFBEBE7CL, 0xEB26DECD4ACDCDDEL,
		0x89BDE4345C3434E4L, 0x327A7548D8484875L,
		0x54AB24FF1CFFFF24L, 0x8DF78F7A8E7A7A8FL,
		0x64F4EA90AD9090EAL, 0x9DC23E5FE15F5F3EL,
		0x3D1DA020602020A0L, 0x0F67D568B86868D5L,
		0xCAD0721A2E1A1A72L, 0xB7192CAEEFAEAE2CL,
		0x7DC95EB4C1B4B45EL, 0xCE9A1954FC545419L,
		0x7FECE593A89393E5L, 0x2F0DAA22662222AAL,
		0x6307E964AC6464E9L, 0x2ADB12F10EF1F112L,
		0xCCBFA273957373A2L, 0x82905A123612125AL,
		0x7A3A5D40C040405DL, 0x4840280818080828L,
		0x9556E8C358C3C3E8L, 0xDF337BEC29ECEC7BL,
		0x4D9690DB70DBDB90L, 0xC0611FA1FEA1A11FL,
		0x911C838D8A8D8D83L, 0xC8F5C93D473D3DC9L,
		0x5BCCF197A49797F1L, 0x0000000000000000L,
		0xF936D4CF4CCFCFD4L, 0x6E45872B7D2B2B87L,
		0xE197B3769A7676B3L, 0xE664B0829B8282B0L,
		0x28FEA9D667D6D6A9L, 0xC3D8771B2D1B1B77L,
		0x74C15BB5C2B5B55BL, 0xBE1129AFECAFAF29L,
		0x1D77DF6ABE6A6ADFL, 0xEABA0D50F050500DL,
		0x57124C45CF45454CL, 0x38CB18F308F3F318L,
		0xAD9DF030503030F0L, 0xC42B74EF2CEFEF74L,
		0xDAE5C33F413F3FC3L, 0xC7921C55FF55551CL,
		0xDB7910A2FBA2A210L, 0xE90365EA23EAEA65L,
		0x6A0FEC65AF6565ECL, 0x03B968BAD3BABA68L,
		0x4A65932F712F2F93L, 0x8E4EE7C05DC0C0E7L,
		0x60BE81DE7FDEDE81L, 0xFCE06C1C241C1C6CL,
		0x46BB2EFD1AFDFD2EL, 0x1F52644DD74D4D64L,
		0x76E4E092AB9292E0L, 0xFA8FBC759F7575BCL,
		0x36301E060A06061EL, 0xAE24988A838A8A98L,
		0x4BF940B2CBB2B240L, 0x856359E637E6E659L,
		0x7E70360E120E0E36L, 0xE7F8631F211F1F63L,
		0x5537F762A66262F7L, 0x3AEEA3D461D4D4A3L,
		0x812932A8E5A8A832L, 0x52C4F496A79696F4L,
		0x629B3AF916F9F93AL, 0xA366F6C552C5C5F6L,
		0x1035B1256F2525B1L, 0xABF22059EB595920L,
		0xD054AE84918484AEL, 0xC5B7A772967272A7L,
		0xECD5DD394B3939DDL, 0x165A614CD44C4C61L,
		0x94CA3B5EE25E5E3BL, 0x9FE7857888787885L,
		0xE5DDD838483838D8L, 0x9814868C898C8C86L,
		0x17C6B2D16ED1D1B2L, 0xE4410BA5F2A5A50BL,
		0xA1434DE23BE2E24DL, 0x4E2FF861A36161F8L,
		0x42F145B3C8B3B345L, 0x3415A521632121A5L,
		0x0894D69CB99C9CD6L, 0xEEF0661E221E1E66L,
		0x61225243C5434352L, 0xB176FCC754C7C7FCL,
		0x4FB32BFC19FCFC2BL, 0x242014040C040414L,
		0xE3B20851F3515108L, 0x25BCC799B69999C7L,
		0x224FC46DB76D6DC4L, 0x6568390D170D0D39L,
		0x798335FA13FAFA35L, 0x69B684DF7CDFDF84L,
		0xA9D79B7E827E7E9BL, 0x193DB4246C2424B4L,
		0xFEC5D73B4D3B3BD7L, 0x9A313DABE0ABAB3DL,
		0xF03ED1CE4FCECED1L, 0x9988551133111155L,
		0x830C898F8C8F8F89L, 0x044A6B4ED24E4E6BL,
		0x66D151B7C4B7B751L, 0xE00B60EB20EBEB60L,
		0xC1FDCC3C443C3CCCL, 0xFD7CBF819E8181BFL,
		0x40D4FE94A19494FEL, 0x1CEB0CF704F7F70CL,
		0x18A167B9D6B9B967L, 0x8B985F133513135FL,
		0x517D9C2C742C2C9CL, 0x05D6B8D368D3D3B8L,
		0x8C6B5CE734E7E75CL, 0x3957CB6EB26E6ECBL,
		0xAA6EF3C451C4C4F3L, 0x1B180F030503030FL,
		0xDC8A1356FA565613L, 0x5E1A4944CC444449L,
		0xA0DF9E7F817F7F9EL, 0x882137A9E6A9A937L,
		0x674D822A7E2A2A82L, 0x0AB16DBBD0BBBB6DL,
		0x8746E2C15EC1C1E2L, 0xF1A20253F5535302L,
		0x72AE8BDC79DCDC8BL, 0x5358270B1D0B0B27L,
		0x019CD39DBA9D9DD3L, 0x2B47C16CB46C6CC1L,
		0xA495F531533131F5L, 0xF387B9749C7474B9L,
		0x15E309F607F6F609L, 0x4C0A4346CA464643L,
		0xA50926ACE9ACAC26L, 0xB53C978986898997L,
		0xB4A044143C141444L, 0xBA5B42E13EE1E142L,
		0xA6B04E163A16164EL, 0xF7CDD23A4E3A3AD2L,
		0x066FD069BB6969D0L, 0x41482D091B09092DL,
		0xD7A7AD70907070ADL, 0x6FD954B6C7B6B654L,
		0x1ECEB7D06DD0D0B7L, 0xD63B7EED2AEDED7EL,
		0xE22EDBCC49CCCCDBL, 0x682A5742C6424257L,
		0x2CB4C298B59898C2L, 0xED490EA4F1A4A40EL,
		0x755D882878282888L, 0x86DA315CE45C5C31L,
		0x6B933FF815F8F83FL, 0xC244A486978686A4L
	};

	private static final long[] T2 = {
		0xC0781828181878D8L, 0x05AF23652323AF26L,
		0x7EF9C657C6C6F9B8L, 0x136FE825E8E86FFBL,
		0x4CA187948787A1CBL, 0xA962B8D5B8B86211L,
		0x0805010301010509L, 0x426E4FD14F4F6E0DL,
		0xADEE365A3636EE9BL, 0x5904A6F7A6A604FFL,
		0xDEBDD26BD2D2BD0CL, 0xFB06F502F5F5060EL,
		0xEF80798B79798096L, 0x5FCE6FB16F6FCE30L,
		0xFCEF91AE9191EF6DL, 0xAA0752F6525207F8L,
		0x27FD60A06060FD47L, 0x8976BCD9BCBC7635L,
		0xACCD9BB09B9BCD37L, 0x048C8E8F8E8E8C8AL,
		0x7115A3F8A3A315D2L, 0x603C0C140C0C3C6CL,
		0xFF8A7B8D7B7B8A84L, 0xB5E1355F3535E180L,
		0xE8691D271D1D69F5L, 0x5347E03DE0E047B3L,
		0xF6ACD764D7D7AC21L, 0x5EEDC25BC2C2ED9CL,
		0x6D962E722E2E9643L, 0x627A4BDD4B4B7A29L,
		0xA321FE1FFEFE215DL, 0x821657F9575716D5L,
		0xA841153F151541BDL, 0x9FB677997777B6E8L,
		0xA5EB37593737EB92L, 0x7B56E532E5E5569EL,
		0x8CD99FBC9F9FD913L, 0xD317F00DF0F01723L,
		0x6A7F4ADE4A4A7F20L, 0x9E95DA73DADA9544L,
		0xFA2558E8585825A2L, 0x06CAC946C9C9CACFL,
		0x558D297B29298D7CL, 0x50220A1E0A0A225AL,
		0xE14FB1CEB1B14F50L, 0x691AA0FDA0A01AC9L,
		0x7FDA6BBD6B6BDA14L, 0x5CAB85928585ABD9L,
		0x8173BDDABDBD733CL, 0xD2345DE75D5D348FL,
		0x8050103010105090L, 0xF303F401F4F40307L,
		0x16C0CB40CBCBC0DDL, 0xEDC63E423E3EC6D3L,
		0x2811050F0505112DL, 0x1FE667A96767E678L,
		0x7353E431E4E45397L, 0x25BB27692727BB02L,
		0x325841C341415873L, 0x2C9D8B808B8B9DA7L,
		0x5101A7F4A7A701F6L, 0xCF947D877D7D94B2L,
		0xDCFB95A29595FB49L, 0x8E9FD875D8D89F56L,
		0x8B30FB10FBFB3070L, 0x2371EE2FEEEE71CDL,
		0xC7917C847C7C91BBL, 0x17E366AA6666E371L,
		0xA68EDD7ADDDD8E7BL, 0xB84B173917174BAFL,
		0x024647C947474645L, 0x84DC9EBF9E9EDC1AL,
		0x1EC5CA43CACAC5D4L, 0x75992D772D2D9958L,
		0x9179BFDCBFBF792EL, 0x381B070907071B3FL,
		0x0123ADEAADAD23ACL, 0xEA2F5AEE5A5A2FB0L,
		0x6CB583988383B5EFL, 0x85FF33553333FFB6L,
		0x3FF263A56363F25CL, 0x100A020602020A12L,
		0x3938AAE3AAAA3893L, 0xAFA871937171A8DEL,
		0x0ECFC845C8C8CFC6L, 0xC87D192B19197DD1L,
		0x727049DB4949703BL, 0x869AD976D9D99A5FL,
		0xC31DF20BF2F21D31L, 0x4B48E338E3E348A8L,
		0xE22A5BED5B5B2AB9L, 0x34928885888892BCL,
		0xA4C89AB39A9AC83EL, 0x2DBE266A2626BE0BL,
		0x8DFA32563232FABFL, 0xE94AB0CDB0B04A59L,
		0x1B6AE926E9E96AF2L, 0x78330F110F0F3377L,
		0xE6A6D562D5D5A633L, 0x74BA809D8080BAF4L,
		0x997CBEDFBEBE7C27L, 0x26DECD4ACDCDDEEBL,
		0xBDE4345C3434E489L, 0x7A7548D848487532L,
		0xAB24FF1CFFFF2454L, 0xF78F7A8E7A7A8F8DL,
		0xF4EA90AD9090EA64L, 0xC23E5FE15F5F3E9DL,
		0x1DA020602020A03DL, 0x67D568B86868D50FL,
		0xD0721A2E1A1A72CAL, 0x192CAEEFAEAE2CB7L,
		0xC95EB4C1B4B45E7DL, 0x9A1954FC545419CEL,
		0xECE593A89393E57FL, 0x0DAA22662222AA2FL,
		0x07E964AC6464E963L, 0xDB12F10EF1F1122AL,
		0xBFA273957373A2CCL, 0x905A123612125A82L,
		0x3A5D40C040405D7AL, 0x4028081808082848L,
		0x56E8C358C3C3E895L, 0x337BEC29ECEC7BDFL,
		0x9690DB70DBDB904DL, 0x611FA1FEA1A11FC0L,
		0x1C838D8A8D8D8391L, 0xF5C93D473D3DC9C8L,
		0xCCF197A49797F15BL, 0x0000000000000000L,
		0x36D4CF4CCFCFD4F9L, 0x45872B7D2B2B876EL,
		0x97B3769A7676B3E1L, 0x64B0829B8282B0E6L,
		0xFEA9D667D6D6A928L, 0xD8771B2D1B1B77C3L,
		0xC15BB5C2B5B55B74L, 0x1129AFECAFAF29BEL,
		0x77DF6ABE6A6ADF1DL, 0xBA0D50F050500DEAL,
		0x124C45CF45454C57L, 0xCB18F308F3F31838L,
		0x9DF030503030F0ADL, 0x2B74EF2CEFEF74C4L,
		0xE5C33F413F3FC3DAL, 0x921C55FF55551CC7L,
		0x7910A2FBA2A210DBL, 0x0365EA23EAEA65E9L,
		0x0FEC65AF6565EC6AL, 0xB968BAD3BABA6803L,
		0x65932F712F2F934AL, 0x4EE7C05DC0C0E78EL,
		0xBE81DE7FDEDE8160L, 0xE06C1C241C1C6CFCL,
		0xBB2EFD1AFDFD2E46L, 0x52644DD74D4D641FL,
		0xE4E092AB9292E076L, 0x8FBC759F7575BCFAL,
		0x301E060A06061E36L, 0x24988A838A8A98AEL,
		0xF940B2CBB2B2404BL, 0x6359E637E6E65985L,
		0x70360E120E0E367EL, 0xF8631F211F1F63E7L,
		0x37F762A66262F755L, 0xEEA3D461D4D4A33AL,
		0x2932A8E5A8A83281L, 0xC4F496A79696F452L,
		0x9B3AF916F9F93A62L, 0x66F6C552C5C5F6A3L,
		0x35B1256F2525B110L, 0xF22059EB595920ABL,
		0x54AE84918484AED0L, 0xB7A772967272A7C5L,
		0xD5DD394B3939DDECL, 0x5A614CD44C4C6116L,
		0xCA3B5EE25E5E3B94L, 0xE78578887878859FL,
		0xDDD838483838D8E5L, 0x14868C898C8C8698L,
		0xC6B2D16ED1D1B217L, 0x410BA5F2A5A50BE4L,
		0x434DE23BE2E24DA1L, 0x2FF861A36161F84EL,
		0xF145B3C8B3B34542L, 0x15A521632121A534L,
		0x94D69CB99C9CD608L, 0xF0661E221E1E66EEL,
		0x225243C543435261L, 0x76FCC754C7C7FCB1L,
		0xB32BFC19FCFC2B4FL, 0x2014040C04041424L,
		0xB20851F3515108E3L, 0xBCC799B69999C725L,
		0x4FC46DB76D6DC422L, 0x68390D170D0D3965L,
		0x8335FA13FAFA3579L, 0xB684DF7CDFDF8469L,
		0xD79B7E827E7E9BA9L, 0x3DB4246C2424B419L,
		0xC5D73B4D3B3BD7FEL, 0x313DABE0ABAB3D9AL,
		0x3ED1CE4FCECED1F0L, 0x8855113311115599L,
		0x0C898F8C8F8F8983L, 0x4A6B4ED24E4E6B04L,
		0xD151B7C4B7B75166L, 0x0B60EB20EBEB60E0L,
		0xFDCC3C443C3CCCC1L, 0x7CBF819E8181BFFDL,
		0xD4FE94A19494FE40L, 0xEB0CF704F7F70C1CL,
		0xA167B9D6B9B96718L, 0x985F133513135F8BL,
		0x7D9C2C742C2C9C51L, 0xD6B8D368D3D3B805L,
		0x6B5CE734E7E75C8CL, 0x57CB6EB26E6ECB39L,
		0x6EF3C451C4C4F3AAL, 0x180F030503030F1BL,
		0x8A1356FA565613DCL, 0x1A4944CC4444495EL,
		0xDF9E7F817F7F9EA0L, 0x2137A9E6A9A93788L,
		0x4D822A7E2A2A8267L, 0xB16DBBD0BBBB6D0AL,
		0x46E2C15EC1C1E287L, 0xA20253F5535302F1L,
		0xAE8BDC79DCDC8B72L, 0x58270B1D0B0B2753L,
		0x9CD39DBA9D9DD301L, 0x47C16CB46C6CC12BL,
		0x95F531533131F5A4L, 0x87B9749C7474B9F3L,
		0xE309F607F6F60915L, 0x0A4346CA4646434CL,
		0x0926ACE9ACAC26A5L, 0x3C978986898997B5L,
		0xA044143C141444B4L, 0x5B42E13EE1E142BAL,
		0xB04E163A16164EA6L, 0xCDD23A4E3A3AD2F7L,
		0x6FD069BB6969D006L, 0x482D091B09092D41L,
		0xA7AD70907070ADD7L, 0xD954B6C7B6B6546FL,
		0xCEB7D06DD0D0B71EL, 0x3B7EED2AEDED7ED6L,
		0x2EDBCC49CCCCDBE2L, 0x2A5742C642425768L,
		0xB4C298B59898C22CL, 0x490EA4F1A4A40EEDL,
		0x5D88287828288875L, 0xDA315CE45C5C3186L,
		0x933FF815F8F83F6BL, 0x44A486978686A4C2L
	};

	private static final long[] T3 = {
		0x781828181878D8C0L, 0xAF23652323AF2605L,
		0xF9C657C6C6F9B87EL, 0x6FE825E8E86FFB13L,
		0xA187948787A1CB4CL, 0x62B8D5B8B86211A9L,
		0x0501030101050908L, 0x6E4FD14F4F6E0D42L,
		0xEE365A3636EE9BADL, 0x04A6F7A6A604FF59L,
		0xBDD26BD2D2BD0CDEL, 0x06F502F5F5060EFBL,
		0x80798B79798096EFL, 0xCE6FB16F6FCE305FL,
		0xEF91AE9191EF6DFCL, 0x0752F6525207F8AAL,
		0xFD60A06060FD4727L, 0x76BCD9BCBC763589L,
		0xCD9BB09B9BCD37ACL, 0x8C8E8F8E8E8C8A04L,
		0x15A3F8A3A315D271L, 0x3C0C140C0C3C6C60L,
		0x8A7B8D7B7B8A84FFL, 0xE1355F3535E180B5L,
		0x691D271D1D69F5E8L, 0x47E03DE0E047B353L,
		0xACD764D7D7AC21F6L, 0xEDC25BC2C2ED9C5EL,
		0x962E722E2E96436DL, 0x7A4BDD4B4B7A2962L,
		0x21FE1FFEFE215DA3L, 0x1657F9575716D582L,
		0x41153F151541BDA8L, 0xB677997777B6E89FL,
		0xEB37593737EB92A5L, 0x56E532E5E5569E7BL,
		0xD99FBC9F9FD9138CL, 0x17F00DF0F01723D3L,
		0x7F4ADE4A4A7F206AL, 0x95DA73DADA95449EL,
		0x2558E8585825A2FAL, 0xCAC946C9C9CACF06L,
		0x8D297B29298D7C55L, 0x220A1E0A0A225A50L,
		0x4FB1CEB1B14F50E1L, 0x1AA0FDA0A01AC969L,
		0xDA6BBD6B6BDA147FL, 0xAB85928585ABD95CL,
		0x73BDDABDBD733C81L, 0x345DE75D5D348FD2L,
		0x5010301010509080L, 0x03F401F4F40307F3L,
		0xC0CB40CBCBC0DD16L, 0xC63E423E3EC6D3EDL,
		0x11050F0505112D28L, 0xE667A96767E6781FL,
		0x53E431E4E4539773L, 0xBB27692727BB0225L,
		0x5841C34141587332L, 0x9D8B808B8B9DA72CL,
		0x01A7F4A7A701F651L, 0x947D877D7D94B2CFL,
		0xFB95A29595FB49DCL, 0x9FD875D8D89F568EL,
		0x30FB10FBFB30708BL, 0x71EE2FEEEE71CD23L,
		0x917C847C7C91BBC7L, 0xE366AA6666E37117L,
		0x8EDD7ADDDD8E7BA6L, 0x4B173917174BAFB8L,
		0x4647C94747464502L, 0xDC9EBF9E9EDC1A84L,
		0xC5CA43CACAC5D41EL, 0x992D772D2D995875L,
		0x79BFDCBFBF792E91L, 0x1B070907071B3F38L,
		0x23ADEAADAD23AC01L, 0x2F5AEE5A5A2FB0EAL,
		0xB583988383B5EF6CL, 0xFF33553333FFB685L,
		0xF263A56363F25C3FL, 0x0A020602020A1210L,
		0x38AAE3AAAA389339L, 0xA871937171A8DEAFL,
		0xCFC845C8C8CFC60EL, 0x7D192B19197DD1C8L,
		0x7049DB4949703B72L, 0x9AD976D9D99A5F86L,
		0x1DF20BF2F21D31C3L, 0x48E338E3E348A84BL,
		0x2A5BED5B5B2AB9E2L, 0x928885888892BC34L,
		0xC89AB39A9AC83EA4L, 0xBE266A2626BE0B2DL,
		0xFA32563232FABF8DL, 0x4AB0CDB0B04A59E9L,
		0x6AE926E9E96AF21BL, 0x330F110F0F337778L,
		0xA6D562D5D5A633E6L, 0xBA809D8080BAF474L,
		0x7CBEDFBEBE7C2799L, 0xDECD4ACDCDDEEB26L,
		0xE4345C3434E489BDL, 0x7548D8484875327AL,
		0x24FF1CFFFF2454ABL, 0x8F7A8E7A7A8F8DF7L,
		0xEA90AD9090EA64F4L, 0x3E5FE15F5F3E9DC2L,
		0xA020602020A03D1DL, 0xD568B86868D50F67L,
		0x721A2E1A1A72CAD0L, 0x2CAEEFAEAE2CB719L,
		0x5EB4C1B4B45E7DC9L, 0x1954FC545419CE9AL,
		0xE593A89393E57FECL, 0xAA22662222AA2F0DL,
		0xE964AC6464E96307L, 0x12F10EF1F1122ADBL,
		0xA273957373A2CCBFL, 0x5A123612125A8290L,
		0x5D40C040405D7A3AL, 0x2808180808284840L,
		0xE8C358C3C3E89556L, 0x7BEC29ECEC7BDF33L,
		0x90DB70DBDB904D96L, 0x1FA1FEA1A11FC061L,
		0x838D8A8D8D83911CL, 0xC93D473D3DC9C8F5L,
		0xF197A49797F15BCCL, 0x0000000000000000L,
		0xD4CF4CCFCFD4F936L, 0x872B7D2B2B876E45L,
		0xB3769A7676B3E197L, 0xB0829B8282B0E664L,
		0xA9D667D6D6A928FEL, 0x771B2D1B1B77C3D8L,
		0x5BB5C2B5B55B74C1L, 0x29AFECAFAF29BE11L,
		0xDF6ABE6A6ADF1D77L, 0x0D50F050500DEABAL,
		0x4C45CF45454C5712L, 0x18F308F3F31838CBL,
		0xF030503030F0AD9DL, 0x74EF2CEFEF74C42BL,
		0xC33F413F3FC3DAE5L, 0x1C55FF55551CC792L,
		0x10A2FBA2A210DB79L, 0x65EA23EAEA65E903L,
		0xEC65AF6565EC6A0FL, 0x68BAD3BABA6803B9L,
		0x932F712F2F934A65L, 0xE7C05DC0C0E78E4EL,
		0x81DE7FDEDE8160BEL, 0x6C1C241C1C6CFCE0L,
		0x2EFD1AFDFD2E46BBL, 0x644DD74D4D641F52L,
		0xE092AB9292E076E4L, 0xBC759F7575BCFA8FL,
		0x1E060A06061E3630L, 0x988A838A8A98AE24L,
		0x40B2CBB2B2404BF9L, 0x59E637E6E6598563L,
		0x360E120E0E367E70L, 0x631F211F1F63E7F8L,
		0xF762A66262F75537L, 0xA3D461D4D4A33AEEL,
		0x32A8E5A8A8328129L, 0xF496A79696F452C4L,
		0x3AF916F9F93A629BL, 0xF6C552C5C5F6A366L,
		0xB1256F2525B11035L, 0x2059EB595920ABF2L,
		0xAE84918484AED054L, 0xA772967272A7C5B7L,
		0xDD394B3939DDECD5L, 0x614CD44C4C61165AL,
		0x3B5EE25E5E3B94CAL, 0x8578887878859FE7L,
		0xD838483838D8E5DDL, 0x868C898C8C869814L,
		0xB2D16ED1D1B217C6L, 0x0BA5F2A5A50BE441L,
		0x4DE23BE2E24DA143L, 0xF861A36161F84E2FL,
		0x45B3C8B3B34542F1L, 0xA521632121A53415L,
		0xD69CB99C9CD60894L, 0x661E221E1E66EEF0L,
		0x5243C54343526122L, 0xFCC754C7C7FCB176L,
		0x2BFC19FCFC2B4FB3L, 0x14040C0404142420L,
		0x0851F3515108E3B2L, 0xC799B69999C725BCL,
		0xC46DB76D6DC4224FL, 0x390D170D0D396568L,
		0x35FA13FAFA357983L, 0x84DF7CDFDF8469B6L,
		0x9B7E827E7E9BA9D7L, 0xB4246C2424B4193DL,
		0xD73B4D3B3BD7FEC5L, 0x3DABE0ABAB3D9A31L,
		0xD1CE4FCECED1F03EL, 0x5511331111559988L,
		0x898F8C8F8F89830CL, 0x6B4ED24E4E6B044AL,
		0x51B7C4B7B75166D1L, 0x60EB20EBEB60E00BL,
		0xCC3C443C3CCCC1FDL, 0xBF819E8181BFFD7CL,
		0xFE94A19494FE40D4L, 0x0CF704F7F70C1CEBL,
		0x67B9D6B9B96718A1L, 0x5F133513135F8B98L,
		0x9C2C742C2C9C517DL, 0xB8D368D3D3B805D6L,
		0x5CE734E7E75C8C6BL, 0xCB6EB26E6ECB3957L,
		0xF3C451C4C4F3AA6EL, 0x0F030503030F1B18L,
		0x1356FA565613DC8AL, 0x4944CC4444495E1AL,
		0x9E7F817F7F9EA0DFL, 0x37A9E6A9A9378821L,
		0x822A7E2A2A82674DL, 0x6DBBD0BBBB6D0AB1L,
		0xE2C15EC1C1E28746L, 0x0253F5535302F1A2L,
		0x8BDC79DCDC8B72AEL, 0x270B1D0B0B275358L,
		0xD39DBA9D9DD3019CL, 0xC16CB46C6CC12B47L,
		0xF531533131F5A495L, 0xB9749C7474B9F387L,
		0x09F607F6F60915E3L, 0x4346CA4646434C0AL,
		0x26ACE9ACAC26A509L, 0x978986898997B53CL,
		0x44143C141444B4A0L, 0x42E13EE1E142BA5BL,
		0x4E163A16164EA6B0L, 0xD23A4E3A3AD2F7CDL,
		0xD069BB6969D0066FL, 0x2D091B09092D4148L,
		0xAD70907070ADD7A7L, 0x54B6C7B6B6546FD9L,
		0xB7D06DD0D0B71ECEL, 0x7EED2AEDED7ED63BL,
		0xDBCC49CCCCDBE22EL, 0x5742C6424257682AL,
		0xC298B59898C22CB4L, 0x0EA4F1A4A40EED49L,
		0x882878282888755DL, 0x315CE45C5C3186DAL,
		0x3FF815F8F83F6B93L, 0xA486978686A4C244L
	};

	private static final long[] T4 = {
		0x1828181878D8C078L, 0x23652323AF2605AFL,
		0xC657C6C6F9B87EF9L, 0xE825E8E86FFB136FL,
		0x87948787A1CB4CA1L, 0xB8D5B8B86211A962L,
		0x0103010105090805L, 0x4FD14F4F6E0D426EL,
		0x365A3636EE9BADEEL, 0xA6F7A6A604FF5904L,
		0xD26BD2D2BD0CDEBDL, 0xF502F5F5060EFB06L,
		0x798B79798096EF80L, 0x6FB16F6FCE305FCEL,
		0x91AE9191EF6DFCEFL, 0x52F6525207F8AA07L,
		0x60A06060FD4727FDL, 0xBCD9BCBC76358976L,
		0x9BB09B9BCD37ACCDL, 0x8E8F8E8E8C8A048CL,
		0xA3F8A3A315D27115L, 0x0C140C0C3C6C603CL,
		0x7B8D7B7B8A84FF8AL, 0x355F3535E180B5E1L,
		0x1D271D1D69F5E869L, 0xE03DE0E047B35347L,
		0xD764D7D7AC21F6ACL, 0xC25BC2C2ED9C5EEDL,
		0x2E722E2E96436D96L, 0x4BDD4B4B7A29627AL,
		0xFE1FFEFE215DA321L, 0x57F9575716D58216L,
		0x153F151541BDA841L, 0x77997777B6E89FB6L,
		0x37593737EB92A5EBL, 0xE532E5E5569E7B56L,
		0x9FBC9F9FD9138CD9L, 0xF00DF0F01723D317L,
		0x4ADE4A4A7F206A7FL, 0xDA73DADA95449E95L,
		0x58E8585825A2FA25L, 0xC946C9C9CACF06CAL,
		0x297B29298D7C558DL, 0x0A1E0A0A225A5022L,
		0xB1CEB1B14F50E14FL, 0xA0FDA0A01AC9691AL,
		0x6BBD6B6BDA147FDAL, 0x85928585ABD95CABL,
		0xBDDABDBD733C8173L, 0x5DE75D5D348FD234L,
		0x1030101050908050L, 0xF401F4F40307F303L,
		0xCB40CBCBC0DD16C0L, 0x3E423E3EC6D3EDC6L,
		0x050F0505112D2811L, 0x67A96767E6781FE6L,
		0xE431E4E453977353L, 0x27692727BB0225BBL,
		0x41C3414158733258L, 0x8B808B8B9DA72C9DL,
		0xA7F4A7A701F65101L, 0x7D877D7D94B2CF94L,
		0x95A29595FB49DCFBL, 0xD875D8D89F568E9FL,
		0xFB10FBFB30708B30L, 0xEE2FEEEE71CD2371L,
		0x7C847C7C91BBC791L, 0x66AA6666E37117E3L,
		0xDD7ADDDD8E7BA68EL, 0x173917174BAFB84BL,
		0x47C9474746450246L, 0x9EBF9E9EDC1A84DCL,
		0xCA43CACAC5D41EC5L, 0x2D772D2D99587599L,
		0xBFDCBFBF792E9179L, 0x070907071B3F381BL,
		0xADEAADAD23AC0123L, 0x5AEE5A5A2FB0EA2FL,
		0x83988383B5EF6CB5L, 0x33553333FFB685FFL,
		0x63A56363F25C3FF2L, 0x020602020A12100AL,
		0xAAE3AAAA38933938L, 0x71937171A8DEAFA8L,
		0xC845C8C8CFC60ECFL, 0x192B19197DD1C87DL,
		0x49DB4949703B7270L, 0xD976D9D99A5F869AL,
		0xF20BF2F21D31C31DL, 0xE338E3E348A84B48L,
		0x5BED5B5B2AB9E22AL, 0x8885888892BC3492L,
		0x9AB39A9AC83EA4C8L, 0x266A2626BE0B2DBEL,
		0x32563232FABF8DFAL, 0xB0CDB0B04A59E94AL,
		0xE926E9E96AF21B6AL, 0x0F110F0F33777833L,
		0xD562D5D5A633E6A6L, 0x809D8080BAF474BAL,
		0xBEDFBEBE7C27997CL, 0xCD4ACDCDDEEB26DEL,
		0x345C3434E489BDE4L, 0x48D8484875327A75L,
		0xFF1CFFFF2454AB24L, 0x7A8E7A7A8F8DF78FL,
		0x90AD9090EA64F4EAL, 0x5FE15F5F3E9DC23EL,
		0x20602020A03D1DA0L, 0x68B86868D50F67D5L,
		0x1A2E1A1A72CAD072L, 0xAEEFAEAE2CB7192CL,
		0xB4C1B4B45E7DC95EL, 0x54FC545419CE9A19L,
		0x93A89393E57FECE5L, 0x22662222AA2F0DAAL,
		0x64AC6464E96307E9L, 0xF10EF1F1122ADB12L,
		0x73957373A2CCBFA2L, 0x123612125A82905AL,
		0x40C040405D7A3A5DL, 0x0818080828484028L,
		0xC358C3C3E89556E8L, 0xEC29ECEC7BDF337BL,
		0xDB70DBDB904D9690L, 0xA1FEA1A11FC0611FL,
		0x8D8A8D8D83911C83L, 0x3D473D3DC9C8F5C9L,
		0x97A49797F15BCCF1L, 0x0000000000000000L,
		0xCF4CCFCFD4F936D4L, 0x2B7D2B2B876E4587L,
		0x769A7676B3E197B3L, 0x829B8282B0E664B0L,
		0xD667D6D6A928FEA9L, 0x1B2D1B1B77C3D877L,
		0xB5C2B5B55B74C15BL, 0xAFECAFAF29BE1129L,
		0x6ABE6A6ADF1D77DFL, 0x50F050500DEABA0DL,
		0x45CF45454C57124CL, 0xF308F3F31838CB18L,
		0x30503030F0AD9DF0L, 0xEF2CEFEF74C42B74L,
		0x3F413F3FC3DAE5C3L, 0x55FF55551CC7921CL,
		0xA2FBA2A210DB7910L, 0xEA23EAEA65E90365L,
		0x65AF6565EC6A0FECL, 0xBAD3BABA6803B968L,
		0x2F712F2F934A6593L, 0xC05DC0C0E78E4EE7L,
		0xDE7FDEDE8160BE81L, 0x1C241C1C6CFCE06CL,
		0xFD1AFDFD2E46BB2EL, 0x4DD74D4D641F5264L,
		0x92AB9292E076E4E0L, 0x759F7575BCFA8FBCL,
		0x060A06061E36301EL, 0x8A838A8A98AE2498L,
		0xB2CBB2B2404BF940L, 0xE637E6E659856359L,
		0x0E120E0E367E7036L, 0x1F211F1F63E7F863L,
		0x62A66262F75537F7L, 0xD461D4D4A33AEEA3L,
		0xA8E5A8A832812932L, 0x96A79696F452C4F4L,
		0xF916F9F93A629B3AL, 0xC552C5C5F6A366F6L,
		0x256F2525B11035B1L, 0x59EB595920ABF220L,
		0x84918484AED054AEL, 0x72967272A7C5B7A7L,
		0x394B3939DDECD5DDL, 0x4CD44C4C61165A61L,
		0x5EE25E5E3B94CA3BL, 0x78887878859FE785L,
		0x38483838D8E5DDD8L, 0x8C898C8C86981486L,
		0xD16ED1D1B217C6B2L, 0xA5F2A5A50BE4410BL,
		0xE23BE2E24DA1434DL, 0x61A36161F84E2FF8L,
		0xB3C8B3B34542F145L, 0x21632121A53415A5L,
		0x9CB99C9CD60894D6L, 0x1E221E1E66EEF066L,
		0x43C5434352612252L, 0xC754C7C7FCB176FCL,
		0xFC19FCFC2B4FB32BL, 0x040C040414242014L,
		0x51F3515108E3B208L, 0x99B69999C725BCC7L,
		0x6DB76D6DC4224FC4L, 0x0D170D0D39656839L,
		0xFA13FAFA35798335L, 0xDF7CDFDF8469B684L,
		0x7E827E7E9BA9D79BL, 0x246C2424B4193DB4L,
		0x3B4D3B3BD7FEC5D7L, 0xABE0ABAB3D9A313DL,
		0xCE4FCECED1F03ED1L, 0x1133111155998855L,
		0x8F8C8F8F89830C89L, 0x4ED24E4E6B044A6BL,
		0xB7C4B7B75166D151L, 0xEB20EBEB60E00B60L,
		0x3C443C3CCCC1FDCCL, 0x819E8181BFFD7CBFL,
		0x94A19494FE40D4FEL, 0xF704F7F70C1CEB0CL,
		0xB9D6B9B96718A167L, 0x133513135F8B985FL,
		0x2C742C2C9C517D9CL, 0xD368D3D3B805D6B8L,
		0xE734E7E75C8C6B5CL, 0x6EB26E6ECB3957CBL,
		0xC451C4C4F3AA6EF3L, 0x030503030F1B180FL,
		0x56FA565613DC8A13L, 0x44CC4444495E1A49L,
		0x7F817F7F9EA0DF9EL, 0xA9E6A9A937882137L,
		0x2A7E2A2A82674D82L, 0xBBD0BBBB6D0AB16DL,
		0xC15EC1C1E28746E2L, 0x53F5535302F1A202L,
		0xDC79DCDC8B72AE8BL, 0x0B1D0B0B27535827L,
		0x9DBA9D9DD3019CD3L, 0x6CB46C6CC12B47C1L,
		0x31533131F5A495F5L, 0x749C7474B9F387B9L,
		0xF607F6F60915E309L, 0x46CA4646434C0A43L,
		0xACE9ACAC26A50926L, 0x8986898997B53C97L,
		0x143C141444B4A044L, 0xE13EE1E142BA5B42L,
		0x163A16164EA6B04EL, 0x3A4E3A3AD2F7CDD2L,
		0x69BB6969D0066FD0L, 0x091B09092D41482DL,
		0x70907070ADD7A7ADL, 0xB6C7B6B6546FD954L,
		0xD06DD0D0B71ECEB7L, 0xED2AEDED7ED63B7EL,
		0xCC49CCCCDBE22EDBL, 0x42C6424257682A57L,
		0x98B59898C22CB4C2L, 0xA4F1A4A40EED490EL,
		0x2878282888755D88L, 0x5CE45C5C3186DA31L,
		0xF815F8F83F6B933FL, 0x86978686A4C244A4L
	};

	private static final long[] T5 = {
		0x28181878D8C07818L, 0x652323AF2605AF23L,
		0x57C6C6F9B87EF9C6L, 0x25E8E86FFB136FE8L,
		0x948787A1CB4CA187L, 0xD5B8B86211A962B8L,
		0x0301010509080501L, 0xD14F4F6E0D426E4FL,
		0x5A3636EE9BADEE36L, 0xF7A6A604FF5904A6L,
		0x6BD2D2BD0CDEBDD2L, 0x02F5F5060EFB06F5L,
		0x8B79798096EF8079L, 0xB16F6FCE305FCE6FL,
		0xAE9191EF6DFCEF91L, 0xF6525207F8AA0752L,
		0xA06060FD4727FD60L, 0xD9BCBC76358976BCL,
		0xB09B9BCD37ACCD9BL, 0x8F8E8E8C8A048C8EL,
		0xF8A3A315D27115A3L, 0x140C0C3C6C603C0CL,
		0x8D7B7B8A84FF8A7BL, 0x5F3535E180B5E135L,
		0x271D1D69F5E8691DL, 0x3DE0E047B35347E0L,
		0x64D7D7AC21F6ACD7L, 0x5BC2C2ED9C5EEDC2L,
		0x722E2E96436D962EL, 0xDD4B4B7A29627A4BL,
		0x1FFEFE215DA321FEL, 0xF9575716D5821657L,
		0x3F151541BDA84115L, 0x997777B6E89FB677L,
		0x593737EB92A5EB37L, 0x32E5E5569E7B56E5L,
		0xBC9F9FD9138CD99FL, 0x0DF0F01723D317F0L,
		0xDE4A4A7F206A7F4AL, 0x73DADA95449E95DAL,
		0xE8585825A2FA2558L, 0x46C9C9CACF06CAC9L,
		0x7B29298D7C558D29L, 0x1E0A0A225A50220AL,
		0xCEB1B14F50E14FB1L, 0xFDA0A01AC9691AA0L,
		0xBD6B6BDA147FDA6BL, 0x928585ABD95CAB85L,
		0xDABDBD733C8173BDL, 0xE75D5D348FD2345DL,
		0x3010105090805010L, 0x01F4F40307F303F4L,
		0x40CBCBC0DD16C0CBL, 0x423E3EC6D3EDC63EL,
		0x0F0505112D281105L, 0xA96767E6781FE667L,
		0x31E4E453977353E4L, 0x692727BB0225BB27L,
		0xC341415873325841L, 0x808B8B9DA72C9D8BL,
		0xF4A7A701F65101A7L, 0x877D7D94B2CF947DL,
		0xA29595FB49DCFB95L, 0x75D8D89F568E9FD8L,
		0x10FBFB30708B30FBL, 0x2FEEEE71CD2371EEL,
		0x847C7C91BBC7917CL, 0xAA6666E37117E366L,
		0x7ADDDD8E7BA68EDDL, 0x3917174BAFB84B17L,
		0xC947474645024647L, 0xBF9E9EDC1A84DC9EL,
		0x43CACAC5D41EC5CAL, 0x772D2D995875992DL,
		0xDCBFBF792E9179BFL, 0x0907071B3F381B07L,
		0xEAADAD23AC0123ADL, 0xEE5A5A2FB0EA2F5AL,
		0x988383B5EF6CB583L, 0x553333FFB685FF33L,
		0xA56363F25C3FF263L, 0x0602020A12100A02L,
		0xE3AAAA38933938AAL, 0x937171A8DEAFA871L,
		0x45C8C8CFC60ECFC8L, 0x2B19197DD1C87D19L,
		0xDB4949703B727049L, 0x76D9D99A5F869AD9L,
		0x0BF2F21D31C31DF2L, 0x38E3E348A84B48E3L,
		0xED5B5B2AB9E22A5BL, 0x85888892BC349288L,
		0xB39A9AC83EA4C89AL, 0x6A2626BE0B2DBE26L,
		0x563232FABF8DFA32L, 0xCDB0B04A59E94AB0L,
		0x26E9E96AF21B6AE9L, 0x110F0F337778330FL,
		0x62D5D5A633E6A6D5L, 0x9D8080BAF474BA80L,
		0xDFBEBE7C27997CBEL, 0x4ACDCDDEEB26DECDL,
		0x5C3434E489BDE434L, 0xD8484875327A7548L,
		0x1CFFFF2454AB24FFL, 0x8E7A7A8F8DF78F7AL,
		0xAD9090EA64F4EA90L, 0xE15F5F3E9DC23E5FL,
		0x602020A03D1DA020L, 0xB86868D50F67D568L,
		0x2E1A1A72CAD0721AL, 0xEFAEAE2CB7192CAEL,
		0xC1B4B45E7DC95EB4L, 0xFC545419CE9A1954L,
		0xA89393E57FECE593L, 0x662222AA2F0DAA22L,
		0xAC6464E96307E964L, 0x0EF1F1122ADB12F1L,
		0x957373A2CCBFA273L, 0x3612125A82905A12L,
		0xC040405D7A3A5D40L, 0x1808082848402808L,
		0x58C3C3E89556E8C3L, 0x29ECEC7BDF337BECL,
		0x70DBDB904D9690DBL, 0xFEA1A11FC0611FA1L,
		0x8A8D8D83911C838DL, 0x473D3DC9C8F5C93DL,
		0xA49797F15BCCF197L, 0x0000000000000000L,
		0x4CCFCFD4F936D4CFL, 0x7D2B2B876E45872BL,
		0x9A7676B3E197B376L, 0x9B8282B0E664B082L,
		0x67D6D6A928FEA9D6L, 0x2D1B1B77C3D8771BL,
		0xC2B5B55B74C15BB5L, 0xECAFAF29BE1129AFL,
		0xBE6A6ADF1D77DF6AL, 0xF050500DEABA0D50L,
		0xCF45454C57124C45L, 0x08F3F31838CB18F3L,
		0x503030F0AD9DF030L, 0x2CEFEF74C42B74EFL,
		0x413F3FC3DAE5C33FL, 0xFF55551CC7921C55L,
		0xFBA2A210DB7910A2L, 0x23EAEA65E90365EAL,
		0xAF6565EC6A0FEC65L, 0xD3BABA6803B968BAL,
		0x712F2F934A65932FL, 0x5DC0C0E78E4EE7C0L,
		0x7FDEDE8160BE81DEL, 0x241C1C6CFCE06C1CL,
		0x1AFDFD2E46BB2EFDL, 0xD74D4D641F52644DL,
		0xAB9292E076E4E092L, 0x9F7575BCFA8FBC75L,
		0x0A06061E36301E06L, 0x838A8A98AE24988AL,
		0xCBB2B2404BF940B2L, 0x37E6E659856359E6L,
		0x120E0E367E70360EL, 0x211F1F63E7F8631FL,
		0xA66262F75537F762L, 0x61D4D4A33AEEA3D4L,
		0xE5A8A832812932A8L, 0xA79696F452C4F496L,
		0x16F9F93A629B3AF9L, 0x52C5C5F6A366F6C5L,
		0x6F2525B11035B125L, 0xEB595920ABF22059L,
		0x918484AED054AE84L, 0x967272A7C5B7A772L,
		0x4B3939DDECD5DD39L, 0xD44C4C61165A614CL,
		0xE25E5E3B94CA3B5EL, 0x887878859FE78578L,
		0x483838D8E5DDD838L, 0x898C8C869814868CL,
		0x6ED1D1B217C6B2D1L, 0xF2A5A50BE4410BA5L,
		0x3BE2E24DA1434DE2L, 0xA36161F84E2FF861L,
		0xC8B3B34542F145B3L, 0x632121A53415A521L,
		0xB99C9CD60894D69CL, 0x221E1E66EEF0661EL,
		0xC543435261225243L, 0x54C7C7FCB176FCC7L,
		0x19FCFC2B4FB32BFCL, 0x0C04041424201404L,
		0xF3515108E3B20851L, 0xB69999C725BCC799L,
		0xB76D6DC4224FC46DL, 0x170D0D396568390DL,
		0x13FAFA35798335FAL, 0x7CDFDF8469B684DFL,
		0x827E7E9BA9D79B7EL, 0x6C2424B4193DB424L,
		0x4D3B3BD7FEC5D73BL, 0xE0ABAB3D9A313DABL,
		0x4FCECED1F03ED1CEL, 0x3311115599885511L,
		0x8C8F8F89830C898FL, 0xD24E4E6B044A6B4EL,
		0xC4B7B75166D151B7L, 0x20EBEB60E00B60EBL,
		0x443C3CCCC1FDCC3CL, 0x9E8181BFFD7CBF81L,
		0xA19494FE40D4FE94L, 0x04F7F70C1CEB0CF7L,
		0xD6B9B96718A167B9L, 0x3513135F8B985F13L,
		0x742C2C9C517D9C2CL, 0x68D3D3B805D6B8D3L,
		0x34E7E75C8C6B5CE7L, 0xB26E6ECB3957CB6EL,
		0x51C4C4F3AA6EF3C4L, 0x0503030F1B180F03L,
		0xFA565613DC8A1356L, 0xCC4444495E1A4944L,
		0x817F7F9EA0DF9E7FL, 0xE6A9A937882137A9L,
		0x7E2A2A82674D822AL, 0xD0BBBB6D0AB16DBBL,
		0x5EC1C1E28746E2C1L, 0xF5535302F1A20253L,
		0x79DCDC8B72AE8BDCL, 0x1D0B0B275358270BL,
		0xBA9D9DD3019CD39DL, 0xB46C6CC12B47C16CL,
		0x533131F5A495F531L, 0x9C7474B9F387B974L,
		0x07F6F60915E309F6L, 0xCA4646434C0A4346L,
		0xE9ACAC26A50926ACL, 0x86898997B53C9789L,
		0x3C141444B4A04414L, 0x3EE1E142BA5B42E1L,
		0x3A16164EA6B04E16L, 0x4E3A3AD2F7CDD23AL,
		0xBB6969D0066FD069L, 0x1B09092D41482D09L,
		0x907070ADD7A7AD70L, 0xC7B6B6546FD954B6L,
		0x6DD0D0B71ECEB7D0L, 0x2AEDED7ED63B7EEDL,
		0x49CCCCDBE22EDBCCL, 0xC6424257682A5742L,
		0xB59898C22CB4C298L, 0xF1A4A40EED490EA4L,
		0x78282888755D8828L, 0xE45C5C3186DA315CL,
		0x15F8F83F6B933FF8L, 0x978686A4C244A486L
	};

	private static final long[] T6 = {
		0x181878D8C0781828L, 0x2323AF2605AF2365L,
		0xC6C6F9B87EF9C657L, 0xE8E86FFB136FE825L,
		0x8787A1CB4CA18794L, 0xB8B86211A962B8D5L,
		0x0101050908050103L, 0x4F4F6E0D426E4FD1L,
		0x3636EE9BADEE365AL, 0xA6A604FF5904A6F7L,
		0xD2D2BD0CDEBDD26BL, 0xF5F5060EFB06F502L,
		0x79798096EF80798BL, 0x6F6FCE305FCE6FB1L,
		0x9191EF6DFCEF91AEL, 0x525207F8AA0752F6L,
		0x6060FD4727FD60A0L, 0xBCBC76358976BCD9L,
		0x9B9BCD37ACCD9BB0L, 0x8E8E8C8A048C8E8FL,
		0xA3A315D27115A3F8L, 0x0C0C3C6C603C0C14L,
		0x7B7B8A84FF8A7B8DL, 0x3535E180B5E1355FL,
		0x1D1D69F5E8691D27L, 0xE0E047B35347E03DL,
		0xD7D7AC21F6ACD764L, 0xC2C2ED9C5EEDC25BL,
		0x2E2E96436D962E72L, 0x4B4B7A29627A4BDDL,
		0xFEFE215DA321FE1FL, 0x575716D5821657F9L,
		0x151541BDA841153FL, 0x7777B6E89FB67799L,
		0x3737EB92A5EB3759L, 0xE5E5569E7B56E532L,
		0x9F9FD9138CD99FBCL, 0xF0F01723D317F00DL,
		0x4A4A7F206A7F4ADEL, 0xDADA95449E95DA73L,
		0x585825A2FA2558E8L, 0xC9C9CACF06CAC946L,
		0x29298D7C558D297BL, 0x0A0A225A50220A1EL,
		0xB1B14F50E14FB1CEL, 0xA0A01AC9691AA0FDL,
		0x6B6BDA147FDA6BBDL, 0x8585ABD95CAB8592L,
		0xBDBD733C8173BDDAL, 0x5D5D348FD2345DE7L,
		0x1010509080501030L, 0xF4F40307F303F401L,
		0xCBCBC0DD16C0CB40L, 0x3E3EC6D3EDC63E42L,
		0x0505112D2811050FL, 0x6767E6781FE667A9L,
		0xE4E453977353E431L, 0x2727BB0225BB2769L,
		0x41415873325841C3L, 0x8B8B9DA72C9D8B80L,
		0xA7A701F65101A7F4L, 0x7D7D94B2CF947D87L,
		0x9595FB49DCFB95A2L, 0xD8D89F568E9FD875L,
		0xFBFB30708B30FB10L, 0xEEEE71CD2371EE2FL,
		0x7C7C91BBC7917C84L, 0x6666E37117E366AAL,
		0xDDDD8E7BA68EDD7AL, 0x17174BAFB84B1739L,
		0x47474645024647C9L, 0x9E9EDC1A84DC9EBFL,
		0xCACAC5D41EC5CA43L, 0x2D2D995875992D77L,
		0xBFBF792E9179BFDCL, 0x07071B3F381B0709L,
		0xADAD23AC0123ADEAL, 0x5A5A2FB0EA2F5AEEL,
		0x8383B5EF6CB58398L, 0x3333FFB685FF3355L,
		0x6363F25C3FF263A5L, 0x02020A12100A0206L,
		0xAAAA38933938AAE3L, 0x7171A8DEAFA87193L,
		0xC8C8CFC60ECFC845L, 0x19197DD1C87D192BL,
		0x4949703B727049DBL, 0xD9D99A5F869AD976L,
		0xF2F21D31C31DF20BL, 0xE3E348A84B48E338L,
		0x5B5B2AB9E22A5BEDL, 0x888892BC34928885L,
		0x9A9AC83EA4C89AB3L, 0x2626BE0B2DBE266AL,
		0x3232FABF8DFA3256L, 0xB0B04A59E94AB0CDL,
		0xE9E96AF21B6AE926L, 0x0F0F337778330F11L,
		0xD5D5A633E6A6D562L, 0x8080BAF474BA809DL,
		0xBEBE7C27997CBEDFL, 0xCDCDDEEB26DECD4AL,
		0x3434E489BDE4345CL, 0x484875327A7548D8L,
		0xFFFF2454AB24FF1CL, 0x7A7A8F8DF78F7A8EL,
		0x9090EA64F4EA90ADL, 0x5F5F3E9DC23E5FE1L,
		0x2020A03D1DA02060L, 0x6868D50F67D568B8L,
		0x1A1A72CAD0721A2EL, 0xAEAE2CB7192CAEEFL,
		0xB4B45E7DC95EB4C1L, 0x545419CE9A1954FCL,
		0x9393E57FECE593A8L, 0x2222AA2F0DAA2266L,
		0x6464E96307E964ACL, 0xF1F1122ADB12F10EL,
		0x7373A2CCBFA27395L, 0x12125A82905A1236L,
		0x40405D7A3A5D40C0L, 0x0808284840280818L,
		0xC3C3E89556E8C358L, 0xECEC7BDF337BEC29L,
		0xDBDB904D9690DB70L, 0xA1A11FC0611FA1FEL,
		0x8D8D83911C838D8AL, 0x3D3DC9C8F5C93D47L,
		0x9797F15BCCF197A4L, 0x0000000000000000L,
		0xCFCFD4F936D4CF4CL, 0x2B2B876E45872B7DL,
		0x7676B3E197B3769AL, 0x8282B0E664B0829BL,
		0xD6D6A928FEA9D667L, 0x1B1B77C3D8771B2DL,
		0xB5B55B74C15BB5C2L, 0xAFAF29BE1129AFECL,
		0x6A6ADF1D77DF6ABEL, 0x50500DEABA0D50F0L,
		0x45454C57124C45CFL, 0xF3F31838CB18F308L,
		0x3030F0AD9DF03050L, 0xEFEF74C42B74EF2CL,
		0x3F3FC3DAE5C33F41L, 0x55551CC7921C55FFL,
		0xA2A210DB7910A2FBL, 0xEAEA65E90365EA23L,
		0x6565EC6A0FEC65AFL, 0xBABA6803B968BAD3L,
		0x2F2F934A65932F71L, 0xC0C0E78E4EE7C05DL,
		0xDEDE8160BE81DE7FL, 0x1C1C6CFCE06C1C24L,
		0xFDFD2E46BB2EFD1AL, 0x4D4D641F52644DD7L,
		0x9292E076E4E092ABL, 0x7575BCFA8FBC759FL,
		0x06061E36301E060AL, 0x8A8A98AE24988A83L,
		0xB2B2404BF940B2CBL, 0xE6E659856359E637L,
		0x0E0E367E70360E12L, 0x1F1F63E7F8631F21L,
		0x6262F75537F762A6L, 0xD4D4A33AEEA3D461L,
		0xA8A832812932A8E5L, 0x9696F452C4F496A7L,
		0xF9F93A629B3AF916L, 0xC5C5F6A366F6C552L,
		0x2525B11035B1256FL, 0x595920ABF22059EBL,
		0x8484AED054AE8491L, 0x7272A7C5B7A77296L,
		0x3939DDECD5DD394BL, 0x4C4C61165A614CD4L,
		0x5E5E3B94CA3B5EE2L, 0x7878859FE7857888L,
		0x3838D8E5DDD83848L, 0x8C8C869814868C89L,
		0xD1D1B217C6B2D16EL, 0xA5A50BE4410BA5F2L,
		0xE2E24DA1434DE23BL, 0x6161F84E2FF861A3L,
		0xB3B34542F145B3C8L, 0x2121A53415A52163L,
		0x9C9CD60894D69CB9L, 0x1E1E66EEF0661E22L,
		0x43435261225243C5L, 0xC7C7FCB176FCC754L,
		0xFCFC2B4FB32BFC19L, 0x040414242014040CL,
		0x515108E3B20851F3L, 0x9999C725BCC799B6L,
		0x6D6DC4224FC46DB7L, 0x0D0D396568390D17L,
		0xFAFA35798335FA13L, 0xDFDF8469B684DF7CL,
		0x7E7E9BA9D79B7E82L, 0x2424B4193DB4246CL,
		0x3B3BD7FEC5D73B4DL, 0xABAB3D9A313DABE0L,
		0xCECED1F03ED1CE4FL, 0x1111559988551133L,
		0x8F8F89830C898F8CL, 0x4E4E6B044A6B4ED2L,
		0xB7B75166D151B7C4L, 0xEBEB60E00B60EB20L,
		0x3C3CCCC1FDCC3C44L, 0x8181BFFD7CBF819EL,
		0x9494FE40D4FE94A1L, 0xF7F70C1CEB0CF704L,
		0xB9B96718A167B9D6L, 0x13135F8B985F1335L,
		0x2C2C9C517D9C2C74L, 0xD3D3B805D6B8D368L,
		0xE7E75C8C6B5CE734L, 0x6E6ECB3957CB6EB2L,
		0xC4C4F3AA6EF3C451L, 0x03030F1B180F0305L,
		0x565613DC8A1356FAL, 0x4444495E1A4944CCL,
		0x7F7F9EA0DF9E7F81L, 0xA9A937882137A9E6L,
		0x2A2A82674D822A7EL, 0xBBBB6D0AB16DBBD0L,
		0xC1C1E28746E2C15EL, 0x535302F1A20253F5L,
		0xDCDC8B72AE8BDC79L, 0x0B0B275358270B1DL,
		0x9D9DD3019CD39DBAL, 0x6C6CC12B47C16CB4L,
		0x3131F5A495F53153L, 0x7474B9F387B9749CL,
		0xF6F60915E309F607L, 0x4646434C0A4346CAL,
		0xACAC26A50926ACE9L, 0x898997B53C978986L,
		0x141444B4A044143CL, 0xE1E142BA5B42E13EL,
		0x16164EA6B04E163AL, 0x3A3AD2F7CDD23A4EL,
		0x6969D0066FD069BBL, 0x09092D41482D091BL,
		0x7070ADD7A7AD7090L, 0xB6B6546FD954B6C7L,
		0xD0D0B71ECEB7D06DL, 0xEDED7ED63B7EED2AL,
		0xCCCCDBE22EDBCC49L, 0x424257682A5742C6L,
		0x9898C22CB4C298B5L, 0xA4A40EED490EA4F1L,
		0x282888755D882878L, 0x5C5C3186DA315CE4L,
		0xF8F83F6B933FF815L, 0x8686A4C244A48697L
	};

	private static final long[] T7 = {
		0x1878D8C078182818L, 0x23AF2605AF236523L,
		0xC6F9B87EF9C657C6L, 0xE86FFB136FE825E8L,
		0x87A1CB4CA1879487L, 0xB86211A962B8D5B8L,
		0x0105090805010301L, 0x4F6E0D426E4FD14FL,
		0x36EE9BADEE365A36L, 0xA604FF5904A6F7A6L,
		0xD2BD0CDEBDD26BD2L, 0xF5060EFB06F502F5L,
		0x798096EF80798B79L, 0x6FCE305FCE6FB16FL,
		0x91EF6DFCEF91AE91L, 0x5207F8AA0752F652L,
		0x60FD4727FD60A060L, 0xBC76358976BCD9BCL,
		0x9BCD37ACCD9BB09BL, 0x8E8C8A048C8E8F8EL,
		0xA315D27115A3F8A3L, 0x0C3C6C603C0C140CL,
		0x7B8A84FF8A7B8D7BL, 0x35E180B5E1355F35L,
		0x1D69F5E8691D271DL, 0xE047B35347E03DE0L,
		0xD7AC21F6ACD764D7L, 0xC2ED9C5EEDC25BC2L,
		0x2E96436D962E722EL, 0x4B7A29627A4BDD4BL,
		0xFE215DA321FE1FFEL, 0x5716D5821657F957L,
		0x1541BDA841153F15L, 0x77B6E89FB6779977L,
		0x37EB92A5EB375937L, 0xE5569E7B56E532E5L,
		0x9FD9138CD99FBC9FL, 0xF01723D317F00DF0L,
		0x4A7F206A7F4ADE4AL, 0xDA95449E95DA73DAL,
		0x5825A2FA2558E858L, 0xC9CACF06CAC946C9L,
		0x298D7C558D297B29L, 0x0A225A50220A1E0AL,
		0xB14F50E14FB1CEB1L, 0xA01AC9691AA0FDA0L,
		0x6BDA147FDA6BBD6BL, 0x85ABD95CAB859285L,
		0xBD733C8173BDDABDL, 0x5D348FD2345DE75DL,
		0x1050908050103010L, 0xF40307F303F401F4L,
		0xCBC0DD16C0CB40CBL, 0x3EC6D3EDC63E423EL,
		0x05112D2811050F05L, 0x67E6781FE667A967L,
		0xE453977353E431E4L, 0x27BB0225BB276927L,
		0x415873325841C341L, 0x8B9DA72C9D8B808BL,
		0xA701F65101A7F4A7L, 0x7D94B2CF947D877DL,
		0x95FB49DCFB95A295L, 0xD89F568E9FD875D8L,
		0xFB30708B30FB10FBL, 0xEE71CD2371EE2FEEL,
		0x7C91BBC7917C847CL, 0x66E37117E366AA66L,
		0xDD8E7BA68EDD7ADDL, 0x174BAFB84B173917L,
		0x474645024647C947L, 0x9EDC1A84DC9EBF9EL,
		0xCAC5D41EC5CA43CAL, 0x2D995875992D772DL,
		0xBF792E9179BFDCBFL, 0x071B3F381B070907L,
		0xAD23AC0123ADEAADL, 0x5A2FB0EA2F5AEE5AL,
		0x83B5EF6CB5839883L, 0x33FFB685FF335533L,
		0x63F25C3FF263A563L, 0x020A12100A020602L,
		0xAA38933938AAE3AAL, 0x71A8DEAFA8719371L,
		0xC8CFC60ECFC845C8L, 0x197DD1C87D192B19L,
		0x49703B727049DB49L, 0xD99A5F869AD976D9L,
		0xF21D31C31DF20BF2L, 0xE348A84B48E338E3L,
		0x5B2AB9E22A5BED5BL, 0x8892BC3492888588L,
		0x9AC83EA4C89AB39AL, 0x26BE0B2DBE266A26L,
		0x32FABF8DFA325632L, 0xB04A59E94AB0CDB0L,
		0xE96AF21B6AE926E9L, 0x0F337778330F110FL,
		0xD5A633E6A6D562D5L, 0x80BAF474BA809D80L,
		0xBE7C27997CBEDFBEL, 0xCDDEEB26DECD4ACDL,
		0x34E489BDE4345C34L, 0x4875327A7548D848L,
		0xFF2454AB24FF1CFFL, 0x7A8F8DF78F7A8E7AL,
		0x90EA64F4EA90AD90L, 0x5F3E9DC23E5FE15FL,
		0x20A03D1DA0206020L, 0x68D50F67D568B868L,
		0x1A72CAD0721A2E1AL, 0xAE2CB7192CAEEFAEL,
		0xB45E7DC95EB4C1B4L, 0x5419CE9A1954FC54L,
		0x93E57FECE593A893L, 0x22AA2F0DAA226622L,
		0x64E96307E964AC64L, 0xF1122ADB12F10EF1L,
		0x73A2CCBFA2739573L, 0x125A82905A123612L,
		0x405D7A3A5D40C040L, 0x0828484028081808L,
		0xC3E89556E8C358C3L, 0xEC7BDF337BEC29ECL,
		0xDB904D9690DB70DBL, 0xA11FC0611FA1FEA1L,
		0x8D83911C838D8A8DL, 0x3DC9C8F5C93D473DL,
		0x97F15BCCF197A497L, 0x0000000000000000L,
		0xCFD4F936D4CF4CCFL, 0x2B876E45872B7D2BL,
		0x76B3E197B3769A76L, 0x82B0E664B0829B82L,
		0xD6A928FEA9D667D6L, 0x1B77C3D8771B2D1BL,
		0xB55B74C15BB5C2B5L, 0xAF29BE1129AFECAFL,
		0x6ADF1D77DF6ABE6AL, 0x500DEABA0D50F050L,
		0x454C57124C45CF45L, 0xF31838CB18F308F3L,
		0x30F0AD9DF0305030L, 0xEF74C42B74EF2CEFL,
		0x3FC3DAE5C33F413FL, 0x551CC7921C55FF55L,
		0xA210DB7910A2FBA2L, 0xEA65E90365EA23EAL,
		0x65EC6A0FEC65AF65L, 0xBA6803B968BAD3BAL,
		0x2F934A65932F712FL, 0xC0E78E4EE7C05DC0L,
		0xDE8160BE81DE7FDEL, 0x1C6CFCE06C1C241CL,
		0xFD2E46BB2EFD1AFDL, 0x4D641F52644DD74DL,
		0x92E076E4E092AB92L, 0x75BCFA8FBC759F75L,
		0x061E36301E060A06L, 0x8A98AE24988A838AL,
		0xB2404BF940B2CBB2L, 0xE659856359E637E6L,
		0x0E367E70360E120EL, 0x1F63E7F8631F211FL,
		0x62F75537F762A662L, 0xD4A33AEEA3D461D4L,
		0xA832812932A8E5A8L, 0x96F452C4F496A796L,
		0xF93A629B3AF916F9L, 0xC5F6A366F6C552C5L,
		0x25B11035B1256F25L, 0x5920ABF22059EB59L,
		0x84AED054AE849184L, 0x72A7C5B7A7729672L,
		0x39DDECD5DD394B39L, 0x4C61165A614CD44CL,
		0x5E3B94CA3B5EE25EL, 0x78859FE785788878L,
		0x38D8E5DDD8384838L, 0x8C869814868C898CL,
		0xD1B217C6B2D16ED1L, 0xA50BE4410BA5F2A5L,
		0xE24DA1434DE23BE2L, 0x61F84E2FF861A361L,
		0xB34542F145B3C8B3L, 0x21A53415A5216321L,
		0x9CD60894D69CB99CL, 0x1E66EEF0661E221EL,
		0x435261225243C543L, 0xC7FCB176FCC754C7L,
		0xFC2B4FB32BFC19FCL, 0x0414242014040C04L,
		0x5108E3B20851F351L, 0x99C725BCC799B699L,
		0x6DC4224FC46DB76DL, 0x0D396568390D170DL,
		0xFA35798335FA13FAL, 0xDF8469B684DF7CDFL,
		0x7E9BA9D79B7E827EL, 0x24B4193DB4246C24L,
		0x3BD7FEC5D73B4D3BL, 0xAB3D9A313DABE0ABL,
		0xCED1F03ED1CE4FCEL, 0x1155998855113311L,
		0x8F89830C898F8C8FL, 0x4E6B044A6B4ED24EL,
		0xB75166D151B7C4B7L, 0xEB60E00B60EB20EBL,
		0x3CCCC1FDCC3C443CL, 0x81BFFD7CBF819E81L,
		0x94FE40D4FE94A194L, 0xF70C1CEB0CF704F7L,
		0xB96718A167B9D6B9L, 0x135F8B985F133513L,
		0x2C9C517D9C2C742CL, 0xD3B805D6B8D368D3L,
		0xE75C8C6B5CE734E7L, 0x6ECB3957CB6EB26EL,
		0xC4F3AA6EF3C451C4L, 0x030F1B180F030503L,
		0x5613DC8A1356FA56L, 0x44495E1A4944CC44L,
		0x7F9EA0DF9E7F817FL, 0xA937882137A9E6A9L,
		0x2A82674D822A7E2AL, 0xBB6D0AB16DBBD0BBL,
		0xC1E28746E2C15EC1L, 0x5302F1A20253F553L,
		0xDC8B72AE8BDC79DCL, 0x0B275358270B1D0BL,
		0x9DD3019CD39DBA9DL, 0x6CC12B47C16CB46CL,
		0x31F5A495F5315331L, 0x74B9F387B9749C74L,
		0xF60915E309F607F6L, 0x46434C0A4346CA46L,
		0xAC26A50926ACE9ACL, 0x8997B53C97898689L,
		0x1444B4A044143C14L, 0xE142BA5B42E13EE1L,
		0x164EA6B04E163A16L, 0x3AD2F7CDD23A4E3AL,
		0x69D0066FD069BB69L, 0x092D41482D091B09L,
		0x70ADD7A7AD709070L, 0xB6546FD954B6C7B6L,
		0xD0B71ECEB7D06DD0L, 0xED7ED63B7EED2AEDL,
		0xCCDBE22EDBCC49CCL, 0x4257682A5742C642L,
		0x98C22CB4C298B598L, 0xA40EED490EA4F1A4L,
		0x2888755D88287828L, 0x5C3186DA315CE45CL,
		0xF83F6B933FF815F8L, 0x86A4C244A4869786L
	};

	private static final long[] RC = {
		0x4F01B887E8C62318L,
		0x52916F79F5D2A636L,
		0x357B0CA38E9BBC60L,
		0x57FE4B2EC2D7E01DL,
		0xDA4AF09FE5377715L,
		0x856BA0B10A29C958L,
		0x67053ECBF4105DBDL,
		0xD8957DA78B4127E4L,
		0x9E4717DD667CEEFBL,
		0x33835AAD07BF2DCAL
	};

	/** @see Digest */
	public String toString()
	{
		return "Whirlpool-1";
	}
}
