package de.saumya.mojo.rspec;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.List;
import java.util.Properties;

import org.apache.maven.plugin.MojoExecutionException;

import de.saumya.mojo.gem.AbstractGemMojo;

/**
 * executes the jruby command.
 * 
 * @goal test
 * @phase test
 * @requiresDependencyResolution test
 */
public class RSpecMojo extends AbstractGemMojo {

    /**
     * The project base directory
     * 
     * @parameter expression="${basedir}"
     * @required
     * @readonly
     */
    protected File                   basedir;

    /**
     * The classpath elements of the project being tested.
     * 
     * @parameter expression="${project.testClasspathElements}"
     * @required
     * @readonly
     */
    protected List<String>           classpathElements;

    /**
     * The flag to skip tests (optional, defaults to "false")
     * 
     * @parameter expression="${maven.test.skip}" default-value="false"
     */
    protected boolean                skipTests;

    /**
     * The flag to skip tests (optional, defaults to "false")
     * 
     * @parameter expression="${skipTests}" default-value="false"
     */
    protected boolean                skip;

    /**
     * The directory containing the RSpec source files
     * 
     * @parameter expression="spec"
     */
    protected String                 specSourceDirectory;

    /**
     * The directory where the RSpec report will be written to
     * 
     * @parameter expression="${basedir}/target"
     * @required
     */
    protected String                 reportDirectory;

    /**
     * The name of the RSpec report (optional, defaults to "rspec-report.html")
     * 
     * @parameter expression="rspec-report.html"
     */
    protected String                 reportName;

    /**
     * List of system properties to set for the tests.
     * 
     * @parameter
     */
    protected Properties             systemProperties;

    private final RSpecScriptFactory rspecScriptFactory = new RSpecScriptFactory();
    private final ShellScriptFactory shellScriptFactory = new ShellScriptFactory();

    private File specSourceDirectory() {
        return new File(launchDirectory(), this.specSourceDirectory);
    }

    @Override
    public void executeWithGems() throws MojoExecutionException {
        if (this.skipTests || this.skip) {
            getLog().info("Skipping RSpec tests");
            return;
        }

        final File specSourceDirectory = specSourceDirectory();
        if (!specSourceDirectory.exists()) {
            getLog().info("Skipping RSpec tests since " + specSourceDirectory
                    + " is missing");
            return;
        }
        getLog().info("Running RSpec tests from " + specSourceDirectory);

        final String reportPath = new File(this.reportDirectory,
                this.reportName).getAbsolutePath();

        initScriptFactory(this.rspecScriptFactory, reportPath);
        initScriptFactory(this.shellScriptFactory, reportPath);

        try {
            this.rspecScriptFactory.emit();
        }
        catch (final Exception e) {
            getLog().error("error emitting .rb", e);
        }
        try {
            this.shellScriptFactory.emit();
        }
        catch (final Exception e) {
            getLog().error("error emitting .sh", e);
        }

        execute(this.rspecScriptFactory.getScriptFile().getPath());

        final File reportFile = new File(reportPath);

        Reader in = null;
        try {
            in = new FileReader(reportFile);
            final BufferedReader reader = new BufferedReader(in);

            String line = null;

            while ((line = reader.readLine()) != null) {
                if (line.contains("0 failures")) {
                    return;
                }
            }
        }
        catch (final IOException e) {
            throw new MojoExecutionException("Unable to read test report file: "
                    + reportFile);
        }
        finally {
            if (in != null) {
                try {
                    in.close();
                }
                catch (final IOException e) {
                    throw new MojoExecutionException(e.getMessage());
                }
            }
        }

        throw new MojoExecutionException("There were test failures");
    }

    private void initScriptFactory(final ScriptFactory factory,
            final String reportPath) {
        factory.setBaseDir(this.basedir.getAbsolutePath());
        factory.setClasspathElements(this.classpathElements);
        factory.setOutputDir(new File(this.reportDirectory));
        factory.setReportPath(reportPath);
        factory.setSourceDir(specSourceDirectory().getAbsolutePath());
        Properties props = this.systemProperties;
        if (props == null) {
            props = new Properties();
        }
        factory.setSystemProperties(props);
    }

}
