/*
 * Decompiled with CFR 0.152.
 */
package de.samply.reporter.report;

import de.samply.reporter.context.CellContext;
import de.samply.reporter.context.CellStyleContext;
import de.samply.reporter.context.Context;
import de.samply.reporter.context.ContextGenerator;
import de.samply.reporter.exporter.ExporterClient;
import de.samply.reporter.exporter.ExporterClientException;
import de.samply.reporter.logger.BufferedLoggerFactory;
import de.samply.reporter.logger.Logger;
import de.samply.reporter.report.ReportGeneratorException;
import de.samply.reporter.report.RunningReportsManager;
import de.samply.reporter.report.metainfo.ReportMetaInfo;
import de.samply.reporter.report.metainfo.ReportMetaInfoManager;
import de.samply.reporter.report.metainfo.ReportMetaInfoManagerException;
import de.samply.reporter.report.workbook.WorkbookManager;
import de.samply.reporter.report.workbook.WorkbookManagerFactory;
import de.samply.reporter.script.ScriptEngineException;
import de.samply.reporter.script.ScriptEngineManager;
import de.samply.reporter.script.ScriptResult;
import de.samply.reporter.template.ColumnTemplate;
import de.samply.reporter.template.ReportTemplate;
import de.samply.reporter.template.SheetTemplate;
import de.samply.reporter.template.script.Script;
import de.samply.reporter.utils.ExternalSheetUtils;
import de.samply.reporter.utils.ExternalSheetUtilsException;
import de.samply.reporter.utils.FileUtils;
import de.samply.reporter.utils.PercentageLogger;
import de.samply.reporter.zip.ExporterUnzipper;
import de.samply.reporter.zip.ExporterUnzipperException;
import de.samply.reporter.zip.Zipper;
import de.samply.reporter.zip.ZipperException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.CellRangeAddress;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.FileSystemUtils;

@Component
public class ReportGenerator {
    private static final Logger logger = BufferedLoggerFactory.getLogger(ReportGenerator.class);
    private final ExporterClient exporterClient;
    private final ExporterUnzipper exporterUnzipper;
    private final ScriptEngineManager scriptEngineManager;
    private final ContextGenerator contextGenerator;
    private final WorkbookManagerFactory workbookManagerFactory;
    private final ReportMetaInfoManager reportMetaInfoManager;
    private final Integer workbookWindow;
    private final RunningReportsManager runningReportsManager;

    public ReportGenerator(ExporterClient exporterClient, ExporterUnzipper exporterUnzipper, ScriptEngineManager scriptEngineManager, ContextGenerator contextGenerator, WorkbookManagerFactory workbookManagerFactory, ReportMetaInfoManager reportMetaInfoManager, RunningReportsManager runningReportsManager, @Value(value="${EXCEL_WORKBOOK_WINDOW:30000000}") int workbookWindow) {
        this.exporterClient = exporterClient;
        this.exporterUnzipper = exporterUnzipper;
        this.scriptEngineManager = scriptEngineManager;
        this.contextGenerator = contextGenerator;
        this.workbookWindow = workbookWindow;
        this.reportMetaInfoManager = reportMetaInfoManager;
        this.workbookManagerFactory = workbookManagerFactory;
        this.runningReportsManager = runningReportsManager;
    }

    public void generate(ReportTemplate template, ReportMetaInfo reportMetaInfo) throws ReportGeneratorException {
        try {
            this.runningReportsManager.addRunningReportId(reportMetaInfo.id());
            this.exporterClient.fetchExportFiles(filePath -> this.generate(template, filePath, reportMetaInfo), template);
        }
        catch (ExporterClientException | RuntimeException e) {
            this.runningReportsManager.removeRunningReportId(reportMetaInfo.id());
            BufferedLoggerFactory.clearBuffer();
            throw new ReportGeneratorException(e);
        }
    }

    private void generate(ReportTemplate template, String filePath, ReportMetaInfo reportMetaInfo) {
        logger.info("Extracting paths...");
        Path[] paths = this.extractPaths(filePath);
        logger.info("Generating context...");
        Context context = this.contextGenerator.generate(template, paths);
        logger.info("Generating raw report...");
        Map scriptResultMap = this.scriptEngineManager.generateRawReport(template, context);
        logger.info("Generating excel file");
        WorkbookManager workbookManager = this.workbookManagerFactory.create();
        logger.info("Filling excel file with data...");
        this.fillWorkbookWithData(workbookManager, template, scriptResultMap);
        logger.info("Adding format to excel file...");
        workbookManager.apply(workbook -> this.addFormatToWorkbook(workbook, template, context));
        logger.info("Writing excel file...");
        workbookManager.writeWorkbook(reportMetaInfo.path());
        logger.info("Removing temporal files...");
        this.removeTemporalFiles(paths[0].getParent(), scriptResultMap.values());
        logger.info("Zipping files if necessary...");
        this.createZipIfMoreThanOneFile(workbookManager, reportMetaInfo);
        this.runningReportsManager.removeRunningReportId(reportMetaInfo.id());
        logger.info("Excel file generated satisfactory.");
        BufferedLoggerFactory.clearBuffer();
    }

    private void removeTemporalFiles(Path sourceFilesDirectory, Collection<ScriptResult> scriptResults) {
        try {
            this.removeTemporalFilesWithoutExceptionHandling(sourceFilesDirectory, scriptResults);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void removeTemporalFilesWithoutExceptionHandling(Path sourceFilesDirectory, Collection<ScriptResult> scriptResults) throws IOException {
        FileSystemUtils.deleteRecursively((Path)sourceFilesDirectory);
        scriptResults.forEach(scriptResult -> {
            try {
                Files.delete(scriptResult.rawResult());
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        });
    }

    private Path[] extractPaths(String filePath) {
        try {
            return this.exporterUnzipper.extractFiles(filePath);
        }
        catch (ExporterUnzipperException e) {
            throw new RuntimeException(e);
        }
    }

    private void fillWorkbookWithData(WorkbookManager workbookManager, ReportTemplate template, Map<Script, ScriptResult> scriptResultMap) {
        template.getSheetTemplates().forEach(sheetTemplate -> {
            logger.info("Adding sheet '" + sheetTemplate.getName() + "'...");
            if (sheetTemplate.getFileUrl() != null || sheetTemplate.getFilePath() != null) {
                this.addSheetFromSourceExcelFile(workbookManager, sheetTemplate);
            } else {
                this.fillSheetWithData(workbookManager, sheetTemplate, scriptResultMap);
            }
        });
    }

    private void addSheetFromSourceExcelFile(WorkbookManager workbookManager, SheetTemplate template) {
        try {
            ExternalSheetUtils.addSheetFromSourceExcelFile((WorkbookManager)workbookManager, (SheetTemplate)template);
        }
        catch (ExternalSheetUtilsException e) {
            throw new RuntimeException(e);
        }
    }

    private void fillSheetWithData(WorkbookManager workbookManager, SheetTemplate template, Map<Script, ScriptResult> scriptResultMap) {
        ScriptResult result;
        if (template.getValuesScript() != null && (result = scriptResultMap.get(template.getValuesScript().getScript())) != null) {
            this.fillSheetWithData(workbookManager, template, result);
            logger.info("Adding autosize to sheet '" + template.getName() + "'...");
            workbookManager.apply(template, arg_0 -> this.autoSizeSheet(arg_0));
            logger.info("Adding auto filter to sheet '" + template.getName() + "'...");
            workbookManager.apply(template, arg_0 -> this.addAutoFilter(arg_0));
        }
    }

    private void fillSheetWithData(WorkbookManager workbookManager, SheetTemplate template, ScriptResult result) {
        try {
            this.fillSheetWithDataWithoutExceptionHandling(workbookManager, template, result);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void fillSheetWithDataWithoutExceptionHandling(WorkbookManager workbookManager, SheetTemplate template, ScriptResult result) throws IOException {
        PercentageLogger percentageLogger = new PercentageLogger(logger, (int)FileUtils.fetchNumberOfLines((Path)result.rawResult()), "Filling sheet" + template.getName() + " with data...");
        Files.readAllLines(result.rawResult()).forEach(line -> {
            this.fillRowWithData(workbookManager.createRow(template), line, result);
            percentageLogger.incrementCounter();
        });
        workbookManager.fetchLastSheetAndCreateIfNotExist(template);
    }

    private void fillRowWithData(Row row, String line, ScriptResult result) {
        AtomicInteger columnIndex = new AtomicInteger(0);
        Arrays.stream(line.split(result.csvConfig().delimiter())).forEach(value -> this.setCellValue(row.createCell(columnIndex.getAndIncrement()), value));
    }

    private void setCellValue(Cell cell, String value) {
        if (value == null) {
            cell.setCellValue("");
        } else if (NumberUtils.isParsable((String)value)) {
            if (NumberUtils.isDigits((String)value)) {
                cell.setCellValue((double)Integer.valueOf(value).intValue());
            } else {
                cell.setCellValue(Double.valueOf(value).doubleValue());
            }
        } else {
            cell.setCellValue(value);
        }
    }

    private void autoSizeSheet(Sheet sheet) {
        Row headerRow;
        int lastRowNum = sheet.getLastRowNum();
        if (lastRowNum >= 0 && (headerRow = sheet.getRow(0)) != null) {
            for (int j = 0; j <= headerRow.getLastCellNum(); ++j) {
                sheet.autoSizeColumn(j);
            }
        }
    }

    private void addAutoFilter(Sheet sheet) {
        int rowStartIndex = 0;
        int rowEndIndex = sheet.getLastRowNum();
        if (rowEndIndex >= 0 && sheet.getRow(0) != null) {
            int columnStartIndex = 0;
            int columnEndIndex = sheet.getRow(0).getLastCellNum() - 1;
            CellRangeAddress cra = new CellRangeAddress(rowStartIndex, rowEndIndex, columnStartIndex, columnEndIndex);
            sheet.setAutoFilter(cra);
        }
    }

    private void addFormatToWorkbook(Workbook workbook, ReportTemplate template, Context context) {
        template.getSheetTemplates().forEach(sheetTemplate -> {
            CellStyleContext cellStyleContext = new CellStyleContext(workbook);
            AtomicInteger columnNumber = new AtomicInteger(0);
            logger.info("Adding general format to all cells...");
            sheetTemplate.getFormatScripts().forEach(formatScript -> {
                if (formatScript.getScript() != null) {
                    this.addFormatToAllCellsOfASheet(workbook, sheetTemplate, cellStyleContext, context, formatScript.getScript());
                }
            });
            logger.info("Adding column format to column cells...");
            sheetTemplate.getColumnTemplates().forEach(columnTemplate -> this.addFormatToWorkbook(workbook, sheetTemplate, columnTemplate, columnNumber.getAndIncrement(), cellStyleContext, context));
        });
    }

    private void addFormatToWorkbook(Workbook workbook, SheetTemplate sheetTemplate, ColumnTemplate columnTemplate, int columnNumber, CellStyleContext cellStyleContext, Context context) {
        Sheet sheet = workbook.getSheet(sheetTemplate.getName());
        if (sheet != null) {
            logger.info("Adding header format to column '" + columnTemplate.getName() + "' in sheet '" + sheetTemplate.getName() + "'...");
            this.addHeaderFormatToWorkbook(columnTemplate, sheet, columnNumber, cellStyleContext, context);
            logger.info("Adding value format to column '" + columnTemplate.getName() + "' in sheet '" + sheetTemplate.getName() + "'...");
            this.addValueFormatToWorkbook(columnTemplate, sheet, columnNumber, cellStyleContext, context);
        }
    }

    private void addHeaderFormatToWorkbook(ColumnTemplate template, Sheet sheet, int columnNumber, CellStyleContext cellStyleContext, Context context) {
        Script script;
        if (template.getHeaderFormatScript() != null && (script = template.getHeaderFormatScript().getScript()) != null) {
            this.fetchCellContext(script, cellStyleContext, context).applyCellStyleToCell(sheet.getRow(0).getCell(columnNumber));
        }
    }

    private void addValueFormatToWorkbook(ColumnTemplate template, Sheet sheet, int columnNumber, CellStyleContext cellStyleContext, Context context) {
        Script script;
        if (template.getValueFormatScript() != null && (script = template.getValueFormatScript().getScript()) != null) {
            CellContext cellContext = this.fetchCellContext(script, cellStyleContext, context);
            int lastRowNum = sheet.getLastRowNum();
            if (lastRowNum > 0) {
                for (int i = 1; i <= lastRowNum; ++i) {
                    cellContext.applyCellStyleToCell(sheet.getRow(i).getCell(columnNumber));
                }
            }
        }
    }

    private void addFormatToAllCellsOfASheet(Workbook workbook, SheetTemplate sheetTemplate, CellStyleContext cellStyleContext, Context context, Script script) {
        CellContext cellContext = this.fetchCellContext(script, cellStyleContext, context);
        Sheet sheet = workbook.getSheet(sheetTemplate.getName());
        if (sheet != null) {
            logger.info("Adding format to sheet '" + sheetTemplate.getName() + "'...");
            cellContext.applySheetStyleToSheet(sheet);
            PercentageLogger percentageLogger = new PercentageLogger(logger, sheet.getLastRowNum(), "Adding format to all cells of sheet '" + sheetTemplate.getName() + "'...");
            sheet.forEach(row -> {
                row.forEach(arg_0 -> ((CellContext)cellContext).applyCellStyleToCell(arg_0));
                percentageLogger.incrementCounter();
            });
        }
    }

    private CellContext fetchCellContext(Script script, CellStyleContext cellStyleContext, Context context) {
        try {
            return this.scriptEngineManager.generateCellContext(script, cellStyleContext, context);
        }
        catch (ScriptEngineException e) {
            throw new RuntimeException(e);
        }
    }

    private void createZipIfMoreThanOneFile(WorkbookManager workbookManager, ReportMetaInfo reportMetaInfo) {
        try {
            this.createZipIfMoreThanOneFileWithoutHandlingException(workbookManager, reportMetaInfo);
        }
        catch (ReportMetaInfoManagerException | ZipperException | IOException e) {
            logger.info(ExceptionUtils.getStackTrace((Throwable)e));
        }
    }

    private void createZipIfMoreThanOneFileWithoutHandlingException(WorkbookManager workbookManager, ReportMetaInfo reportMetaInfo) throws ReportMetaInfoManagerException, IOException, ZipperException {
        if (workbookManager.isMultiWorkbook()) {
            Path zippedPath = Zipper.zip((List)workbookManager.fetchRealPaths(reportMetaInfo.path()));
            this.reportMetaInfoManager.reset();
            this.reportMetaInfoManager.addReportMetaInfoToFile(new ReportMetaInfo(reportMetaInfo.id(), zippedPath, reportMetaInfo.timestamp(), reportMetaInfo.templateId()));
        }
    }
}

