package de.samply.auth.client.jwt;

import java.math.BigInteger;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** A small helper, that loads RSA keys from Base64 encoded values. */
public class KeyLoader {

  private static final String[] SUPPORTED_ALGORITHMS = new String[] {"RSA", "EC"};

  private static final Logger logger = LoggerFactory.getLogger(KeyLoader.class);

  /** Disable instantiation. */
  private KeyLoader() {}

  /**
   * Generates a RSA public key by using the modulus and public exponent.
   *
   * @param base64urlModulus the base64url encoded modulus
   * @param base64urlPublicExponent the base64url encoded public key exponent
   * @return the RSA public key
   */
  public static PublicKey loadKey(String base64urlModulus, String base64urlPublicExponent) {
    RSAPublicKeySpec keySpec =
        new RSAPublicKeySpec(
            new BigInteger(Base64.decodeBase64(base64urlModulus)),
            new BigInteger(Base64.decodeBase64(base64urlPublicExponent)));

    try {
      KeyFactory kf = KeyFactory.getInstance("RSA");
      return kf.generatePublic(keySpec);
    } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
      logger.error("Exception: ", e);
    }

    throw new UnsupportedOperationException("Unknown Key Format");
  }

  /**
   * Generates a RSA public key by using the base64 encoded DER formatted public key (X509).
   *
   * @param base64der the base64 DER encoded public key
   * @return the RSA public key
   */
  public static PublicKey loadKey(String base64der) {
    try {
      X509EncodedKeySpec keySpec = new X509EncodedKeySpec(Base64.decodeBase64(base64der));

      for (String algorithm : SUPPORTED_ALGORITHMS) {
        try {
          KeyFactory kf = KeyFactory.getInstance(algorithm);
          return kf.generatePublic(keySpec);
        } catch (InvalidKeySpecException e) {
          /* Ignore, because we just try the next algorithm! */
        }
      }
    } catch (NoSuchAlgorithmException e) {
      logger.error("Exception: ", e);
    }

    throw new UnsupportedOperationException("Unknown Key Format");
  }

  /**
   * Uses the given private RSA key to create the public key. This method requires a private RSA
   * key.
   */
  public static PublicKey loadPublicRsaKey(PrivateKey privateKey) {
    try {
      if (!(privateKey instanceof RSAPrivateCrtKey)) {
        throw new UnsupportedOperationException("The given key is not a private RSA key!");
      }

      KeyFactory factory = KeyFactory.getInstance("RSA");
      RSAPrivateCrtKey rsaPrivate = (RSAPrivateCrtKey) privateKey;

      RSAPublicKeySpec spec =
          new RSAPublicKeySpec(rsaPrivate.getModulus(), rsaPrivate.getPublicExponent());
      return factory.generatePublic(spec);
    } catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
      logger.error("Exception: ", e);
    }

    throw new UnsupportedOperationException("Unknown Key Format");
  }

  /**
   * Generates a RSA private key by using the base64 encoded DER formatted private key (PKCS8 DER
   * format).
   *
   * @param base64der the base64 DER PKCS8 encoded private key
   * @return the RSA private key
   */
  public static PrivateKey loadPrivateKey(String base64der) {
    try {
      byte[] encodedPrivate = Base64.decodeBase64(base64der);
      PKCS8EncodedKeySpec privateSpec = new PKCS8EncodedKeySpec(encodedPrivate);

      for (String algorithm : SUPPORTED_ALGORITHMS) {
        try {
          KeyFactory kf = KeyFactory.getInstance(algorithm);
          return kf.generatePrivate(privateSpec);
        } catch (InvalidKeySpecException e) {
          /* Ignore, because we just try the next algorithm! */
        }
      }
    } catch (NoSuchAlgorithmException e) {
      logger.error("Exception: ", e);
    }

    throw new UnsupportedOperationException("Unknown Key Format");
  }
}
