package de.rwth.swc.coffee4j.engine.characterization;

import de.rwth.swc.coffee4j.engine.TestResult;

import java.util.List;
import java.util.Map;

/**
 * Describes an algorithm used to characterize failure-inducing combinations in combinatorial tests. A combination
 * is considered failure-inducing if every test input containing it fails.
 */
public interface FaultCharacterizationAlgorithm {
    
    /**
     * Refines the internal list of suspicious of faulty combinations. If this
     * method is called for the first time, an initial list of suspicious
     * combinations is created. This process may take a while.
     * After the refinement of the list, test inputs are generated. These test
     * inputs should be executed and this method should then be called with the
     * results of these test inputs until no test inputs are returned. Through
     * this process the underlying algorithm can refine and shorten it's list of
     * suspicious combinations over time.
     *
     * <p>When no combinations are returned any more,
     * {@link #computeFailureInducingCombinations()} can be used to compute a
     * final list of most likely faulty combinations.
     *
     * @param testResults the results of the initial test suite or previous test
     *                    inputs generated by this method. Must not be
     *                    {@code null} or empty.
     * @return a list of further test inputs which need to be executed to refine
     * the list of suspicious combinations
     * @throws IllegalStateException    if the algorithm has not been initialized
     * @throws NullPointerException     if testResults is {@code null}
     * @throws IllegalArgumentException if testResults is empty
     */
    List<int[]> computeNextTestInputs(Map<int[], TestResult> testResults);
    
    /**
     * Computes a list of most likely failure inducing combinations refined
     * from previous calls to {@link #computeNextTestInputs(Map)}. The
     * combinations returned by this method are not guaranteed to be faulty,
     * but it is guaranteed that no test input executed which contained this
     * combination was successful.
     *
     * @return a list of faulty combinations. The list may be ranked, depending
     * on the underlying algorithm. If this list is ranked the
     * combinations on positions with smaller indices are more likely to
     * be failure inducing
     * @throws IllegalStateException if the algorithm has not been initialized
     */
    List<int[]> computeFailureInducingCombinations();
    
}
