package de.rwth.swc.coffee4j.engine.report;

import de.rwth.swc.coffee4j.engine.characterization.FaultCharacterizationAlgorithm;
import de.rwth.swc.coffee4j.engine.manager.CombinatorialTestConfiguration;
import de.rwth.swc.coffee4j.engine.generator.TestInputGroup;
import de.rwth.swc.coffee4j.engine.generator.TestInputGroupGenerator;

import java.util.List;

/**
 * An interface defining a reporter listening for events during the generation of combinatorial test inputs.
 * Can be used together with the {@link CombinatorialTestConfiguration}.
 * All methods are implemented as empty default methods to that any implementing class must only overwrite methods
 * it needs.
 */
public interface GenerationReporter extends Reporter {
    
    /**
     * Called if a new {@link TestInputGroup} has been generated by a {@link TestInputGroupGenerator}. This is only called
     * once per {@link TestInputGroup}, and the identifier can be used in all other methods to connect calles for the
     * same group.
     *
     * @param testInputGroup the group which was generated
     * @param generator      the generator which generated the given group
     */
    default void testInputGroupGenerated(TestInputGroup testInputGroup, TestInputGroupGenerator generator) {
    }
    
    /**
     * Called if all operations for a {@link TestInputGroup} are finished. After this has been called it is guaranteed
     * that no further test inputs are generated for the group, so no other methods will be called for the same group.
     * This is needed additionally to {@link #faultCharacterizationFinished(TestInputGroup, List)} since some groups may
     * disable fault characterization and a reporter still needs to be notified after the group is finished. If fault
     * characterization is used for the group, {@link #faultCharacterizationFinished(TestInputGroup, List)} has the same meaning.
     *
     * @param testInputGroup the group which has been finished
     */
    default void testInputGroupFinished(TestInputGroup testInputGroup) {
    }
    
    /**
     * Called if fault characterization for the given group has started. Consequently, this method is only called once per
     * {@link TestInputGroup}, and never if fault characterization is not enabled.
     *
     * @param testInputGroup the group for which the characterization process started
     * @param algorithm      the algorithm which will be used to perform fault characterization. Since this is the same instance
     *                       as used in the test, implementing classes could theoretically modify it, but is is strongly
     *                       advised not to modify this algorithm because many unintended side effects could happen.
     */
    default void faultCharacterizationStarted(TestInputGroup testInputGroup, FaultCharacterizationAlgorithm algorithm) {
    }
    
    /**
     * Called after all fault characterization has been finished. Directly after this method
     * {@link #testInputGroupFinished(TestInputGroup)} will be called.
     *
     * @param testInputGroup              the group for which fault characterization has been finished
     * @param failureInducingCombinations all failure inducing combinations found. These may or may not be ordered
     *                                    depending of the algorithm used. They also do not necessarily contain
     *                                    combinations guaranteed to be failure-inducing but rather those "most
     *                                    probably failure-inducing"
     */
    default void faultCharacterizationFinished(TestInputGroup testInputGroup, List<int[]> failureInducingCombinations) {
    }
    
    /**
     * Called if new test inputs have been generated by the {@link FaultCharacterizationAlgorithm} provided to
     * {@link #faultCharacterizationStarted(TestInputGroup, FaultCharacterizationAlgorithm)}. This method may be called multiple
     * times or never depending on the definition of a combinatorial test. It will only ever be called between calls
     * to {@link #faultCharacterizationStarted(TestInputGroup, FaultCharacterizationAlgorithm)} and
     * {@link #faultCharacterizationFinished(TestInputGroup, List)}.
     *
     * @param testInputGroup the group for which further test inputs were generated for fault characterization
     * @param testInputs     all generated test inputs
     */
    default void faultCharacterizationTestInputsGenerated(TestInputGroup testInputGroup, List<int[]> testInputs) {
    }
    
}
