/*
 * Copyright 2014 Ruediger Moeller.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.nustaq.fastcast.impl;

import org.nustaq.fastcast.api.FCSubscriber;
import org.nustaq.fastcast.util.FCLog;
import org.nustaq.offheap.structs.structtypes.StructString;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;

/*
 * Created with IntelliJ IDEA.
 * User: ruedi
 * Date: 14.08.13
 * Time: 00:42
 * To change this template use File | Settings | File Templates.
 */

/**
 * multiplexes messages of different publishers to their
 * associated PacketReceiveBuffer
 */
public class ReceiveBufferDispatcher {

    ConcurrentHashMap<StructString,PacketReceiveBuffer> bufferMap = new ConcurrentHashMap<StructString, PacketReceiveBuffer>();

    int packetSize;
    String nodeId;
    int historySize;
    int topic;
    FCSubscriber receiver;
    Topic topicEntry;

    public ReceiveBufferDispatcher(int packetSize, String nodeId, Topic entry, FCSubscriber rec) {
        receiver = rec;
        this.packetSize = packetSize;
        this.nodeId = nodeId;
        this.historySize = entry.getSubscriberConf().getReceiveBufferPackets();
        this.topic = entry.getTopicId();
        topicEntry = entry;
    }

    public Topic getTopicEntry() {
        return topicEntry;
    }

    public PacketReceiveBuffer getBuffer(StructString sender) {
        PacketReceiveBuffer receiveBuffer = bufferMap.get(sender);
        if ( receiveBuffer == null ) {
            int hSize = historySize;
            if ( ((long)hSize*packetSize) > Integer.MAX_VALUE-2*packetSize ) {
                final int newHist = (Integer.MAX_VALUE - 2 * packetSize) / packetSize;
                topicEntry.getSubscriberConf().receiveBufferPackets(newHist);
                FCLog.get().warn("int overflow, degrading history size from "+hSize+" to "+newHist);
                historySize = newHist;
            }
            receiveBuffer = new PacketReceiveBuffer(packetSize,nodeId,historySize,sender.toString(), topicEntry, receiver);
            bufferMap.put((StructString) sender.createCopy(),receiveBuffer);
        }
        return receiveBuffer;
    }

    /**
     * if a sender stops sending, remove from map to free memory
     * @param senderName
     */
    public void cleanup(String senderName) {
        StructString struct = new StructString(senderName);
        PacketReceiveBuffer packetReceiveBuffer = bufferMap.get(struct);
        bufferMap.remove(struct);
        if ( packetReceiveBuffer != null ) {
            packetReceiveBuffer.terminate();
            FCSubscriber subscriber = packetReceiveBuffer.getTopicEntry().getSubscriber();
            if ( subscriber != null ) {
                subscriber.senderTerminated(senderName);
            }
        }
        else {
            FCLog.get().warn("cannot find packetReceiver to terminate");
        }
    }

    public void cleanupTopic() {
        ArrayList<String> keys = new ArrayList(bufferMap.keySet());
        for (int i = 0; i < keys.size(); i++) {
            String o =  keys.get(i);
            cleanup(o);
        }
    }

    public void getTimedOutSenders(long now, long timeout, List<String> res) {
        for (Iterator<PacketReceiveBuffer> iterator = bufferMap.values().iterator(); iterator.hasNext(); ) {
            PacketReceiveBuffer next = iterator.next();
            if ( now - next.getLastHBMillis() > timeout ) {
                res.add(next.getReceivesFrom());
            }
        }
    }
}
