/**
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see </http:>//www.gnu.org/licenses/>.
 */
package de.richtercloud.execution.tools

import com.google.common.collect.ImmutableMap
import de.richtercloud.execution.tools.ExecutionUtils.PATH_TEMPLATE
import de.richtercloud.execution.tools.ExecutionUtils.SH_DEFAULT
import de.richtercloud.execution.tools.ExecutionUtils.createProcess
import java.io.File
import java.io.IOException
import java.io.OutputStream
import java.lang.ProcessBuilder.Redirect
import java.util.Arrays
import org.apache.commons.lang3.tuple.ImmutableTriple
import org.apache.commons.lang3.tuple.Triple
import org.slf4j.Logger
import org.slf4j.LoggerFactory

/**
 *
 * @author richter
 */
object ExecutionUtils {
    private val LOGGER = LoggerFactory.getLogger(ExecutionUtils::class.java)
    val SH_DEFAULT = "sh"
    private val PATH_TEMPLATE = "PATH"

    @JvmStatic
    @Throws(IOException::class)
    fun createProcess(path: String,
                      stdoutOutputStream: OutputStream,
                      stderrOutputStream: OutputStream,
                      vararg commands: String): Triple<Process, OutputReaderThread, OutputReaderThread> {
        return createProcess(null,
                path,
                stdoutOutputStream,
                stderrOutputStream,
                *commands)
    }

    @JvmStatic
    @Throws(IOException::class)
    fun createProcess(directory: File,
                      stdoutOutputStream: OutputStream,
                      stderrOutputStream: OutputStream,
                      vararg commands: String): Triple<Process, OutputReaderThread, OutputReaderThread> {
        return createProcess(directory,
                System.getenv(PATH_TEMPLATE),
                stdoutOutputStream,
                stderrOutputStream,
                *commands)
    }

    @JvmStatic
    @Throws(IOException::class)
    fun createProcess(directory: File?,
                      path: String,
                      stdoutOutputStream: OutputStream,
                      stderrOutputStream: OutputStream,
                      vararg commands: String): Triple<Process, OutputReaderThread, OutputReaderThread> {
        return createProcess(directory,
                ImmutableMap.builder<String, String>()
                        .put(PATH_TEMPLATE, path)
                        .build(),
                SH_DEFAULT,
                stdoutOutputStream,
                stderrOutputStream,
                *commands)
    }

    /**
     * Allows sharing code between different process creation routines.
     *
     * @param directory the directory in which to execute the process
     * @param env the environment mapping
     * @param sh the shell in which to wrap the command execution
     * @param stdoutOutputStream the output stream to collect the stdout output
     * @param stderrOutputStream the output stream to collect the stderr output
     * @param commands the commands to create a [Process] for
     * @return the created process and references to the
     * [OutputReaderThread] for `stdout` and `stderr`
     * depending on whether `stdoutStream` or `stderrStream`
     * have been not `null`
     * @throws IOException if an I/O exception occurs during process
     * communication
     */
    /*
    internal implementation notes:
    - checking for canceled doesn't make sense here because null would have to
    be returned or an exception thrown in the case of canceled state which
    creates the need to evaluate the return value by callers which is equally
    complex as checking the condition before calls to createProcess
    */
    @JvmStatic
    @Throws(IOException::class)
    fun createProcess(directory: File?,
                      env: Map<String, String>,
                      sh: String,
                      stdoutOutputStream: OutputStream?,
                      stderrOutputStream: OutputStream?,
                      vararg commands: String): Triple<Process, OutputReaderThread, OutputReaderThread> {
        LOGGER.trace(String.format("building process with commands '%s' with environment '%s' running in %s",
                Arrays.asList(*commands),
                env,
                if (directory != null)
                    String.format("directory '%s'",
                            directory.absolutePath)
                else
                    "current directory"))
        val processBuilder = ProcessBuilder(sh, "-c", commands.joinToString(" "))
                .redirectOutput(if (stdoutOutputStream != null) Redirect.PIPE else Redirect.INHERIT)
                .redirectError(if (stderrOutputStream != null) Redirect.PIPE else Redirect.INHERIT)
        //need to wrap commands in a shell in order allow modified path for
        //binary discovery (the unmodifiable PATH of the JVM is used to find
        //the command to execute which doesn't allow any modification after
        //installations by wrapper)
        if (directory != null) {
            processBuilder.directory(directory)
        }
        processBuilder.environment().putAll(env)
        val retValue = processBuilder.start()
        var stdoutReaderThread: OutputReaderThread? = null
        var stderrReaderThread: OutputReaderThread? = null
        if (stdoutOutputStream != null) {
            stdoutReaderThread = OutputReaderThread(retValue.inputStream,
                    OutputReaderThreadMode.OUTPUT_STREAM,
                    stdoutOutputStream)
            stdoutReaderThread.start()
        }
        if (stderrOutputStream != null) {
            stderrReaderThread = OutputReaderThread(retValue.errorStream,
                    OutputReaderThreadMode.OUTPUT_STREAM,
                    stderrOutputStream)
            stderrReaderThread.start()
        }
        return ImmutableTriple(retValue,
                stdoutReaderThread,
                stderrReaderThread)
    }
}
