package de.rhocas.featuregen.generator;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import de.rhocas.featuregen.featureide.model.configuration.Configuration;
import de.rhocas.featuregen.featureide.model.configuration.Feature;
import de.rhocas.featuregen.featureide.model.feature.BranchedFeatureType;
import de.rhocas.featuregen.featureide.model.feature.FeatureModel;
import de.rhocas.featuregen.featureide.model.feature.FeatureModelType;
import de.rhocas.featuregen.featureide.model.feature.FeatureType;
import de.rhocas.featuregen.featureide.model.feature.StructType;
import de.rhocas.featuregen.generator.FeatureNameConverter;
import de.rhocas.featuregen.generator.GeneratorHelper;
import de.rhocas.featuregen.generator.NameProvider;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Unmarshaller;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * This class generates the variant from a FeatureIDE configuration file.
 * 
 * @author Nils Christian Ehmke
 * 
 * @since 2.0.0
 */
@SuppressWarnings("all")
public final class FeatureIDEVariantGenerator {
  @Accessors(AccessorType.NONE)
  private static final class Parameters {
    private final Configuration configurationModel;
    
    private final FeatureModelType featureModel;
    
    private final String packageName;
    
    private final String className;
    
    private final File outputFolder;
    
    private final String featuresPackageName;
    
    private final String prefix;
    
    private final String suffix;
    
    public Parameters(final Configuration configurationModel, final FeatureModelType featureModel, final String packageName, final String className, final File outputFolder, final String featuresPackageName, final String prefix, final String suffix) {
      super();
      this.configurationModel = configurationModel;
      this.featureModel = featureModel;
      this.packageName = packageName;
      this.className = className;
      this.outputFolder = outputFolder;
      this.featuresPackageName = featuresPackageName;
      this.prefix = prefix;
      this.suffix = suffix;
    }
  }
  
  /**
   * The main entry point of the standalone generator.
   * 
   * @param args
   * 		The command line arguments. See {@link #generate(String[])} for details.
   * 
   * @since 2.0.0
   */
  public static void main(final String[] args) {
    final FeatureIDEVariantGenerator generator = new FeatureIDEVariantGenerator();
    generator.generate(args);
  }
  
  /**
   * This method performs the actual generation.
   * 
   * @param args
   * 		The arguments for the generator. This array must contain at least three entries. The arguments are as following:
   *      <ul>
   *      	<li>The path to the FeatureIDE configuration model file.</li>
   *          <li>The path to the FeatureIDE feature model file.</li>
   *          <li>The path to the output folder.</li>
   *          <li>The package name for the newly generated variant.</li>
   *          <li>The simple class name of the new variant.</li>
   *          <li>The package name of the features.</li>
   *          <li>The optional prefix prepended to each feature.</li>
   *          <li>The optional suffix appended to each feature. If this parameter is not given, {@code _FEATURE} will be used instead.</li>
   *      </ul>
   * 
   * @throws IllegalArgumentException
   * 		If the number of arguments is invalid or if one of the model files can not be found.
   * 
   * @since 2.0.0
   */
  public void generate(final String[] args) {
    final FeatureIDEVariantGenerator.Parameters parameters = this.convertAndCheckParameters(args);
    this.generateVariantClass(parameters);
  }
  
  private FeatureIDEVariantGenerator.Parameters convertAndCheckParameters(final String[] args) {
    FeatureIDEVariantGenerator.Parameters _xblockexpression = null;
    {
      int _size = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _lessThan = (_size < 6);
      if (_lessThan) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Invalid number of arguments. Expected at least 6, but was ");
        int _size_1 = ((List<String>)Conversions.doWrapArray(args)).size();
        _builder.append(_size_1);
        _builder.append(".");
        throw new IllegalArgumentException(_builder.toString());
      }
      final String configurationModelFilePath = args[0];
      final File configurationModelFile = new File(configurationModelFilePath);
      boolean _isFile = configurationModelFile.isFile();
      boolean _not = (!_isFile);
      if (_not) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Model file \"");
        _builder_1.append(configurationModelFilePath);
        _builder_1.append("\" can not be found.");
        throw new IllegalArgumentException(_builder_1.toString());
      }
      final Configuration configurationModel = this.readConfigurationModel(configurationModelFile);
      final String featureModelFilePath = args[1];
      final File featureModelFile = new File(featureModelFilePath);
      boolean _isFile_1 = featureModelFile.isFile();
      boolean _not_1 = (!_isFile_1);
      if (_not_1) {
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("Model file \"");
        _builder_2.append(featureModelFilePath);
        _builder_2.append("\" can not be found.");
        throw new IllegalArgumentException(_builder_2.toString());
      }
      final FeatureModelType featureModel = this.readFeatureModel(featureModelFile);
      final String outputFolderPath = args[2];
      final File outputFolder = new File(outputFolderPath);
      final String packageName = args[3];
      final String className = args[4];
      final String featuresPackageName = args[5];
      String _xifexpression = null;
      int _size_2 = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _greaterEqualsThan = (_size_2 >= 7);
      if (_greaterEqualsThan) {
        _xifexpression = args[6];
      } else {
        _xifexpression = "";
      }
      final String prefix = _xifexpression;
      String _xifexpression_1 = null;
      int _size_3 = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _greaterEqualsThan_1 = (_size_3 >= 8);
      if (_greaterEqualsThan_1) {
        _xifexpression_1 = args[7];
      } else {
        _xifexpression_1 = "_FEATURE";
      }
      final String suffix = _xifexpression_1;
      _xblockexpression = new FeatureIDEVariantGenerator.Parameters(configurationModel, featureModel, packageName, className, outputFolder, featuresPackageName, prefix, suffix);
    }
    return _xblockexpression;
  }
  
  private Configuration readConfigurationModel(final File modelFile) {
    try {
      final JAXBContext jaxbContext = JAXBContext.newInstance(Configuration.class);
      final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
      Object _unmarshal = unmarshaller.unmarshal(modelFile);
      return ((Configuration) _unmarshal);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private FeatureModelType readFeatureModel(final File modelFile) {
    try {
      final JAXBContext jaxbContext = JAXBContext.newInstance(FeatureModel.class);
      final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
      Object _unmarshal = unmarshaller.unmarshal(modelFile);
      return ((FeatureModelType) _unmarshal);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void generateVariantClass(final FeatureIDEVariantGenerator.Parameters parameters) {
    @Extension
    final NameProvider nameProvider = new NameProvider();
    @Extension
    final FeatureNameConverter featureNameConverter = new FeatureNameConverter();
    final String rootName = this.getRoot(parameters.configurationModel).getName();
    final String simpleName = parameters.className;
    final Function1<FeatureType, String> _function = (FeatureType it) -> {
      return it.getName();
    };
    final List<String> abstractFeatures = ListExtensions.<FeatureType, String>map(this.findAbstractFeatures(this.getRoot(parameters.featureModel)), _function);
    final Function1<Feature, Boolean> _function_1 = (Feature it) -> {
      return Boolean.valueOf((Objects.equal(it.getAutomatic(), "selected") || Objects.equal(it.getManual(), "selected")));
    };
    final Function1<Feature, String> _function_2 = (Feature it) -> {
      return it.getName();
    };
    final Function1<String, Boolean> _function_3 = (String it) -> {
      boolean _contains = abstractFeatures.contains(it);
      return Boolean.valueOf((!_contains));
    };
    final Function1<String, String> _function_4 = (String it) -> {
      return featureNameConverter.convertToValidSimpleFeatureName(it, parameters.prefix, parameters.suffix);
    };
    final Function1<String, String> _function_5 = (String it) -> {
      String _simpleFeaturesEnumName = nameProvider.getSimpleFeaturesEnumName(rootName);
      String _plus = (_simpleFeaturesEnumName + ".");
      return (_plus + it);
    };
    final Iterable<String> selectedFeatures = IterableExtensions.<String, String>map(IterableExtensions.<String, String>map(IterableExtensions.<String>filter(IterableExtensions.<Feature, String>map(IterableExtensions.<Feature>filter(parameters.configurationModel.getFeature(), _function_1), _function_2), _function_3), _function_4), _function_5);
    final File outputFile = this.prepareOutputFile(parameters);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("package ");
    _builder.append(parameters.packageName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    {
      boolean _notEquals = (!Objects.equal(parameters.packageName, parameters.featuresPackageName));
      if (_notEquals) {
        _builder.append("import ");
        _builder.append(parameters.featuresPackageName);
        _builder.append(".");
        String _simpleVariantInterfaceName = nameProvider.getSimpleVariantInterfaceName(rootName);
        _builder.append(_simpleVariantInterfaceName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("import ");
        _builder.append(parameters.featuresPackageName);
        _builder.append(".");
        String _simpleSelectedFeaturesAnnotationName = nameProvider.getSimpleSelectedFeaturesAnnotationName(rootName);
        _builder.append(_simpleSelectedFeaturesAnnotationName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.append("import ");
        _builder.append(parameters.featuresPackageName);
        _builder.append(".");
        String _simpleFeaturesEnumName = nameProvider.getSimpleFeaturesEnumName(rootName);
        _builder.append(_simpleFeaturesEnumName);
        _builder.append(";");
        _builder.newLineIfNotEmpty();
        _builder.newLine();
      }
    }
    _builder.append("@");
    String _simpleSelectedFeaturesAnnotationName_1 = nameProvider.getSimpleSelectedFeaturesAnnotationName(rootName);
    _builder.append(_simpleSelectedFeaturesAnnotationName_1);
    _builder.append("( {");
    String _join = IterableExtensions.join(selectedFeatures, ", ");
    _builder.append(_join);
    _builder.append("} )");
    _builder.newLineIfNotEmpty();
    _builder.append("public final class ");
    _builder.append(simpleName);
    _builder.append(" implements ");
    String _simpleVariantInterfaceName_1 = nameProvider.getSimpleVariantInterfaceName(rootName);
    _builder.append(_simpleVariantInterfaceName_1);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("private ");
    _builder.append(simpleName, "\t");
    _builder.append("( ) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    final String fileContent = _builder.toString();
    final GeneratorHelper generatorHelper = new GeneratorHelper();
    generatorHelper.writeContentToFileIfChanged(fileContent, outputFile);
  }
  
  private FeatureType getRoot(final FeatureModelType model) {
    FeatureType _xblockexpression = null;
    {
      final StructType struct = model.getStruct();
      FeatureType _xifexpression = null;
      FeatureType _feature = struct.getFeature();
      boolean _tripleNotEquals = (_feature != null);
      if (_tripleNotEquals) {
        _xifexpression = struct.getFeature();
      } else {
        BranchedFeatureType _xifexpression_1 = null;
        BranchedFeatureType _and = struct.getAnd();
        boolean _tripleNotEquals_1 = (_and != null);
        if (_tripleNotEquals_1) {
          _xifexpression_1 = struct.getAnd();
        } else {
          BranchedFeatureType _xifexpression_2 = null;
          BranchedFeatureType _or = struct.getOr();
          boolean _tripleNotEquals_2 = (_or != null);
          if (_tripleNotEquals_2) {
            _xifexpression_2 = struct.getOr();
          } else {
            _xifexpression_2 = struct.getAlt();
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private List<FeatureType> findAbstractFeatures(final FeatureType type) {
    List<FeatureType> _xblockexpression = null;
    {
      final List<FeatureType> features = new ArrayList<FeatureType>();
      if ((type != null)) {
        boolean _equals = Boolean.TRUE.equals(type.isAbstract());
        if (_equals) {
          features.add(type);
        }
        if ((type instanceof BranchedFeatureType)) {
          List<JAXBElement<? extends FeatureType>> _andOrOrOrAlt = ((BranchedFeatureType)type).getAndOrOrOrAlt();
          for (final JAXBElement<? extends FeatureType> feature : _andOrOrOrAlt) {
            List<FeatureType> _findAbstractFeatures = this.findAbstractFeatures(feature.getValue());
            Iterables.<FeatureType>addAll(features, _findAbstractFeatures);
          }
        }
      }
      _xblockexpression = features;
    }
    return _xblockexpression;
  }
  
  private Feature getRoot(final Configuration configurationModel) {
    Feature _xblockexpression = null;
    {
      final List<Feature> features = configurationModel.getFeature();
      _xblockexpression = IterableExtensions.<Feature>head(features);
    }
    return _xblockexpression;
  }
  
  private File prepareOutputFile(final FeatureIDEVariantGenerator.Parameters parameters) {
    File _xblockexpression = null;
    {
      final String packagePath = parameters.packageName.replace(".", "/");
      final File outputFileParent = new File(parameters.outputFolder, packagePath);
      outputFileParent.mkdirs();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(parameters.className);
      _builder.append(".java");
      _xblockexpression = new File(outputFileParent, _builder.toString());
    }
    return _xblockexpression;
  }
}
