package de.rhocas.featuregen.generator;

import com.google.common.collect.Iterables;
import de.rhocas.featuregen.featureide.model.feature.BranchedFeatureType;
import de.rhocas.featuregen.featureide.model.feature.FeatureModel;
import de.rhocas.featuregen.featureide.model.feature.FeatureModelType;
import de.rhocas.featuregen.featureide.model.feature.FeatureType;
import de.rhocas.featuregen.featureide.model.feature.StructType;
import de.rhocas.featuregen.generator.FeatureNameConverter;
import de.rhocas.featuregen.generator.GeneratorHelper;
import de.rhocas.featuregen.generator.NameProvider;
import de.rhocas.featuregen.lib.FeatureGenLabel;
import de.rhocas.featuregen.lib.FeatureGenSelectedFeatures;
import java.io.File;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.Unmarshaller;
import org.eclipse.xtend.lib.annotations.AccessorType;
import org.eclipse.xtend.lib.annotations.Accessors;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * This class generates the features from a FeatureIDE model file.
 * 
 * @author Nils Christian Ehmke
 * 
 * @since 2.0.0
 */
@SuppressWarnings("all")
public final class FeatureIDEFeaturesGenerator {
  @Accessors(AccessorType.NONE)
  private static final class Parameters {
    private final FeatureModelType featureModel;
    
    private final String packageName;
    
    private final File outputFolder;
    
    private final String prefix;
    
    private final String suffix;
    
    public Parameters(final FeatureModelType featureModel, final String packageName, final File outputFolder, final String prefix, final String suffix) {
      super();
      this.featureModel = featureModel;
      this.packageName = packageName;
      this.outputFolder = outputFolder;
      this.prefix = prefix;
      this.suffix = suffix;
    }
  }
  
  /**
   * The main entry point of the standalone generator.
   * 
   * @param args
   * 		The command line arguments. See {@link #generate(String[])} for details.
   * 
   * @since 2.0.0
   */
  public static void main(final String[] args) {
    final FeatureIDEFeaturesGenerator generator = new FeatureIDEFeaturesGenerator();
    generator.generate(args);
  }
  
  /**
   * This method performs the actual generation.
   * 
   * @param args
   * 		The arguments for the generator. This array must contain at least three entries. The arguments are as following:
   *      <ul>
   *      	<li>The path to the FeatureIDE feature model file.</li>
   *          <li>The path to the output folder.</li>
   *          <li>The package name of the newly generated classes.</li>
   *          <li>The optional prefix prepended to each feature.</li>
   *          <li>The optional suffix appended to each feature. If this parameter is not given, {@code _FEATURE} will be used instead.</li>
   *      </ul>
   * 
   * @throws IllegalArgumentException
   * 		If the number of arguments is invalid or if the model file can not be found.
   * 
   * @since 2.0.0
   */
  public void generate(final String[] args) {
    final FeatureIDEFeaturesGenerator.Parameters parameters = this.convertAndCheckParameters(args);
    this.generateVariantInterface(parameters);
    this.generateFeatureEnumeration(parameters);
    this.generateSelectedFeaturesAnnotation(parameters);
    this.generateFeatureCheckServiceClass(parameters);
  }
  
  private FeatureIDEFeaturesGenerator.Parameters convertAndCheckParameters(final String[] args) {
    FeatureIDEFeaturesGenerator.Parameters _xblockexpression = null;
    {
      int _size = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _lessThan = (_size < 3);
      if (_lessThan) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Invalid number of arguments. Expected at least 3, but was ");
        int _size_1 = ((List<String>)Conversions.doWrapArray(args)).size();
        _builder.append(_size_1);
        _builder.append(".");
        throw new IllegalArgumentException(_builder.toString());
      }
      final String modelFilePath = args[0];
      final File modelFile = new File(modelFilePath);
      boolean _isFile = modelFile.isFile();
      boolean _not = (!_isFile);
      if (_not) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("Model file \"");
        _builder_1.append(modelFilePath);
        _builder_1.append("\" can not be found.");
        throw new IllegalArgumentException(_builder_1.toString());
      }
      final FeatureModelType featureModel = this.readFeatureModel(modelFile);
      final String outputFolderPath = args[1];
      final File outputFolder = new File(outputFolderPath);
      final String packageName = args[2];
      String _xifexpression = null;
      int _size_2 = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _greaterEqualsThan = (_size_2 >= 4);
      if (_greaterEqualsThan) {
        _xifexpression = args[3];
      } else {
        _xifexpression = "";
      }
      final String prefix = _xifexpression;
      String _xifexpression_1 = null;
      int _size_3 = ((List<String>)Conversions.doWrapArray(args)).size();
      boolean _greaterEqualsThan_1 = (_size_3 >= 5);
      if (_greaterEqualsThan_1) {
        _xifexpression_1 = args[4];
      } else {
        _xifexpression_1 = "_FEATURE";
      }
      final String suffix = _xifexpression_1;
      _xblockexpression = new FeatureIDEFeaturesGenerator.Parameters(featureModel, packageName, outputFolder, prefix, suffix);
    }
    return _xblockexpression;
  }
  
  private FeatureModelType readFeatureModel(final File modelFile) {
    try {
      final JAXBContext jaxbContext = JAXBContext.newInstance(FeatureModel.class);
      final Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
      Object _unmarshal = unmarshaller.unmarshal(modelFile);
      return ((FeatureModelType) _unmarshal);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private void generateVariantInterface(final FeatureIDEFeaturesGenerator.Parameters parameters) {
    @Extension
    final NameProvider nameProvider = new NameProvider();
    final String rootName = this.getRoot(parameters.featureModel).getName();
    final String simpleName = nameProvider.getSimpleVariantInterfaceName(rootName);
    final File outputFile = this.prepareOutputFile(parameters, simpleName);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("package ");
    _builder.append(parameters.packageName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This is a marker interface for all variants.<br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* <br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This interface is generated.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public interface ");
    _builder.append(simpleName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    final String fileContent = _builder.toString();
    final GeneratorHelper generatorHelper = new GeneratorHelper();
    generatorHelper.writeContentToFileIfChanged(fileContent, outputFile);
  }
  
  private File prepareOutputFile(final FeatureIDEFeaturesGenerator.Parameters parameters, final String elementName) {
    File _xblockexpression = null;
    {
      final String packagePath = parameters.packageName.replace(".", "/");
      final File outputFileParent = new File(parameters.outputFolder, packagePath);
      outputFileParent.mkdirs();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append(elementName);
      _builder.append(".java");
      _xblockexpression = new File(outputFileParent, _builder.toString());
    }
    return _xblockexpression;
  }
  
  private FeatureType getRoot(final FeatureModelType model) {
    FeatureType _xblockexpression = null;
    {
      final StructType struct = model.getStruct();
      FeatureType _xifexpression = null;
      FeatureType _feature = struct.getFeature();
      boolean _tripleNotEquals = (_feature != null);
      if (_tripleNotEquals) {
        _xifexpression = struct.getFeature();
      } else {
        BranchedFeatureType _xifexpression_1 = null;
        BranchedFeatureType _and = struct.getAnd();
        boolean _tripleNotEquals_1 = (_and != null);
        if (_tripleNotEquals_1) {
          _xifexpression_1 = struct.getAnd();
        } else {
          BranchedFeatureType _xifexpression_2 = null;
          BranchedFeatureType _or = struct.getOr();
          boolean _tripleNotEquals_2 = (_or != null);
          if (_tripleNotEquals_2) {
            _xifexpression_2 = struct.getOr();
          } else {
            _xifexpression_2 = struct.getAlt();
          }
          _xifexpression_1 = _xifexpression_2;
        }
        _xifexpression = _xifexpression_1;
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  private void generateFeatureEnumeration(final FeatureIDEFeaturesGenerator.Parameters parameters) {
    @Extension
    final NameProvider nameProvider = new NameProvider();
    @Extension
    final FeatureNameConverter featureNameConverter = new FeatureNameConverter();
    final FeatureType root = this.getRoot(parameters.featureModel);
    final String rootName = root.getName();
    final String simpleName = nameProvider.getSimpleFeaturesEnumName(rootName);
    final File outputFile = this.prepareOutputFile(parameters, simpleName);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("package ");
    _builder.append(parameters.packageName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("import ");
    String _name = FeatureGenLabel.class.getName();
    _builder.append(_name);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This enumeration contains all available features.<br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* <br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This enumeration is generated.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public enum ");
    _builder.append(simpleName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    {
      List<FeatureType> _linearizeFeatures = this.linearizeFeatures(root);
      boolean _hasElements = false;
      for(final FeatureType feature : _linearizeFeatures) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(",", "\t");
        }
        {
          boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(feature.getDescription());
          boolean _not = (!_isNullOrEmpty);
          if (_not) {
            _builder.append("\t");
            _builder.append("/**");
            _builder.newLine();
            _builder.append("\t");
            _builder.append(" ");
            _builder.append("* ");
            String _description = feature.getDescription();
            _builder.append(_description, "\t ");
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append(" ");
            _builder.append("*/");
            _builder.newLine();
          }
        }
        _builder.append("\t");
        _builder.append("@");
        String _simpleName = FeatureGenLabel.class.getSimpleName();
        _builder.append(_simpleName, "\t");
        _builder.append("( \"");
        String _name_1 = feature.getName();
        _builder.append(_name_1, "\t");
        _builder.append("\" )");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        String _convertToValidSimpleFeatureName = featureNameConverter.convertToValidSimpleFeatureName(feature.getName(), parameters.prefix, parameters.suffix);
        _builder.append(_convertToValidSimpleFeatureName, "\t");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("}");
    _builder.newLine();
    final String fileContent = _builder.toString();
    final GeneratorHelper generatorHelper = new GeneratorHelper();
    generatorHelper.writeContentToFileIfChanged(fileContent, outputFile);
  }
  
  private List<FeatureType> linearizeFeatures(final FeatureType type) {
    List<FeatureType> _xblockexpression = null;
    {
      final List<FeatureType> features = new ArrayList<FeatureType>();
      if ((type != null)) {
        boolean _equals = Boolean.TRUE.equals(type.isAbstract());
        boolean _not = (!_equals);
        if (_not) {
          features.add(type);
        }
        if ((type instanceof BranchedFeatureType)) {
          List<JAXBElement<? extends FeatureType>> _andOrOrOrAlt = ((BranchedFeatureType)type).getAndOrOrOrAlt();
          for (final JAXBElement<? extends FeatureType> feature : _andOrOrOrAlt) {
            List<FeatureType> _linearizeFeatures = this.linearizeFeatures(feature.getValue());
            Iterables.<FeatureType>addAll(features, _linearizeFeatures);
          }
        }
      }
      _xblockexpression = features;
    }
    return _xblockexpression;
  }
  
  private void generateSelectedFeaturesAnnotation(final FeatureIDEFeaturesGenerator.Parameters parameters) {
    @Extension
    final NameProvider nameProvider = new NameProvider();
    final String rootName = this.getRoot(parameters.featureModel).getName();
    final String simpleName = nameProvider.getSimpleSelectedFeaturesAnnotationName(rootName);
    final File outputFile = this.prepareOutputFile(parameters, simpleName);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("package ");
    _builder.append(parameters.packageName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("import ");
    String _name = Retention.class.getName();
    _builder.append(_name);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_1 = RetentionPolicy.class.getName();
    _builder.append(_name_1);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_2 = Target.class.getName();
    _builder.append(_name_2);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_3 = ElementType.class.getName();
    _builder.append(_name_3);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_4 = FeatureGenSelectedFeatures.class.getName();
    _builder.append(_name_4);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This annotation is used to mark which features the annotated variant provides.<br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* <br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This annotation is generated.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("@");
    String _simpleName = Retention.class.getSimpleName();
    _builder.append(_simpleName);
    _builder.append("( ");
    String _simpleName_1 = RetentionPolicy.class.getSimpleName();
    _builder.append(_simpleName_1);
    _builder.append(".");
    String _name_5 = RetentionPolicy.RUNTIME.name();
    _builder.append(_name_5);
    _builder.append(" )");
    _builder.newLineIfNotEmpty();
    _builder.append("@");
    String _simpleName_2 = Target.class.getSimpleName();
    _builder.append(_simpleName_2);
    _builder.append("( ");
    String _simpleName_3 = ElementType.class.getSimpleName();
    _builder.append(_simpleName_3);
    _builder.append(".");
    String _name_6 = ElementType.TYPE.name();
    _builder.append(_name_6);
    _builder.append(" )");
    _builder.newLineIfNotEmpty();
    _builder.append("@");
    String _simpleName_4 = FeatureGenSelectedFeatures.class.getSimpleName();
    _builder.append(_simpleName_4);
    _builder.append("( )");
    _builder.newLineIfNotEmpty();
    _builder.append("public @interface ");
    _builder.append(simpleName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* The selected features.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("\t");
    String _simpleFeaturesEnumName = nameProvider.getSimpleFeaturesEnumName(rootName);
    _builder.append(_simpleFeaturesEnumName, "\t");
    _builder.append("[] value( );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    final String fileContent = _builder.toString();
    final GeneratorHelper generatorHelper = new GeneratorHelper();
    generatorHelper.writeContentToFileIfChanged(fileContent, outputFile);
  }
  
  private void generateFeatureCheckServiceClass(final FeatureIDEFeaturesGenerator.Parameters parameters) {
    @Extension
    final NameProvider nameProvider = new NameProvider();
    final String rootName = this.getRoot(parameters.featureModel).getName();
    final String simpleName = nameProvider.getSimpleFeatureCheckServiceClassName(rootName);
    final String simpleSelectedFeaturesAnnotationName = nameProvider.getSimpleSelectedFeaturesAnnotationName(rootName);
    final String simpleFeaturesEnumName = nameProvider.getSimpleFeaturesEnumName(rootName);
    final File outputFile = this.prepareOutputFile(parameters, simpleName);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("package ");
    _builder.append(parameters.packageName);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("import ");
    String _name = Set.class.getName();
    _builder.append(_name);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_1 = EnumSet.class.getName();
    _builder.append(_name_1);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_2 = Objects.class.getName();
    _builder.append(_name_2);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_3 = List.class.getName();
    _builder.append(_name_3);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_4 = Arrays.class.getName();
    _builder.append(_name_4);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.append("import ");
    String _name_5 = Collections.class.getName();
    _builder.append(_name_5);
    _builder.append(";");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("/**");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This service allows to check which features are currently active.<br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* <br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* Note that instances of this class are immutable and thus inherent thread safe.<br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* <br>");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("* This service is generated.");
    _builder.newLine();
    _builder.append(" ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("public final class ");
    _builder.append(simpleName);
    _builder.append(" {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("private final ");
    String _simpleName = Set.class.getSimpleName();
    _builder.append(_simpleName, "\t");
    _builder.append("<");
    _builder.append(simpleFeaturesEnumName, "\t");
    _builder.append("> activeFeatures;");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("private final String description;");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("private ");
    _builder.append(simpleName, "\t");
    _builder.append("( final ");
    String _simpleName_1 = List.class.getSimpleName();
    _builder.append(_simpleName_1, "\t");
    _builder.append("<");
    _builder.append(simpleFeaturesEnumName, "\t");
    _builder.append("> selectedFeatures, final String variantName ) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("activeFeatures = ");
    String _simpleName_2 = EnumSet.class.getSimpleName();
    _builder.append(_simpleName_2, "\t\t");
    _builder.append(".noneOf( ");
    _builder.append(simpleFeaturesEnumName, "\t\t");
    _builder.append(".class );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("activeFeatures.addAll( selectedFeatures );");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("description = \"");
    _builder.append(simpleName, "\t\t");
    _builder.append(" [\" + variantName + \"]\";");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* Checks whether the given feature is currently active or not.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @param feature");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* \t\tThe feature to check. Must not be {@code null}.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* ");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @return true if and only if the given feature is active.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* ");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @throws NullPointerException");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*\t\tIf the given feature is {@code null}.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public boolean isFeatureActive( final ");
    _builder.append(simpleFeaturesEnumName, "\t");
    _builder.append(" feature ) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    String _simpleName_3 = Objects.class.getSimpleName();
    _builder.append(_simpleName_3, "\t\t");
    _builder.append(".requireNonNull( feature, \"The feature must not be null.\" );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("return activeFeatures.contains( feature );");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("@");
    String _simpleName_4 = Override.class.getSimpleName();
    _builder.append(_simpleName_4, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("public String toString( ) {");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("return description;");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* Creates a new instance of this service with the features of the given variant.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @param variant");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*\t\tThe new variant. Must not be {@code null} and must be annotated with {@link ");
    _builder.append(simpleSelectedFeaturesAnnotationName, "\t ");
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.append("\t ");
    _builder.append("*");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @return A new feature check service.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* ");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @throws NullPointerException");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*\t\tIf the given variant is {@code null} or not annotated with {@link ");
    _builder.append(simpleSelectedFeaturesAnnotationName, "\t ");
    _builder.append("}.");
    _builder.newLineIfNotEmpty();
    _builder.append("\t ");
    _builder.append("*/");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public static ");
    _builder.append(simpleName, "\t");
    _builder.append(" of( final Class<? extends ");
    String _simpleVariantInterfaceName = nameProvider.getSimpleVariantInterfaceName(rootName);
    _builder.append(_simpleVariantInterfaceName, "\t");
    _builder.append("> variant ) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    String _simpleName_5 = Objects.class.getSimpleName();
    _builder.append(_simpleName_5, "\t\t");
    _builder.append(".requireNonNull( variant, \"The variant must not be null.\" );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("final ");
    _builder.append(simpleSelectedFeaturesAnnotationName, "\t\t");
    _builder.append(" selectedFeaturesAnnotation = variant.getAnnotation( ");
    _builder.append(simpleSelectedFeaturesAnnotationName, "\t\t");
    _builder.append(".class );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    String _simpleName_6 = Objects.class.getSimpleName();
    _builder.append(_simpleName_6, "\t\t");
    _builder.append(".requireNonNull( selectedFeaturesAnnotation, \"The variant must be annotated with ");
    _builder.append(simpleSelectedFeaturesAnnotationName, "\t\t");
    _builder.append(".\" );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("final ");
    String _simpleName_7 = List.class.getSimpleName();
    _builder.append(_simpleName_7, "\t\t");
    _builder.append("<");
    _builder.append(simpleFeaturesEnumName, "\t\t");
    _builder.append("> selectedFeatures = ");
    String _simpleName_8 = Arrays.class.getSimpleName();
    _builder.append(_simpleName_8, "\t\t");
    _builder.append(".asList( selectedFeaturesAnnotation.value( ) );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("return new ");
    _builder.append(simpleName, "\t\t");
    _builder.append("( selectedFeatures, variant.getSimpleName( ) );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("/**");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* Creates a new instance of this service without any active features.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("* @return A new feature check service.");
    _builder.newLine();
    _builder.append("\t ");
    _builder.append("*/\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("public static ");
    _builder.append(simpleName, "\t");
    _builder.append(" empty( ) {");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("return new ");
    _builder.append(simpleName, "\t\t");
    _builder.append("( ");
    String _simpleName_9 = Collections.class.getSimpleName();
    _builder.append(_simpleName_9, "\t\t");
    _builder.append(".emptyList( ), \"Empty\" );");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("}");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("}");
    _builder.newLine();
    final String fileContent = _builder.toString();
    final GeneratorHelper generatorHelper = new GeneratorHelper();
    generatorHelper.writeContentToFileIfChanged(fileContent, outputFile);
  }
}
