package de.redsix.dmncheck.util;

import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Supplier;

public final class Eithers {
  private Eithers() {
  }

  public static <A, B> Either<A, B> left(A left) {
    return new Left<>(left);
  }

  public static <A, B> Either<A, B> right(B right) {
    return new Right<>(right);
  }

  public static <A, B> CaseOfMatchers.TotalMatcher_Left<A, B> caseOf(Either<A, B> either) {
    return new CaseOfMatchers.TotalMatcher_Left<A, B>(either);
  }

  public static <A, B> Optional<A> getLeft(Either<A, B> either) {
    return either.match((left) -> Optional.of(left),
    (right) -> Optional.empty());}

  public static <A, B> Optional<B> getRight(Either<A, B> either) {
    return either.match((left) -> Optional.empty(),
    (right) -> Optional.of(right));}

  private static final class Left<A, B> extends Either<A, B> {
    private final A left;

    Left(A left) {
      this.left = left;
    }

    @Override
    public <X> X match(Function<A, X> left, Function<B, X> right) {
      return left.apply(this.left);
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean equals(Object obj) {
      return (obj instanceof Either) && ((Either<A, B>) obj).match((left) -> this.left.equals(left),
          (right) -> false);
    }

    @Override
    public int hashCode() {
      return 23 + this.left.hashCode();
    }

    @Override
    public String toString() {
      return "left(" + this.left + ")";
    }
  }

  private static final class Right<A, B> extends Either<A, B> {
    private final B right;

    Right(B right) {
      this.right = right;
    }

    @Override
    public <X> X match(Function<A, X> left, Function<B, X> right) {
      return right.apply(this.right);
    }

    @Override
    @SuppressWarnings("unchecked")
    public boolean equals(Object obj) {
      return (obj instanceof Either) && ((Either<A, B>) obj).match((left) -> false,
          (right) -> this.right.equals(right));
    }

    @Override
    public int hashCode() {
      return 29 + this.right.hashCode();
    }

    @Override
    public String toString() {
      return "right(" + this.right + ")";
    }
  }

  public static class CaseOfMatchers {
    private CaseOfMatchers() {
    }

    public static final class TotalMatcher_Left<A, B> {
      private final Either<A, B> _either;

      TotalMatcher_Left(Either<A, B> _either) {
        this._either = _either;
      }

      public final <X> TotalMatcher_Right<A, B, X> left(Function<A, X> left) {
        return new TotalMatcher_Right<>(this._either, left);
      }

      public final <X> TotalMatcher_Right<A, B, X> left_(X x) {
        return this.left((left) -> x);
      }

      public final <X> PartialMatcher<A, B, X> right(Function<B, X> right) {
        return new PartialMatcher<>(this._either, null, right);
      }

      public final <X> PartialMatcher<A, B, X> right_(X x) {
        return this.right((right) -> x);
      }
    }

    public static final class TotalMatcher_Right<A, B, X> extends PartialMatcher<A, B, X> {
      TotalMatcher_Right(Either<A, B> _either, Function<A, X> left) {
        super(_either, left, null);
      }

      public final X right(Function<B, X> right) {
        Function<A, X> left = super.left;
        return ((PartialMatcher<A, B, X>) this)._either.match(left, right);
      }

      public final X right_(X x) {
        return this.right((right) -> x);
      }
    }

    public static class PartialMatcher<A, B, X> {
      private final Either<A, B> _either;

      private final Function<A, X> left;

      private final Function<B, X> right;

      PartialMatcher(Either<A, B> _either, Function<A, X> left, Function<B, X> right) {
        this._either = _either;
        this.left = left;
        this.right = right;
      }

      public final X otherwise(Supplier<X> otherwise) {
        Function<A, X> left = (this.left != null) ? this.left : (left_) -> otherwise.get();
        Function<B, X> right = (this.right != null) ? this.right : (right_) -> otherwise.get();
        return this._either.match(left, right);
      }

      public final X otherwise_(X x) {
        return this.otherwise(() -> x);
      }

      public final Optional<X> otherwiseEmpty() {
        Function<A, Optional<X>> left = (this.left != null) ? (left_) -> Optional.of(this.left.apply(left_))
            : (left_) -> Optional.empty();
        Function<B, Optional<X>> right = (this.right != null) ? (right_) -> Optional.of(this.right.apply(right_))
            : (right_) -> Optional.empty();
        return this._either.match(left, right);
      }
    }
  }
}
