package de.pseudonymisierung.controlnumbers;

import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.Mac;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Base64;

import de.pseudonymisierung.controlnumbers.ControlNumberGenerator.Builder;

/** Generator for control numbers using a keyed hash algorithm.
 *
 * Use {@link #builder(String)} to get a {@link Builder} instance, which has
 * methods for setting various parameters. */
public class EncryptedControlNumberGenerator extends ControlNumberGenerator {

	private Mac hmacMd5;
	private Mac hmacSha1;
	private Cipher aes;
	private Base64 base64;
	private String keyId;

	@Override
	protected synchronized byte[] getMd5Hash(String input) {
		return hmacMd5.doFinal(input.getBytes(this.encoding));
	}

	@Override
	protected synchronized byte[] getSha1Hash(String input) {
		return hmacSha1.doFinal(input.getBytes(this.encoding));
	}

	/** Get a builder object with methods to set parameters to non-default
	 * values and to get the resulting instance of
	 * {@link EncryptedControlNumberGenerator}.
	 *
	 * @param passphrase
	 *            The passphrase used for encryption. Must be a non-empty
	 *            string.
	 * @return A builder for creating instances of
	 *         {@link EncryptedControlNumberGenerator}. */
	public static Builder builder(String passphrase) {
		return new Builder(passphrase);
	}

	/** Symmetric encryption of a string using a key based on the configured
	 * passphrase. Used for external id.
	 *
	 * @param value
	 *            The string to encrypt.
	 * @return The encrypted string in Base64 encoding. */
	public synchronized String encrypt(String value) {
		byte[] data;
		try {
			data = aes.doFinal(value.getBytes("UTF8"));
		} catch (UnsupportedEncodingException e) {
			throw new Error("EncryptedControlNumberGenerator needs UTF8 encoding to be available.", e);
		} catch (Exception e) {
			// doFinal declares BadPaddingException and
			// IllegalBlocksizeException, but these
			// should not occur here as the encryption algorithm is fixed.
			throw new Error("Unexpected error while performing encryption.", e);
		}
		return base64.encodeAsString(data);
	}

	/** Builder for control number generators with keyed hashes. */
	public static class Builder extends AbstractBuilder<EncryptedControlNumberGenerator> {

		private String passphrase;

		private Builder(String passphrase) {
			super();
			if (passphrase == null || passphrase.equals("")) {
				throw new IllegalArgumentException("Passphrase must be a valid, non-empty string.");
			}
			this.passphrase = passphrase;
		}

		@Override
		public EncryptedControlNumberGenerator build() {
			return new EncryptedControlNumberGenerator(this);
		}

	}

	protected EncryptedControlNumberGenerator(Builder builder) {
		super(builder);

		try {
			// Always use UTF8 for interpreting passphrase (ensures compatibility with ID-Manager)
			MessageDigest md5 = MessageDigest.getInstance("MD5");
			byte key[] = md5.digest(builder.passphrase.getBytes("UTF8"));
			// Construct encrypted Hash functions
			SecretKeySpec keySpec = new SecretKeySpec(key, "HmacSHA1");
			hmacSha1 = Mac.getInstance("HmacSHA1");
			hmacSha1.init(keySpec);

			keySpec = new SecretKeySpec(key, "HmacMD5");
			hmacMd5 = Mac.getInstance("HmacMD5");
			hmacMd5.init(keySpec);

			keySpec = new SecretKeySpec(key, "AES");
			aes = Cipher.getInstance("AES");
			aes.init(Cipher.ENCRYPT_MODE, keySpec);
		} catch (NoSuchAlgorithmException e) {
			throw new Error(
					"EncryptedControlNumberGenerator needs HmacSHA1 and HmacMD5 implementations to be available.", e);
		} catch (NoSuchPaddingException e) {
			throw new Error("Unsupported padding.", e);
		} catch (InvalidKeyException e) {
			// Hashing the passphrase should ensure that the key is valid.
			throw new Error("Hashed passphrase did not yield valid key.", e);
		} catch (UnsupportedEncodingException e) {
			// Should not happen as key encoding is fixed (UTF8)
			throw new Error(e); 
		}
		base64 = new Base64();
	}
}
