//$URL: https://dasding.googlecode.com/svn/tags/dasding-0.3/api/src/main/java/de/piratech/dasding/api/rest/EventIdResource.java $
//$Id: EventIdResource.java 44 2012-10-11 13:48:26Z amuthmann@gmail.com $
/*
 * Copyright 2012 Piratech.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.piratech.dasding.api.rest;

import de.micromata.opengis.kml.v_2_2_0.Document;
import de.micromata.opengis.kml.v_2_2_0.Folder;
import de.piratech.dasding.data.Action;
import de.piratech.dasding.data.Event;
import java.util.Date;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.FormParam;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import org.apache.commons.lang3.StringUtils;
import org.dummycreator.ClassBindings;
import org.dummycreator.DummyCreator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Resource used api calls for a specific event
 *
 * @author deveth0
 */
@Path("/events/id/{id}")
public class EventIdResource {

  private static final Logger LOG = LoggerFactory.getLogger(EventIdResource.class);

  /**
   * Returns a single event by id
   *
   * http://localhost:8080/DasDing/api/events/id/1
   *
   * GET
   *
   * @param id
   * @return
   */
  @GET
  @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
  public Event getEvent(
          @PathParam("id") String id) {
    LOG.debug("getEvent({})", id);
    //TODO: load event from DB
    DummyCreator dc = new DummyCreator(ClassBindings.defaultBindings());
    Event event = dc.create(Event.class);
    event.setId(id);
    return event;
  }

  /**
   * Returns the event as kml
   *
   * http://localhost:8080/DasDing/api/events/id/1.kml
   *
   * GET
   *
   * @param id
   * @return
   */
  @GET
  @Produces("application/vnd.google-earth.kml+xml")
  public Document getEventAsKML(
          @PathParam("id") Long id) {
    LOG.debug("getEventAsKML({})", id);
    //TODO: load event from DB
    DummyCreator dc = new DummyCreator(ClassBindings.defaultBindings());
    Event event = dc.create(Event.class);

    final Document document = new Document();
    document.createAndAddPlacemark().withName(event.getName())
            .withDescription(event.getDescription())
            .createAndSetPoint().addToCoordinates(event.getLocation().getLongitude(), event.getLocation().getLatitude());

    if (!event.getActions().isEmpty()) {
      Folder actions = document.createAndAddFolder().withName("actions");
      for (Action action : event.getActions()) {
        System.out.println("action: " + action.getLocation().getLongitude());
        actions.createAndAddPlacemark().withName(action.getName())
                .withDescription(action.getDescription())
                .createAndSetPoint().addToCoordinates(action.getLocation().getLongitude(), action.getLocation().getLatitude());
      }
    }
    return document;
  }

  /**
   * Updates the Event with the given id
   *
   * http://localhost:8080/DasDing/api/events/id/1
   *
   * PUT
   *
   * @param id
   * @param name
   * @param description
   * @param time
   *
   * @return
   */
  @PUT
  @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
  @Consumes(MediaType.APPLICATION_FORM_URLENCODED)
  public Event updateEvent(
          @PathParam("id") String id,
          @QueryParam("name") String name,
          @QueryParam("description") String description,
          @QueryParam("time") Long time) {
    LOG.debug("updateEvent(id={}, name={}, description={}, time={})", id, name, description, time);
    //TODO: load event from db
    Event event = new Event();
    event.setId(id);
    if (StringUtils.isNotBlank(name)) {
      event.setName(name);
    }
    if (StringUtils.isNotBlank(description)) {
      event.setDescription(description);
    }
    if (time != null) {
      event.setTime(new Date(time));
    }
    if (!event.isValid()) {
      throw new IllegalArgumentException("Event is not valid");
    }
    return event;
  }

  /**
   * Sets the given event to status "Canceled"
   *
   * http://localhost:8080/DasDing/api/events/id/1
   *
   * DELETE
   *
   * @param id
   */
  @DELETE
  @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
  public Event deleteEvent(
          @PathParam("id") String id) {
    LOG.debug("deleteEvent({})", id);
    //TODO: set event status to canceled
    Event event = new Event();
    event.setId(id);
    event.setStatus(Event.EVENTSTATUS.CANCELED);
    return event;
  }

  /**
   * Creates an action for the given event
   *
   * http://localhost:8080/DasDing/api/events/1/createAction
   *
   * POST
   *
   * Restricted to the groups owner
   *
   * @param name (required)
   * @param description
   * @param time
   *
   * @return
   */
  @Path("/createAction")
  @POST
  @Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
  public Event createAction(
          @PathParam("id") String id,
          @FormParam("name") String name,
          @FormParam("description") String description,
          @FormParam("time") Long time) {
    LOG.debug("createAction({})", id);
    Action action = new Action();
    action.setName(name);
    action.setDescription(description);
    action.setTime(new Date(time));
    if (!action.isValid()) {
      throw new IllegalArgumentException("action not valid");
    }

    //TODO: load event from DB and update it
    Event event = new Event();
    event.setId(id);
    event.addAction(action);
    return event;
  }
}
